/**
 * Copyright 2014-2017 Super Wayne
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.isuper.telegram.bot;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.isuper.common.utils.Preconditions;
import org.isuper.telegram.api.TelegramApiBuilder;
import org.isuper.telegram.api.TelegramBotApi;
import org.isuper.telegram.api.models.Update;
import org.isuper.telegram.api.models.http.ApiErrorResponse;
import org.isuper.telegram.api.models.http.GetUpdatesPayload;
import org.isuper.telegram.api.models.http.ApiResponse;
import org.isuper.telegram.utils.TelegramUtils;

import retrofit2.Response;

/**
 * @author Super Wayne
 *
 */
public class StandaloneTelegramBot {
	
	public static final int DEFAULT_GET_UPDATES_LIMIT = 100;
	public static final int DEFAULT_GET_UPDATES_TIMEOUT_IN_SEC = 60;
	public static final String[] DEFAULT_ALLOWED_UPDATE_TYPE = null;
	
	private static final ExecutorService ES = Executors.newFixedThreadPool(2);
	
	private final String apiToken;
	private final UpdateHandler updateHander;
	
	private Integer limit;
	private Integer timeout;
	private String[] allowedUpdates;
	
	private BotSessionStore sessionStore;
	private UpdateQuerier querier;
	private UpdateProcessor processor;
	
	/**
	 * @param apiToken
	 * 						The token of Telegram bot API.
	 * @param updateHander
	 * 						Handler of update.
	 */
	public StandaloneTelegramBot(String apiToken, UpdateHandler updateHander) {
		Preconditions.notEmptyString(apiToken, "Telegram API token should be specified");
		this.apiToken = apiToken;
		Preconditions.notNull(updateHander, "Updates handler should be specified");
		this.updateHander = updateHander;
	}

	/**
	 * 
	 */
	public void start() {
		BotSessionStore store = this.sessionStore == null ? new MemoryBasedBotSessionStore() : this.sessionStore;
		try {
			this.processor = new UpdateProcessor(store, this.updateHander);
			ES.submit(this.processor);
			this.querier = new UpdateQuerier(store, this.apiToken, this.limit, this.timeout, this.allowedUpdates);
			ES.submit(this.querier).get();
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			this.stop();
			ES.shutdown();
			store.save();
		}
	}
	
	/**
	 * @param sessionStore
	 * 				The session store which used to save and load received updates and bot status.
	 * @return
	 * 				The bot.
	 */
	public StandaloneTelegramBot setBotSessionStore(BotSessionStore sessionStore) {
		this.sessionStore = sessionStore;
		return this;
	}
	
	/**
	 * @param limit
	 * 				Limits the number of updates to be retrieved. Values between 1—100 are accepted. Defaults to 100.
	 * @return
	 * 				The bot.
	 */
	public StandaloneTelegramBot setGetUpdatesLimit(Integer limit) {
		this.limit = limit == null ? DEFAULT_GET_UPDATES_LIMIT : limit;
		return this;
	}
	
	/**
	 * @param timeout
	 * 				Timeout in seconds for long polling.
	 * @return
	 * 				The bot.
	 */
	public StandaloneTelegramBot setGetUpdatesTimeoutInSec(Integer timeout) {
		this.timeout = timeout == null || timeout <= 0 ? null : timeout;
		return this;
	}
	
	/**
	 * @param allowedUpdates
	 * 				List the types of updates you want your bot to receive. 
	 * @return
	 * 				The bot.
	 */
	public StandaloneTelegramBot setAllowedUpdateTypes(String... allowedUpdates) {
		this.allowedUpdates = allowedUpdates;
		return this;
	}
	
	public void stop() {
		if (this.querier != null) {
			this.querier.stop();
		}
		if (this.processor != null) {
			this.processor.stop();
		}
	}
	
	private static class UpdateProcessor implements Runnable {
		
		private static final Logger LOGGER = LogManager.getLogger(UpdateProcessor.class.getName());
		
		private final BotSessionStore sessionStore;
		private final UpdateHandler updateHander;
		
		private boolean stopped = false;

		/**
		 * @param sessionStore
		 * @param updateHander
		 */
		UpdateProcessor(BotSessionStore sessionStore, UpdateHandler updateHander) {
			Preconditions.notNull(sessionStore, "Bot session store should be specified");
			this.sessionStore = sessionStore;
			Preconditions.notNull(updateHander, "Updates handler should be specified");
			this.updateHander = updateHander;
		}

		/* (non-Javadoc)
		 * @see java.lang.Runnable#run()
		 */
		@Override
		public void run() {
			Update update = null;
			while (!this.stopped && !Thread.currentThread().isInterrupted()) {
				try {
					Update tmpUpdate = this.sessionStore.getLastUnprocessedUpdate();
					if (tmpUpdate != null) {
						update = tmpUpdate;
					} else {
						if (!this.sessionStore.hasUpdates()) {
							LOGGER.debug("No more update available.");
							try {
								TimeUnit.MILLISECONDS.sleep(200);
							} catch (InterruptedException e) {
								// Ignore
							}
							continue;
						}
						try {
							update = this.sessionStore.retrieveUpdate(1, TimeUnit.SECONDS);
						} catch (IOException e) {
							// Ignore
						}
					}
					if (update == null) {
						TimeUnit.MILLISECONDS.sleep(200);
						continue;
					}
					boolean processed = this.updateHander.handleUpdate(update);
					if (!processed) {
						this.sessionStore.setLastUnprocessedUpdate(update);
					}
				} catch (InterruptedException e) {
					// Ignore
				}
			}
		}
		
		/**
		 * @return the stopped
		 */
		public void stop() {
			this.stopped = true;
		}
		
	}
	
	private static class UpdateQuerier implements Runnable {
		
		private static final Logger LOGGER = LogManager.getLogger(UpdateQuerier.class.getName());
		
		private final BotSessionStore sessionStore;
		private final TelegramBotApi api;
		private final String apiToken;
		private final Integer limit;
		private final Integer timeout;
		private final String[] allowedUpdates;

		private boolean stopped = false;
		
		/**
		 * @param sessionStore
		 * @param apiToken
		 * @param limit
		 * @param timeout
		 * @param allowedUpdates
		 */
		UpdateQuerier(
				BotSessionStore sessionStore, String apiToken,
				Integer limit, Integer timeout, String... allowedUpdates) {
			Preconditions.notNull(sessionStore, "Updates queue should be specified");
			this.sessionStore = sessionStore;
			this.api = new TelegramApiBuilder().build();
			Preconditions.notEmptyString(apiToken, "Telegram API token should be specified");
			this.apiToken = apiToken;
			this.limit = limit == null ? DEFAULT_GET_UPDATES_LIMIT : limit;
			this.timeout = timeout == null || timeout <= 0 ? null : timeout;
			this.allowedUpdates = allowedUpdates;
		}

		/* (non-Javadoc)
		 * @see java.lang.Runnable#run()
		 */
		@Override
		public void run() {
			while (!this.stopped && !Thread.currentThread().isInterrupted()) {
				if (this.sessionStore.hasUpdates()) {
					LOGGER.debug("There're still some updates need to be proceeded, skip querying latest updates.");
					try {
						TimeUnit.MILLISECONDS.sleep(200);
					} catch (InterruptedException e) {
						// Ignore
					}
					continue;
				}
				final long lastReceivedUpdateId = this.sessionStore.getLastReceivedUpdateId();
				GetUpdatesPayload payload = new GetUpdatesPayload(this.sessionStore.getLastReceivedUpdateId() + 1, this.limit, this.timeout, this.allowedUpdates);
				List<Update> received;
				try {
					Response<ApiResponse<List<Update>>> resp = this.api.getUpdates(this.apiToken, payload).execute();
					if (resp.isSuccessful()) {
						received = resp.body().getResult();
						LOGGER.debug(String.format("Received %d update(s)", received.size()));
						received = received.stream().filter(update -> update != null && update.id > lastReceivedUpdateId).collect(Collectors.toList());
					} else {
						ApiErrorResponse errResp = TelegramUtils.getObjectMapper().readValue(resp.errorBody().bytes(), ApiErrorResponse.class);
						LOGGER.error(errResp.getDescription());
						received = Collections.emptyList();
					}
				} catch (IOException ioe) {
					received = Collections.emptyList();
				}
				if (received.isEmpty()) {
					LOGGER.debug("No new update available.");
					try {
						TimeUnit.MILLISECONDS.sleep(200);
					} catch (InterruptedException e) {
						// Ignore
					}
					continue;
				}
				LOGGER.debug(String.format("Received %d new update(s)", received.size()));
				this.sessionStore.addUpdates(received);
				this.sessionStore.setLastReceivedUpdateId(received.parallelStream().map(update -> update.id).max(Long::compareTo).orElse(0L));
			}
		}

		/**
		 * @return the stopped
		 */
		public void stop() {
			this.stopped = true;
		}
		
	}

}
