/**
 * 
 */
package org.isuper.urlshortener.google;

import java.io.IOException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.isuper.httpclient.AsyncHttpClient;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * @author Super Wang
 *
 */
public class GoogleURLShortener {
	
	private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper(new JsonFactory());
	
	private final AsyncHttpClient client;
	
	private final String endpoint;
	private final String originHost;
	
	private GoogleURLShortener(String apiKey, String originHost) throws IOException {
		this.client = AsyncHttpClient.newInstance();
		this.endpoint = String.format("https://www.googleapis.com/urlshortener/v1/url?key=%s", apiKey);
		this.originHost = originHost;
	}
	
	/**
	 * @param apiKey
	 * 				The API key for Google URL Shortener API
	 * @param originHost
	 * 				Optional, unless your API is restricted, you can override hostname or IP here.
	 * @return
	 * @throws IOException
	 */
	public static GoogleURLShortener newInstance(String apiKey, String originHost) throws IOException {
		return new GoogleURLShortener(apiKey, originHost);
	}
	
	/**
	 * @param longURL
	 * 				The long URL which need to be shortened
	 * @return
	 * 				Shortened URL
	 */
	public String shortenURL(String longURL) {
		if (longURL == null) {
			return null;
		}
		URLShortenerRequest payload = new URLShortenerRequest(longURL);
		try {
			HttpPost request = new HttpPost(this.endpoint);
			request.setEntity(new StringEntity(OBJECT_MAPPER.writeValueAsString(payload), ContentType.APPLICATION_JSON));
			if (this.originHost != null) {
				request.addHeader("Origin", this.originHost);
			}
			
			Future<HttpResponse> future = this.client.execute(request, null);
			HttpResponse resp = future.get();
			
			StatusLine status = resp.getStatusLine();
			HttpEntity entity = resp.getEntity();
			String contentType = entity.getContentType().getValue();
			if (status.getStatusCode() != 200) {
				throw new IOException(String.format("%d response received from server", status.getStatusCode()));
			}
			
			String content = EntityUtils.toString(entity);
			if (contentType.indexOf("application/json") >= 0) {
				URLShortenerResponse shortenerResp = OBJECT_MAPPER.readValue(content, URLShortenerResponse.class);
				return shortenerResp.id;
			} else {
				throw new IOException(String.format("Expecting a json response, but actually got a %s reponse with content: %s", contentType, content));
			}
		} catch (IOException e) {
			e.printStackTrace();
		} catch (InterruptedException e) {
			e.printStackTrace();
		} catch (ExecutionException e) {
			e.printStackTrace();
		}
		return null;
	}
	
	public static void main(String... args) {
		try {
			GoogleURLShortener shortener = GoogleURLShortener.newInstance("AIzaSyDkbU3Huo2gPvR1s0nJSdtPib4wvdOxUaQ", "apps.isuper.org");
			String shortURL = shortener.shortenURL("https://maps.googleapis.com/maps/api/staticmap?size=320x320&markers=color:blue%7Clabel:7%7C39.97763207713248,116.32519517522732&markers=color:blue%7Clabel:7%7C39.978403181606524,116.31431075710026&markers=color:blue%7Clabel:7%7C39.96860340558745,116.30580681486548&markers=color:green%7Clabel:7%7C39.96815497921694,116.32534231428345&markers=color:blue%7Clabel:7%7C39.97236712394374,116.3194157946399&markers=color:blue%7Clabel:7%7C39.968204584658814,116.32138455412554&markers=color:blue%7Clabel:7%7C39.97705888675155,116.3169801729604&markers=color:blue%7Clabel:7%7C39.970718148279886,116.32000987359184&markers=color:blue%7Clabel:7%7C39.97464977970359,116.30075657597106&markers=color:green%7Clabel:7%7C39.96397108505877,116.31171466212356&markers=color:blue%7Clabel:7%7C39.97032464676573,116.3202894311793&markers=color:blue%7Clabel:7%7C39.96831799003563,116.30617546519134&markers=color:blue%7Clabel:7%7C39.97847962980705,116.31624880725788&markers=color:blue%7Clabel:7%7C39.967801082321245,116.32540440884313&markers=color:green%7Clabel:7%7C39.96359707112445,116.31054728771902&markers=color:blue%7Clabel:7%7C39.978108387020924,116.32534958057109&markers=color:blue%7Clabel:8%7C39.97780498636738,116.30531906343798&markers=color:green%7Clabel:8%7C39.977448895078304,116.3052709302852&markers=color:blue%7Clabel:8%7C39.977215505305914,116.3167647368095&key=AIzaSyDkbU3Huo2gPvR1s0nJSdtPib4wvdOxUaQ");
			System.out.println(shortURL);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

}
