/**
 * 
 */
package org.isuper.urlshortener.google;

import java.io.Closeable;
import java.io.IOException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.isuper.httpclient.AsyncHttpClient;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * @author Super Wang
 *
 */
public class GoogleURLShortener implements Closeable {
	
	private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper(new JsonFactory());
	
	private final AsyncHttpClient client;
	
	private final String endpoint;
	private final String originHost;
	
	/**
	 * @param apiKey
	 * 				The API key for Google URL Shortener API
	 * @param originHost
	 * 				Optional, unless your API is restricted, you can override hostname or IP here.
	 */
	public GoogleURLShortener(String apiKey, String originHost) {
		this(apiKey, originHost, null, 1080);
	}
	
	/**
	 * @param apiKey
	 * 				The API key for Google URL Shortener API
	 * @param originHost
	 * 				Optional, unless your API is restricted, you can override hostname or IP here.
	 * @param proxyHostname
	 * 				Optional, unless you want to proxy your request, set the proxy hostname here
	 * @param proxyPort
	 * 				Optional, unless you want to proxy your request, set the proxy port here
	 */
	public GoogleURLShortener(final String apiKey, final String originHost, final String proxyHostname, final int proxyPort) {
		try {
			this.client = AsyncHttpClient.newInstance(proxyHostname, proxyPort);
			this.endpoint = String.format("https://www.googleapis.com/urlshortener/v1/url?key=%s", apiKey);
			this.originHost = originHost;
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * @param longURL
	 * 				The long URL which need to be shortened
	 * @return
	 * 				Shortened URL
	 */
	public String shortenURL(String longURL) {
		if (longURL == null) {
			return null;
		}
		URLShortenerRequest payload = new URLShortenerRequest(longURL);
		try {
			HttpPost request = new HttpPost(this.endpoint);
			request.setEntity(new StringEntity(OBJECT_MAPPER.writeValueAsString(payload), ContentType.APPLICATION_JSON));
			if (this.originHost != null) {
				request.addHeader("Origin", this.originHost);
			}
			
			Future<HttpResponse> future = this.client.execute(request, null);
			HttpResponse resp = future.get();
			
			StatusLine status = resp.getStatusLine();
			HttpEntity entity = resp.getEntity();
			String contentType = entity.getContentType().getValue();
			if (status.getStatusCode() != 200) {
				throw new IOException(String.format("%d response received from server", status.getStatusCode()));
			}
			
			String content = EntityUtils.toString(entity);
			if (contentType.indexOf("application/json") >= 0) {
				URLShortenerResponse shortenerResp = OBJECT_MAPPER.readValue(content, URLShortenerResponse.class);
				return shortenerResp.id;
			} else {
				throw new IOException(String.format("Expecting a json response, but actually got a %s reponse with content: %s", contentType, content));
			}
		} catch (IOException e) {
			e.printStackTrace();
		} catch (InterruptedException e) {
			e.printStackTrace();
		} catch (ExecutionException e) {
			e.printStackTrace();
		}
		return null;
	}
	
	/* (non-Javadoc)
	 * @see java.io.Closeable#close()
	 */
	@Override
	public void close() throws IOException {
		this.client.close();
	}

}
