/**
 * Copyright 2014-2017 Super Wayne
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.isuper.urlshortener.google;

import java.io.Closeable;
import java.io.IOException;
import java.net.ProxySelector;
import java.util.Arrays;
import java.util.concurrent.ExecutionException;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.client.config.AuthSchemes;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.BasicCookieStore;
import org.apache.http.impl.conn.SystemDefaultRoutePlanner;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.impl.nio.client.HttpAsyncClients;
import org.apache.http.util.EntityUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * @author Super Wayne
 *
 */
public class GoogleURLShortener implements Closeable {
	
	private static final Logger LOGGER = LogManager.getLogger("google-url-shortener");
	
	private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper(new JsonFactory());
	
	private final CloseableHttpAsyncClient httpclient;
	
	private final String endpoint;
	private final String originHost;
	
	/**
	 * @param apiKey
	 * 				The API key for Google URL Shortener API
	 * @param originHost
	 * 				Optional, unless your API is restricted, you can override hostname or IP here.
	 */
	public GoogleURLShortener(String apiKey, String originHost) {
		RequestConfig reqConf = RequestConfig.custom()
				.setCookieSpec(CookieSpecs.DEFAULT)
				.setExpectContinueEnabled(true)
				.setTargetPreferredAuthSchemes(Arrays.asList(AuthSchemes.NTLM, AuthSchemes.DIGEST))
				.setProxyPreferredAuthSchemes(Arrays.asList(AuthSchemes.BASIC))
				.build();
		this.httpclient = HttpAsyncClients.custom()
				.useSystemProperties()
				.setDefaultCookieStore(new BasicCookieStore())
				.setDefaultRequestConfig(reqConf)
				.setRoutePlanner(new SystemDefaultRoutePlanner(ProxySelector.getDefault()))
				.build();
		this.httpclient.start();
		this.endpoint = String.format("https://www.googleapis.com/urlshortener/v1/url?key=%s", apiKey);
		this.originHost = originHost;
	}
	
	/**
	 * @param longURL
	 * 				The long URL which need to be shortened
	 * @return
	 * 				Shortened URL
	 */
	public String shortenURL(String longURL) {
		if (longURL == null) {
			return null;
		}
		URLShortenerRequest payload = new URLShortenerRequest(longURL);
		try {
			HttpPost request = new HttpPost(this.endpoint);
			request.setEntity(new StringEntity(OBJECT_MAPPER.writeValueAsString(payload), ContentType.APPLICATION_JSON));
			if (this.originHost != null) {
				request.addHeader("Origin", this.originHost);
			}
			
			HttpResponse resp = this.httpclient.execute(request, null).get();
			
			StatusLine status = resp.getStatusLine();
			HttpEntity entity = resp.getEntity();
			String contentType = entity.getContentType().getValue();
			if (status.getStatusCode() != 200) {
				LOGGER.error(String.format("%d response received from server", status.getStatusCode()));
				return null;
			}
			
			String content = EntityUtils.toString(entity);
			if (contentType.indexOf("application/json") >= 0) {
				URLShortenerResponse shortenerResp = OBJECT_MAPPER.readValue(content, URLShortenerResponse.class);
				return shortenerResp.id;
			} else {
				LOGGER.error(String.format("Expecting a json response, but actually got a %s reponse with content: %s", contentType, content));
				return null;
			}
		} catch (InterruptedException | ExecutionException | IOException e) {
			LOGGER.error(e.getMessage(), e);
		}
		return null;
	}
	
	/* (non-Javadoc)
	 * @see java.io.Closeable#close()
	 */
	@Override
	public void close() throws IOException {
		this.httpclient.close();
	}

}
