package org.iworkz.genesis.impl;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.annotation.Annotation;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.inject.Inject;
import javax.inject.Provider;
import javax.inject.Qualifier;
import javax.inject.Scope;
import javax.inject.Singleton;

import org.iworkz.common.exception.GenesisException;
import org.iworkz.common.helper.ReflectionHelper;
import org.iworkz.genesis.ImplementationClassProcessor;
import org.iworkz.genesis.Injector;
import org.iworkz.genesis.Module;
import org.iworkz.genesis.PostProcessor;
import org.iworkz.genesis.ValueSupplier;
import org.iworkz.genesis.impl.scope.ScopeContext;

public abstract class AbstractInjector implements Injector {

    private ReflectionHelper reflectionHelper; // prelimanry instance is created during setup and final one in postSetup

    private Map<Class<?>, Supplier<?>> suppliers;

    private Map<Class<?>, Binding<?>> bindings;
    
    private Map<Class<? extends Annotation>, ValueSupplier<?,?>> valueSuppliers;
    
    private Set<Class<? extends Annotation>> singletonAnnotations;

    private Map<Class<?>, ScopeContext> contexts;

    private Set<Module> modules;

    private Set<ClassLoader> registeredClassLoaders;

    private Set<ImplementationFinder> implementationFinders;

    private Set<ImplementationClassProcessor> implementationClassProcessors;

    private Set<PostProcessor> postProcessors;

    protected final Module[] modulesForSetup;

    protected AbstractInjector(Module... modules) {
        this.modulesForSetup = modules;
    }

    public Module[] getConfiguredModules() {
        return this.modulesForSetup;
    }

    protected void setup() {

        Module[] configuredModules = getConfiguredModules();

        this.modules = new HashSet<>();
        if (configuredModules != null && configuredModules.length > 0) {
            for (Module module : configuredModules) {
                this.modules.add(module);
            }
        }

        this.bindings = new HashMap<>();
        this.suppliers = new HashMap<>();
        singletonAnnotations = new LinkedHashSet<>();
        this.valueSuppliers = new HashMap<>();
        this.contexts = new HashMap<>();

        this.suppliers.put(Injector.class, new Supplier<>(this));

        this.registeredClassLoaders = new LinkedHashSet<>();
        this.implementationFinders = new LinkedHashSet<>();
        this.postProcessors = new LinkedHashSet<>();
        this.implementationClassProcessors = new LinkedHashSet<>();
        if (configuredModules != null) {
            /* first part of modules setup */
            for (Module module : configuredModules) {
                module.configure();
                if (module.getClassLoaders() != null) {
                    this.registeredClassLoaders.addAll(module.getClassLoaders());
                }
                if (module.getBindings() != null) {
                    for (Class<?> singletonClass : module.getBindings().keySet()) {
                        Binding<?> binding = module.getBindings().get(singletonClass);
                        this.bindings.put(singletonClass, binding);
                        // TODO merge annotation + annotationClass bindings
                    }
                }
                if (module.getScopes() != null) {
                    for (Class<?> singletonClass : module.getScopes().keySet()) {
                        ScopeContext context = module.getScopes().get(singletonClass);
                        this.contexts.put(singletonClass, context);
                    }
                }
                
                if (module.getSingletonScopes() != null) {
                	singletonAnnotations.addAll(module.getSingletonScopes());
                }
                
                if (module.getValueSuppliers() != null) {
                    for (Class<? extends Annotation> annotationClass : module.getValueSuppliers().keySet()) {
                        ValueSupplier<?, ?> valueSupplier = module.getValueSuppliers().get(annotationClass);
                        this.valueSuppliers.put(annotationClass, valueSupplier);
                    }
                }
                if (module.getImplementationFinders() != null) {
                    for (ImplementationFinder implementationFinder : module.getImplementationFinders()) {
                        if (implementationFinder instanceof AbstractImplementationFinder && this.reflectionHelper != null) {
                            ((AbstractImplementationFinder) implementationFinder).setReflectionHelper(this.reflectionHelper);
                        }
                    }
                    this.implementationFinders.addAll(module.getImplementationFinders());
                }
                if (module.getImplementationClassProcessors() != null) {
                    this.implementationClassProcessors.addAll(module.getImplementationClassProcessors());
                }
                if (module.getPostProcessors() != null) {
                    this.postProcessors.addAll(module.getPostProcessors());
                }
            }
        }

        /* create preliminary ReflectionHelper and set it to the implementationFinders */
        createPreliminaryReflectionHelper();
        configureImplementationFinders();

    }

    /**
     * Supports the bootstrapping of the Injector.
     * Here the injector can already be used (e.g. for internal injections) but it is
     * not available for external use.
     */
    protected void postSetup() {
        /*
         * use the preliminary ReflectionHandler to inject the final ReflectionHandler and overwrite it in the
         * implementationFinders
         */
        createFinalReflectionHelper();
        configureImplementationFinders();
    }

    /*
     * Can not inject this internal used ReflectionHelper because it is needed for setup.
     */
    protected void createPreliminaryReflectionHelper() {
        this.reflectionHelper = new ReflectionHelper();
    }

    protected void createFinalReflectionHelper() {
        this.reflectionHelper = getInstance(ReflectionHelper.class);
    }

    protected void configureImplementationFinders() {
        if (this.implementationFinders != null) {
            for (ImplementationFinder implementationFinder : this.implementationFinders) {
                if (implementationFinder instanceof AbstractImplementationFinder) {
                    ((AbstractImplementationFinder) implementationFinder).setReflectionHelper(this.reflectionHelper);
                }
            }
        }
    }

    @Override
    public void injectMembers(Object instance) {
        if (instance != null) {
            if (this.modules == null) {
                setup();
                postSetup();
            }
            InjectionContext ctx = new InjectionContext();
            injectMembers(instance, instance.getClass(), ctx);
            postProcess(ctx);
        }
    }

    protected void postProcess(InjectionContext ctx) {
        if (ctx.createdInstances != null && this.postProcessors != null) {
            for (PostProcessor postProcessor : this.postProcessors) {
                postProcess(postProcessor, ctx.createdInstances);
            }
        }
    }

    public void postProcess(PostProcessor postProcessor, Map<Object, Set<Object>> createdInstances) {
        for (Entry<Object, Set<Object>> entry : createdInstances.entrySet()) {
            postProcessor.process(entry.getKey(), entry.getValue());
        }
    }
    
    protected boolean isInjected(AccessibleObject field) {
		Annotation injectAnnotation = field.getAnnotation(Inject.class);
		if (injectAnnotation != null) {
			return true;
		}
		Annotation jakartaInjectAnnotation = field.getAnnotation(jakarta.inject.Inject.class);
		if (jakartaInjectAnnotation != null) {
			return true;
		}
		if (!valueSuppliers.isEmpty()) {
			for (Class<? extends Annotation> valueSupplierClass : valueSuppliers.keySet()) {
				Annotation valueSupplierAnnotation = field.getAnnotation(valueSupplierClass);
				if (valueSupplierAnnotation != null) {
					return true;
				}
			}
		}
		return false;
    }

    protected void injectMembers(Object instance, Class<?> classWithMembers, InjectionContext ctx) {
        /* inject fields */
        for (Field field : this.reflectionHelper.getAllFields(classWithMembers)) {
            if (isInjected(field)) {
                try {
                    field.setAccessible(true);
                    Class<?> fieldClass = field.getType();
                    boolean isProvider = false;
                    if (Provider.class == fieldClass || jakarta.inject.Provider.class == fieldClass) {
                        isProvider = true;
                        Type genericType = field.getGenericType();
                        if (genericType instanceof ParameterizedType) {
                            ParameterizedType parameterizedType = (ParameterizedType) genericType;
                            Type typeArgument = parameterizedType.getActualTypeArguments()[0];
                            if (typeArgument instanceof Class) {
                                fieldClass = (Class<?>) typeArgument;
                            }
                        }
                    }
                    Object injectedObject = get(fieldClass, isProvider, field, ctx);
                    ctx.putInjectedObject(instance, injectedObject);
                    field.set(instance, injectedObject);
                } catch (Exception e) {
                    throw new GenesisException("Can not inject field '" + field.getName() + "' of class '"
                            + instance.getClass().getCanonicalName() + "'", e);
                }
            }
        }
        /* inject setters */
        try {
            BeanInfo beanInfo = Introspector.getBeanInfo(classWithMembers);
            for (PropertyDescriptor pd : beanInfo.getPropertyDescriptors()) {
                // System.out.println("Property name = "+beanInfo.getBeanDescriptor().getName()+"."+ pd.getName());
                Method setter = pd.getWriteMethod();
                if (setter != null) {
                    setter.setAccessible(true);
                    if (isInjected(setter)) {
                        Parameter parameter = setter.getParameters()[0];
                        Class<?> parameterClass = setter.getParameterTypes()[0];
                        boolean isProvider = false;
                        if (Provider.class == parameterClass || jakarta.inject.Provider.class == parameterClass) {
                            isProvider = true;
                            Type genericType = setter.getGenericParameterTypes()[0];
                            if (genericType instanceof ParameterizedType) {
                                ParameterizedType parameterizedType = (ParameterizedType) genericType;
                                Type typeArgument = parameterizedType.getActualTypeArguments()[0];
                                if (typeArgument instanceof Class) {
                                    parameterClass = (Class<?>) typeArgument;
                                }
                            }
                        }
                        Object injectedObject = get(parameterClass, isProvider, parameter, ctx);
                        ctx.putInjectedObject(instance, injectedObject);
                        setter.invoke(instance, injectedObject);
                    }
                }
            }
        } catch (Exception e) {
            throw new GenesisException("Can not inject setter of class '" + instance.getClass().getCanonicalName() + "'", e);
        }
    }

    @Override
    public <T> T getInstance(Class<T> instanceClass) {
        if (instanceClass != null) {
            if (this.modules == null) {
                setup();
                postSetup();
            }
            InjectionContext ctx = new InjectionContext();


            T t = get(instanceClass, false, null, ctx);
            postProcess(ctx);
            return t;
        }
        return null;
    }

    @Override
    public <T> Class<? extends T> getImplementationClass(String className) {
        Class<? extends T> injectedClass = loadClass(className);
        return getImplementationClass(injectedClass);
    }

    public <T> Class<? extends T> loadClass(String className) {

        if (className != null) {
            if (this.modules == null) {
                setup();
                postSetup();
            }
            Class<? extends T> injectedClass = null;
            for (ClassLoader classLoader : this.registeredClassLoaders) {
                try {
                    injectedClass = this.reflectionHelper.load(classLoader, className);
                } catch (ClassNotFoundException e) {
                    // ignore
                }
            }
            return injectedClass;
        }
        return null;
    }

    @Override
    public <T> T getInstance(String className) {
        if (className != null) {
            Class<? extends T> injectedClass = loadClass(className);
            if (injectedClass != null) {
                return getInstance(injectedClass);
            } else {
                throw new GenesisException("Class not found '" + className + "'");
            }
        }
        return null;

    }
    
    protected Annotation getQualifierAnnotation(AnnotatedElement annotatedElement) {
        for (Annotation elementAnnotation : annotatedElement.getAnnotations()) {
            Class<?> annotationType = elementAnnotation.annotationType();
            if (annotationType.isAnnotationPresent(Qualifier.class)) {
                return elementAnnotation;
            }
            if (annotationType.isAnnotationPresent(jakarta.inject.Qualifier.class)) {
                return elementAnnotation;
            }
        }
        return null;
    }
    
    @SuppressWarnings("unchecked")
    protected <T> T get(Class<T> instanceClass, boolean isProvider, AnnotatedElement annotatedElement, InjectionContext ctx) {
        Annotation annotation = null;
        if (annotatedElement != null) {
        	annotation = getNameAnnotation(annotatedElement);
            if (annotation == null) {
            	annotation = getQualifierAnnotation(annotatedElement);
                if (annotation == null && annotatedElement instanceof Parameter) {
                	/* try to get qualifier annotation from declaring executable (=Method) */
                	Parameter parameter = (Parameter)annotatedElement;
                	if (parameter.getDeclaringExecutable() != null) {
                		annotation = getQualifierAnnotation(parameter.getDeclaringExecutable());
                	}
                }
            } 
        }
        if (ctx.creationStack.contains(instanceClass)) {
            throw new GenesisException("Circular dependency detected during creation of '" + instanceClass + "', stack = "
                    + ctx.creationStack.toString());
        }
        Supplier<T> supplier = (Supplier<T>) this.suppliers.get(instanceClass);
        if (supplier == null) {
            Binding<T> binding = (Binding<T>) this.bindings.get(instanceClass);
            if (binding != null) {
                /* standard provider */
                supplier = new Supplier<>(instanceClass, binding.implementationClass, binding.getInstance(), this,
                        binding.getScope());
                supplier.setImplementationClassProcessors(implementationClassProcessors);
                /* annotation providers */
                if (binding.nameBindings != null) {
                    supplier.annotationProviders = new HashMap<>();
                    for (String annotationInstance : binding.nameBindings.keySet()) {
                        Binding<T> annotationBinding = binding.nameBindings.get(annotationInstance);
                        supplier.annotationProviders.put(annotationInstance,
                                new Supplier<>(instanceClass, annotationBinding.implementationClass,
                                        annotationBinding.getInstance(), this, annotationBinding.getScope()));
                    }
                }
                /* annotation class providers */
                if (binding.annotationClassBindings != null) {
                    supplier.annotationClassProviders = new HashMap<>();
                    for (Class<? extends Annotation> annotationClass : binding.annotationClassBindings.keySet()) {
                        Binding<T> annotationClassBinding = binding.annotationClassBindings.get(annotationClass);
                        supplier.annotationClassProviders.put(annotationClass,
                                new Supplier<>(instanceClass, annotationClassBinding.implementationClass,
                                        annotationClassBinding.getInstance(), this, annotationClassBinding.getScope()));
                    }
                }
            } else {
                /* default provider */
                supplier = new Supplier<>(instanceClass, null, null, this, null);
                supplier.setImplementationClassProcessors(implementationClassProcessors);
            }
            this.suppliers.put(instanceClass, supplier);
        }

        /* return javax Provider or instance */
        if (isProvider) {
            final Supplier<T> finalProvider = supplier;
            final Annotation finalAnnotation = annotation;
            final InjectionContext parentCtx = ctx;
            return (T) new Provider<T>() {
                @Override
                public T get() {
                    /* provider starts new injection context */
                    if (!parentCtx.creationStack.isEmpty()) {
                        throw new GenesisException(
                                "Provider 'get()' invoked before parent injection finished (probably in a constructor).");
                    }
                    InjectionContext ctx = new InjectionContext();
                    T t = (T) finalProvider.getInstance(AbstractInjector.this, finalAnnotation, ctx);
                    postProcess(ctx);
                    return t;
                }
            };
        } else {
            return (T) supplier.getInstance(this, annotation, ctx);
        }
    }

    @Override
    public <T> Class<? extends T> getImplementationClass(Class<T> injectedClass) {
        Class<? extends T> implementationClass = null;
        for (ClassLoader classLoader : this.registeredClassLoaders) {
            for (ImplementationFinder implementationFinder : this.implementationFinders) {
                Class<? extends T> implementationClassFromModule = implementationFinder.find(classLoader, injectedClass);
                if (implementationClassFromModule != null) {
                    implementationClass = implementationClassFromModule;
                }
            }
        }
        if (implementationClass != null) {
            return implementationClass;
        } else {
            return injectedClass;
        }
    }

    public ScopeContext getContext(Class<?> scope) {
        ScopeContext context = this.contexts.get(scope);
        if (context == null) {
            throw new GenesisException("No context defined for scope '" + scope.getCanonicalName() + "'");
        }
        return context;
    }

	protected <T extends Annotation> ValueSupplier<T,?> getValueSupplier(Class<T> annotation) {
		return (ValueSupplier<T,?>) valueSuppliers.get(annotation);
	}
	
    protected boolean isScopeAnnotation( Class<? extends Annotation> annotationType) {
        if (annotationType.isAnnotationPresent(Scope.class)) {
            return true;
        }
        if (annotationType.isAnnotationPresent(jakarta.inject.Scope.class)) {
            return true;
        } 
        
    	for (Class<? extends Annotation> singletonAnnotation : singletonAnnotations) {
    		if (annotationType.equals(singletonAnnotation)) {
    			return true;
    		}
    	}
        
        return false;
    }
    
    protected boolean isSingleton(Class<? extends Annotation> scopeAnnotationType) {
    	if (Singleton.class.equals(scopeAnnotationType)) {
    		return true;
    	}
    	if (jakarta.inject.Singleton.class.equals(scopeAnnotationType)) {
    		return true;
    	}
    	return singletonAnnotations.contains(scopeAnnotationType);
    }
    
    
    protected Annotation getNameAnnotation(AnnotatedElement annotatedElement) {
        if (annotatedElement.isAnnotationPresent(javax.inject.Named.class)) {
            return annotatedElement.getAnnotation(javax.inject.Named.class);
        } else if (annotatedElement.isAnnotationPresent(jakarta.inject.Named.class)) {
            return annotatedElement.getAnnotation(jakarta.inject.Named.class);
        } else {
        	return null;
        }
    }

}
