package org.iworkz.genesis.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.iworkz.genesis.Injector;
import org.iworkz.genesis.InjectorFactory;
import org.iworkz.genesis.Module;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AbstractInjectorFactory implements InjectorFactory {
	
	private static final Logger logger = LoggerFactory.getLogger(AbstractInjectorFactory.class);

	protected final List<Module> registeredModules = new ArrayList<>();

	private Injector instance;
	
	private boolean bulkRegistration = false;
	
	public AbstractInjectorFactory() {
		registerModule(new AbstractModule() {
//			@Override
//			protected void registerClassLoader(ClassLoader classLoader) {
//				// Do not register classloaders for anonymous class
//			}
		});
	}
	
	public void registerModules(Module ...modules) {
		if (modules != null) {
			bulkRegistration = true;
			for (Module module : modules) {
				registerModule(module);
			}
			bulkRegistration = false;
		}
		updateInjector();
	}

	public void registerModule(Module module) {
		synchronized (registeredModules) {
			logger.debug("register module: {}", module.getClass().getCanonicalName());
			registeredModules.add(module);
			if (!bulkRegistration) {
				updateInjector();
			}
		}
	}
	
	public void unregisterModule(Module module) {
		synchronized (registeredModules) {
			logger.debug("unregister module: {}", module.getClass().getCanonicalName());
			registeredModules.remove(module);
			updateInjector();
		}
	}
	
	@Override
	public Injector getInjector() {
		if (instance == null) {
			instance = new AbstractInjector() {
				@Override
				public Module[] getConfiguredModules() {
					return AbstractInjectorFactory.this.getConfiguredModules();
				}
			};
		}
		return instance;
	}

	protected void updateInjector() {
		logger.debug("updateInjector");
		instance = null;
	}

	protected Module[] getConfiguredModules() {
		synchronized (registeredModules) {
			List<Module> allModules = allModules();
			List<Module> filteredModules = filterModules(allModules);
			sortModules(filteredModules);
			for (Module filteredModule : filteredModules ) {
				logger.debug("filtered module: {} (ranking={})", filteredModule.getClass().getCanonicalName(), filteredModule.getRanking());
			}
			return filteredModules.toArray(new Module[filteredModules.size()]);
		}
	}
	
	protected List<Module> allModules() {
		Set<Module> additionalDependencies = new HashSet<>();
		/* collect only dependencies which are not implemented by another registered module */
		for (Module module : registeredModules) {
			if (module.getDependencies() != null) {
				for (Module dependencyModule : module.getDependencies()) {
					boolean isExtendedByAnotherModule = false;
					for (Module aModule : registeredModules) {
						if (dependencyModule.getClass().isInstance(aModule)) {
							isExtendedByAnotherModule = true;
							break;
						}
					}
					if (!isExtendedByAnotherModule) {
						additionalDependencies.add(dependencyModule);
					}
				}
			}
		}
		/* combine dependency modules and registered modules */
		List<Module> allModules = new ArrayList<>();
		allModules.addAll(additionalDependencies);
		allModules.addAll(registeredModules);
		return allModules;
	}
	
	protected List<Module> filterModules(List<Module> modules) {
		List<Module> filteredModules = new ArrayList<>();
		for (Module module : modules) {
			boolean isExtendedByAnotherModule = false;
			for (Module anotherModule : modules) {
				if (module != anotherModule) {
					if (module.getClass().isInstance(anotherModule)) {
						isExtendedByAnotherModule = true;
						break;
					}
				}
			}
			if (!isExtendedByAnotherModule) {
				filteredModules.add(module);
			}
		}
		return filteredModules;
	}
	
	protected void sortModules(List<Module> modules) {
		Collections.sort(modules, new Comparator<Module>() {
			@Override
			public int compare(Module m1, Module m2) {
				if (m1.getRanking() < m2.getRanking()) {
					return -1;
				} else if (m1.getRanking() > m2.getRanking()) {
					return 1;
				}
				return 0;
			}
		});
	}

}
