package org.iworkz.genesis.impl;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import org.iworkz.common.exception.GenesisException;
import org.iworkz.common.helper.ReflectionHelper;
import org.iworkz.genesis.ImplementationClassProcessor;
import org.iworkz.genesis.Injector;
import org.iworkz.genesis.Module;
import org.iworkz.genesis.PostProcessor;
import org.iworkz.genesis.ValueSupplier;
import org.iworkz.genesis.impl.scope.ScopeContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractModule implements Module {

    private static final Logger logger = LoggerFactory.getLogger(AbstractModule.class);

    private static Map<Class<? extends Module>, Module> moduleInstanceMap = new IdentityHashMap<>();

    private Map<Class<?>, Binding<?>> bindings;
    private Map<Class<?>, ScopeContext> scopes;
    private List<Class<? extends Annotation>> singletonAnnotations;
    
    private Map<Class<? extends Annotation>, ValueSupplier<?,?>> valueSuppliers;


    private Set<ClassLoader> registeredClassLoaders;
    private Set<ImplementationFinder> implementationFinders;
    private Set<PostProcessor> postProcessors;

    private Set<ImplementationClassProcessor> implementationClassProcessors;


    private int ranking;

    protected final Set<Module> dependencies = new HashSet<>();

    private final ImplementationFinder standardImplementationFinder = new AbstractImplementationFinder() {
        @Override
        public <T> Class<? extends T> find(ClassLoader classLoader, Class<T> injectedClass) {
            Class<? extends T> foundClass = null;
            if (injectedClass.isInterface()) {
                foundClass = reflectionHelper.findClassWithPostfix(classLoader, injectedClass, "impl", "Impl");
            }
            return foundClass;
        }
    };

    private final PostProcessor standardPostProcessor = new PostProcessor() {
        ReflectionHelper reflectionHelper = new ReflectionHelper();

        @Override
        public void process(Object createdInstance, Set<Object> injectedObjects) {

            Method postConstructMethod =
                    reflectionHelper.findFirstDeclaredMethod(createdInstance.getClass(), new Predicate<Method>() {
                        @Override
                        public boolean test(Method method) {
                            Annotation jakartaPostConstructAnnotation = method.getAnnotation(jakarta.annotation.PostConstruct.class);
                            Annotation javaxPostConstructAnnotation = method.getAnnotation(javax.annotation.PostConstruct.class);
                            return jakartaPostConstructAnnotation != null || javaxPostConstructAnnotation != null;
                        }
                    });

            if (postConstructMethod != null) {
                try {
                    postConstructMethod.setAccessible(true);
                    postConstructMethod.invoke(createdInstance);
                } catch (final Exception e) {
                    throw new GenesisException("Invocation of post construct method '" + postConstructMethod.getName()
                            + "' failed (class='" + createdInstance + ")", e);
                }
            }

        }
    };

    private final ImplementationClassProcessor standardImplementationClassProcessor = new ImplementationClassProcessor() {

        @Override
        public Class process(Class implementationClass, Injector injector, InjectionContext ctx) {
            return implementationClass;
        }

    };

    protected static <T extends Module> T getInstance(Class<T> moduleClass) {
        synchronized (moduleInstanceMap) {
            T t = (T) moduleInstanceMap.get(moduleClass);
            if (t == null) {
                try {
                    t = moduleClass.newInstance();
                } catch (Exception e) {
                    throw new GenesisException("Module instance can not be created", e);
                }
                moduleInstanceMap.put(moduleClass, t);
            }
            return t;
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public void configure() {
        /* register ClassLoaders */
        Class<? extends Module> moduleClass = getClass();
        while (moduleClass != null) {
            registerClassLoader(moduleClass.getClassLoader());
            Class<?> superClass = moduleClass.getSuperclass();
            if (Module.class.isAssignableFrom(superClass)) {
                moduleClass = (Class<? extends Module>) superClass;
            } else {
                moduleClass = null;
            }
        }
        registerImplementationClassProcessor(getStandardImplementationClassProcessor());
        registerImplementationFinder(getStandardImplementationFinder());
        registerPostProcessor(getStandardPostProcessor());
    }

    protected ImplementationFinder getStandardImplementationFinder() {
        return standardImplementationFinder;
    }

    protected PostProcessor getStandardPostProcessor() {
        return standardPostProcessor;
    }

    protected ImplementationClassProcessor getStandardImplementationClassProcessor() {
        return standardImplementationClassProcessor;
    }

    protected void registerImplementationFinder(ImplementationFinder implementationFinder) {
        if (implementationFinders == null) {
            implementationFinders = new LinkedHashSet<>();
        }
        implementationFinders.add(implementationFinder);
    }

    protected void registerImplementationClassProcessor(ImplementationClassProcessor implementationClassProcessor) {
        if (implementationClassProcessors == null) {
            implementationClassProcessors = new LinkedHashSet<>();
        }
        implementationClassProcessors.add(implementationClassProcessor);
    }

    @Override
    public Set<ImplementationClassProcessor> getImplementationClassProcessors() {
        return implementationClassProcessors;
    }

    @Override
    public Set<ImplementationFinder> getImplementationFinders() {
        return implementationFinders;
    }

    protected void addDependency(Module moduleClass) {
        dependencies.add(moduleClass);
    }

    @Override
    public Set<Module> getDependencies() {
        return dependencies;
    }

    @SuppressWarnings("unchecked")
    protected <T> Binding<T> bind(Class<T> singletonClass) {
        if (bindings == null) {
            bindings = new HashMap<>();
        }

        Binding<T> binding = (Binding<T>) bindings.get(singletonClass);
        if (binding == null) {
            binding = new Binding<>(singletonClass);
            bindings.put(singletonClass, binding);
        }
        return binding;
    }

    protected void bindScope(Class<?> scopeAnnotationClass, ScopeContext scopeClass) {
        if (scopes == null) {
            scopes = new HashMap<>();
        }
        scopes.put(scopeAnnotationClass, scopeClass);
    }
    
    protected void bindSingletonScope(Class<? extends Annotation> singletonAnnotationClass) {
        if (singletonAnnotations == null) {
        	singletonAnnotations = new ArrayList<>();
        }
        singletonAnnotations.add(singletonAnnotationClass);
    }
    
    protected void bindValueSupplier(Class<? extends Annotation> supplierAnnotationClass, ValueSupplier<?,?> valueSupplier) {
        if (valueSuppliers == null) {
        	valueSuppliers = new HashMap<>();
        }
        valueSuppliers.put(supplierAnnotationClass, valueSupplier);
    }

    @Override
    public Map<Class<?>, Binding<?>> getBindings() {
        return bindings;
    }
    
    @Override
    public Map<Class<? extends Annotation>, ValueSupplier<?,?>> getValueSuppliers() {
        return valueSuppliers;
    }
    
    @Override
    public List<Class<? extends Annotation>> getSingletonScopes() {
    	return singletonAnnotations;
    }

    protected void registerClassLoader(ClassLoader classLoader) {
        if (registeredClassLoaders == null) {
            registeredClassLoaders = new HashSet<>();
        }
        registeredClassLoaders.add(classLoader);
    }

    @Override
    public Set<ClassLoader> getClassLoaders() {
        return registeredClassLoaders;
    }

    protected void registerPostProcessor(PostProcessor postProcessor) {
        if (postProcessors == null) {
            postProcessors = new HashSet<>();
        }
        postProcessors.add(postProcessor);
    }

    @Override
    public Set<PostProcessor> getPostProcessors() {
        return postProcessors;
    }

    @Override
    public int getRanking() {
        return ranking;
    }

    @Override
    public void setRanking(int ranking) {
        this.ranking = ranking;
    }

    protected void postProcess(Object instance, Set<Object> injectedObjects) {
    }

    @Override
    public Map<Class<?>, ScopeContext> getScopes() {
        return scopes;
    }

}
