package org.iworkz.genesis.impl;

import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Parameter;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Map;
import java.util.Set;

import javax.inject.Inject;
import javax.inject.Provider;

import org.iworkz.common.exception.GenesisException;
import org.iworkz.genesis.ImplementationClassProcessor;
import org.iworkz.genesis.ValueSupplier;
import org.iworkz.genesis.impl.scope.NoScope;
import org.iworkz.genesis.impl.scope.ScopeContext;

public class Supplier<T> {

    protected final Class<T> injectedClass;
    protected Class<? extends T> implementationClass;
    protected boolean implementationClassPrepared;
    protected boolean singleton;
    protected Class<? extends Annotation> scope;
    protected T instance;
    protected AbstractInjector injector;
    protected Map<String, Supplier<T>> annotationProviders;
    protected Map<Class<? extends Annotation>, Supplier<T>> annotationClassProviders;
    private Set<ImplementationClassProcessor> implementationClassProcessors;


    @SuppressWarnings("unchecked")
    public Supplier(T instance) {
        this((Class<T>) instance.getClass(), (Class<? extends T>) instance.getClass(), instance, null, null);
    }

    public Supplier(Supplier<T> supplier, boolean scoped) {
        injector = supplier.injector;
        injectedClass = supplier.injectedClass;
        instance = supplier.instance;
        implementationClass = supplier.implementationClass;
//		annotationProviders = supplier.annotationProviders;
//		annotationClassProviders = supplier.annotationClassProviders;
        if (scoped) {
            singleton = true; // singleton within scope
        } else {
            singleton = supplier.singleton;
            scope = supplier.scope;
        }
    }

    @SuppressWarnings("unchecked")
    public Supplier(Class<T> injectedClass, Class<? extends T> implementationClass, T instance, AbstractInjector injector,
                    Class<? extends Annotation> scope) {
        this.injector = injector;
        this.injectedClass = injectedClass;
        this.instance = instance;
        /* implementation class */
        if (instance != null) {
            this.implementationClass = (Class<? extends T>) instance.getClass();
        } else if (implementationClass != null) {
            this.implementationClass = implementationClass;
        } else {
            this.implementationClass = injector.getImplementationClass(injectedClass);
        }
        /* scope */
        if (scope != null) {
            if (scope != NoScope.class) {
                this.scope = scope;
            }
        } else if (this.implementationClass != null) {
            this.scope = getScopeAnnotationType(this.implementationClass);
        }
        if (this.scope != null) {
            singleton = injector.isSingleton(this.scope);
        }
    }

    private Class<? extends Annotation> getScopeAnnotationType(Class<? extends T> implementationClass) {
        for (Annotation classAnnotation : implementationClass.getAnnotations()) {
            Class<? extends Annotation> annotationType = classAnnotation.annotationType();
            if (injector.isScopeAnnotation(annotationType)) {
                return annotationType;
            }
        }
        return null;
    }

    public Object getInstance(AbstractInjector injector, Annotation annotation, InjectionContext ctx) {

        /* Supplier for annotation */
        if (annotation != null) {
            if (annotationProviders != null) {
            	String name = getAnnotatedName(annotation);
            	if (name != null) {
            		Supplier<T> annotationProvider = annotationProviders.get(name);
            		if (annotationProvider != null) {
            			return annotationProvider.getInstance(injector, null, ctx);
            		}
            	}
            }
            if (annotationClassProviders != null) {
                Supplier<T> annotationClassProvider = annotationClassProviders.get(annotation.annotationType());
                if (annotationClassProvider != null) {
                    return annotationClassProvider.getInstance(injector, null, ctx);
                }
            }
            Object value = getValueFromSupplier(annotation);
            if (value != null) {
            	return value;
            }
            throw new GenesisException("No binding found for annotation " + annotation.annotationType().getCanonicalName()
                    + " (injected class: " + this.injectedClass.getCanonicalName() + " )");
        }
        /* Supplier for scope */
        if (scope != null && singleton == false && injectedClass != null) {
            ScopeContext scopeContext = injector.getContext(scope);
            Supplier<T> scopedSupplier = scopeContext.get(injectedClass, this, implementationClassProcessors);
            return scopedSupplier.getInstance(injector, null, ctx);
        }

        if (instance != null) {
            return instance;
        }
        prepareImplementationClass(ctx);
        ctx.creationStack.push(injectedClass);
        T newInstance = createInstance(injector, ctx);
        if (singleton) {
            instance = newInstance;
        }
        ctx.creationStack.pop();
        injector.injectMembers(newInstance, newInstance.getClass(), ctx);
        return newInstance;
    }
    
	protected String getAnnotatedName(Annotation annotation) {
		if (annotation instanceof jakarta.inject.Named jakartaNamed) {
			return jakartaNamed.value();
		}
		if (annotation instanceof javax.inject.Named javaxNamed) {
			return javaxNamed.value();
		}
		return null;
	}

	protected <A extends Annotation> Object getValueFromSupplier(A annotation) {
    	Class<A> annotationType = (Class<A>)annotation.annotationType();
    	ValueSupplier<A,?> valueSupplier = injector.getValueSupplier(annotationType);
        if (valueSupplier != null) {
        	return valueSupplier.getValue(annotation, scope);
        } else {
        	return null;
        }
    }

    protected void prepareImplementationClass(InjectionContext ctx) {
        if (!implementationClassPrepared) {
            if (implementationClass == null) {
                implementationClass = injectedClass;
//			throw new IllegalArgumentException("The implementation class must be defined");
            }
            if (implementationClassProcessors != null) {
                for (ImplementationClassProcessor implementationClassProcessor : implementationClassProcessors) {
                    implementationClass = implementationClassProcessor.process(implementationClass, injector, ctx);
                }
            }
            implementationClassPrepared = true;
        }
    }

    protected T createInstance(AbstractInjector injector, InjectionContext ctx) {
        try {
            Constructor<? extends T> constructor = getConstructor(implementationClass);
            if (constructor != null) {
                constructor.setAccessible(true);
                if (constructor.getParameterCount() > 0) {
                    Parameter[] parameters = constructor.getParameters();
                    Type[] genericParameterTypes = constructor.getGenericParameterTypes();
                    Object[] initargs = new Object[parameters.length];
                    for (int i = 0; i < parameters.length; i++) {
                        Parameter parameter = parameters[i];

                        Class<?> parameterClass = parameter.getType();
                        boolean isProvider = false;
                        if (Provider.class == parameterClass || jakarta.inject.Provider.class == parameterClass) {
                            isProvider = true;
                            Type genericType = genericParameterTypes[i];
                            if (genericType instanceof ParameterizedType) {
                                ParameterizedType parameterizedType = (ParameterizedType) genericType;
                                Type typeArgument = parameterizedType.getActualTypeArguments()[0];
                                if (typeArgument instanceof Class) {
                                    parameterClass = (Class<?>) typeArgument;
                                }
                            }
                        }
                        Object parameterValue = injector.get(parameterClass, isProvider, parameter, ctx);
                        if (isProvider) {
                        	  initargs[i] = parameterValue;
                        } else {
                        	  initargs[i] = convertParameter(parameterClass, parameterValue);
                        }
                      
                    }
                    T t = constructor.newInstance(initargs);
                    ctx.putInjectedObjects(t, initargs);
                    return t;
                } else {
                    T t = constructor.newInstance();
                    ctx.createdInstances.put(t, null);
                    return t;
                }
            } else {
                throw new GenesisException(
                        "No suitable constructor found for injection of class: " + implementationClass.getCanonicalName());
            }
        } catch (Exception e) {
            throw new GenesisException("Can not create singleton for class '" + implementationClass.getCanonicalName() + "'", e);
        }
    }
    
    protected Object convertParameter(Class<?> parameterClass, Object parameterValue) {
    	if (parameterValue == null) {
    		return null;
    	}
    	Class<?> classOfParameterValue = parameterValue.getClass();
    	if (!parameterClass.isAssignableFrom(classOfParameterValue)) {
    		if (parameterValue instanceof String) {
    			if (parameterClass.equals(Integer.class)) {
    				return Integer.parseInt((String)parameterValue);
    			} else if (parameterClass.equals(Long.class)) {
    				return Long.parseLong((String)parameterValue);
    			} else if (parameterClass.equals(Boolean.class)) {
    				return Boolean.parseBoolean((String)parameterValue);
    			}
    		}
    		throw new GenesisException("Failed to convert parameter of type " + classOfParameterValue.getCanonicalName() + " to " + parameterClass.getCanonicalName());
    	}
    	return parameterValue;
    }

    @SuppressWarnings("unchecked")
    public Constructor<? extends T> getConstructor(Class<? extends T> implementationClass) {
        Constructor<?> matchingConstructor = null;
        for (Constructor<?> c : implementationClass.getDeclaredConstructors()) {
            if (c.isAnnotationPresent(Inject.class)) {
                return (Constructor<? extends T>) c;
            }
            if (c.isAnnotationPresent(jakarta.inject.Inject.class)) {
                return (Constructor<? extends T>) c;
            }
            if (c.getParameterCount() == 0) {
                matchingConstructor = c;
            }
        }
        if (matchingConstructor == null && implementationClass.isInterface()) {
            throw new GenesisException("Can not create singleton of interface '" + implementationClass.getCanonicalName() + "'");
        }
        return (Constructor<? extends T>) matchingConstructor;
    }

    public void setImplementationClassProcessors(Set<ImplementationClassProcessor> implementationClassProcessors) {
        this.implementationClassProcessors = implementationClassProcessors;
    }

}
