package org.iworkz.genesis.impl.scope;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.iworkz.common.exception.GenesisException;
import org.iworkz.genesis.ImplementationClassProcessor;
import org.iworkz.genesis.impl.Supplier;


public class ScopeContext {

	private final ThreadLocal<Map<Class<?>, Supplier<?>>> suppliers = new ThreadLocal<>();
	
	public void enter() {
		if (suppliers.get() != null) {
			throw new GenesisException("A scoping block is already in progress");
		}
		suppliers.set(new HashMap<>());
	}

	public void exit() {
		if (suppliers.get() == null) {
			throw new GenesisException("No scoping block in progress");
		}
		suppliers.remove();
	}

	@SuppressWarnings("unchecked")
	public <T> Supplier<T> get(Class<T> injectedClass, Supplier<T> unscoped, Set<ImplementationClassProcessor> implementationClassProcessors) {
		Map<Class<?>, Supplier<?>> scopedSuppliersMap = getScopedSuppliersMap(injectedClass);
		Supplier<T> scopedSupplier = (Supplier<T>) scopedSuppliersMap.get(injectedClass);
		if (scopedSupplier == null) {
			scopedSupplier = new Supplier<>(unscoped,true);
			scopedSupplier.setImplementationClassProcessors(implementationClassProcessors);
			scopedSuppliersMap.put(injectedClass, scopedSupplier);
		}
		return scopedSupplier;
	}
	
	@SuppressWarnings({ "rawtypes", "unchecked" })
	public <T> void seed(Class<? super T> injectedClass, T value) {
		Map<Class<?>, Supplier<?>> scopedObjects = getScopedSuppliersMap(injectedClass);
		if (scopedObjects.containsKey(injectedClass)) {
			throw new GenesisException("A value for the key "+injectedClass.getCanonicalName()+" was already seeded in this scope: '"+scopedObjects.get(injectedClass)+"'");
		}
		scopedObjects.put(injectedClass, new Supplier(injectedClass,value.getClass(),value,null,NoScope.class));
	}

	protected <T> Map<Class<?>, Supplier<?>> getScopedSuppliersMap(Class<?> injectedClass) {
		Map<Class<?>, Supplier<?>> scopedSuppliersMap = suppliers.get();
		if (scopedSuppliersMap == null) {
			throw new GenesisException("Cannot access " + injectedClass + " outside of a scoping block");
		}
		return scopedSuppliersMap;
	}

}
