package org.iworkz.habitat.command;

import java.sql.PreparedStatement;
import java.sql.ResultSet;

import javax.inject.Singleton;

import org.iworkz.habitat.dao.FieldNavigator;
import org.iworkz.habitat.dao.GenericDao;
import org.iworkz.habitat.dao.GenericDao.CommandCustomizer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class LoadExecutor extends AbstractExecutor {

    private static final Logger logger = LoggerFactory.getLogger(LoadExecutor.class);

    public <T> void load(GenericDao dao, CommandCustomizer statementAdapter, T obj, Object... key) {
        if (obj == null) {
            throw new IllegalArgumentException("Object to be loaded into must not be null");
        }

        Class<T> objectClass = (Class<T>) obj.getClass();

        String name = "load" + objectClass.getSimpleName() + statementAdapter;

        CommandMetaData commandMetaData = dao.metaDataAccess.getValue(name);
        if (commandMetaData == null) {
            commandMetaData = dao.metaDataAccess.getOrCreateValue(name, (value) -> {
                value.setCommand(this.commandBuilder.buildLoadCommand(dao.getEntityDefinition(), objectClass, statementAdapter));
            });
        }

        String sqlCommand = commandMetaData.getCommand();

        FieldNavigator ii = new FieldNavigator();
        PreparedStatement preparedStatement = null;
        RuntimeException notFoundException = null;
        try {
            preparedStatement = dao.getConnection().prepareStatement(sqlCommand);
            for (int i = 0; i < key.length; i++) {
                this.commandHelper.setParameter(preparedStatement, i + 1, key[i]);
            }
            ResultSet rs = preparedStatement.executeQuery();

            if (!commandMetaData.objectMetaDataProperty.isDefined()) {
                commandMetaData.objectMetaDataProperty.create((object) -> {
                    object.setPropertyDescriptors(this.commandHelper.createPropertyDescriptorsForClass(dao.getEntityDefinition(), objectClass, statementAdapter, rs), statementAdapter);
                });
            }

            if (rs.next()) {
                ii.reset();
                this.commandHelper.readFieldsFromResultSet(obj, commandMetaData.getObjectMetaData(), rs, ii);
                if (statementAdapter != null) {
                    statementAdapter.customReadResultSet(rs, ii, obj);
                }
                return;
            }
            notFoundException = new RuntimeException("Object to load not found");
            throw notFoundException;

        } catch (Exception e) {
            logger.error(sqlCommand);
            if (e == notFoundException) {
                throw this.exceptionFactory.createNotFoundException("Object not found in table " + dao.getEntityDefinition().getName());
            } else {
                throw new RuntimeException("Can not read record from table " + dao.getEntityDefinition().getName(), e);
            }
        } finally {
            if (preparedStatement != null) {
                try {
                    preparedStatement.close();
                } catch (Exception ex) {
                    logger.error("Prepared statement can not be closed", ex);
                }
            }
        }
    }

}
