package org.iworkz.habitat.command;

import java.sql.PreparedStatement;

import javax.inject.Singleton;

import org.iworkz.habitat.dao.ConnectionProvider;
import org.iworkz.habitat.dao.FieldNavigator;
import org.iworkz.habitat.dao.GenericDao;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class DeleteExecutor extends AbstractExecutor {

    private static final Logger logger = LoggerFactory.getLogger(DeleteExecutor.class);

    public void execute(ConnectionProvider task, GenericDao dao, Object... key) {

        String name = "_deleteObject";
        CommandMetaData commandMetaData = dao.metaDataAccess.getValue(name);
        if (commandMetaData == null) {
            commandMetaData = dao.metaDataAccess.getOrCreateValue(name, (value) -> {
                value.setCommand(this.commandBuilder.buildDeleteCommand(dao.getEntityDefinition()));
            });
        }

        String sqlCommand = commandMetaData.getCommand();

        PreparedStatement preparedStatement = null;
        try {

            preparedStatement = getConnection(task, dao).prepareStatement(sqlCommand);
            for (int i = 0; i < key.length; i++) {
                this.commandHelper.setParameter(preparedStatement, i + 1, key[i]);
            }

            int res = preparedStatement.executeUpdate();
            if (res == 0) {
                throw this.exceptionFactory.createConflictException("Record could not be deleted (maybe somebody else updated or deleted it meanwhile)");
            }

        } catch (Exception e) {
            logger.error(sqlCommand);
            throw new RuntimeException("Can not delete record in table " + dao.getEntityDefinition().getName(), e);
        } finally {
            if (preparedStatement != null) {
                try {
                    preparedStatement.close();
                } catch (Exception ex) {
                    logger.error("Prepared statement can not be closed", ex);
                }
            }
        }

    }

    public void executeObject(GenericDao dao, Object obj) {

        String name = "_deleteObject";

        CommandMetaData commandMetaData = dao.metaDataAccess.getValue(name);
        if (commandMetaData == null) {
            commandMetaData = dao.metaDataAccess.getOrCreateValue(name, () -> {
                CommandMetaData value = new CommandMetaData(dao.getEntityDefinition());
                value.setCommand(this.commandBuilder.buildDeleteObjectCommand(dao.getEntityDefinition()));
                return value;
            });
        }

        //        CommandMetaData commandMetaData = dao.getCommandMetaDataMap().get(name);
        //        if (commandMetaData == null) {
        //            synchronized (dao.getCommandMetaDataMap()) {
        //                commandMetaData = dao.getCommandMetaDataMap().get(name);
        //                if (commandMetaData == null) {
        //                    commandMetaData = new CommandMetaData(dao.getEntityDefinition());
        //                    commandMetaData.setCommand(this.commandBuilder.buildDeleteObjectCommand(dao.getEntityDefinition()));
        //                    dao.getCommandMetaDataMap().put(name, commandMetaData);
        //                }
        //            }
        //        }
        String sqlCommand = commandMetaData.getCommand();

        FieldNavigator ii = new FieldNavigator();
        PreparedStatement preparedStatement = null;
        try {

            preparedStatement = dao.getConnection().prepareStatement(sqlCommand);

            if (!commandMetaData.objectMetaDataProperty.isDefined()) {
                commandMetaData.objectMetaDataProperty.create((object) -> {
                    object.setPropertyDescriptors(this.commandHelper.createPropertyDescriptorsForClass(dao.getEntityDefinition(), obj.getClass(), null, null), null);
                });
            }

            this.commandHelper.writeUpdateParameter(preparedStatement, ii, obj, commandMetaData);
            int res = preparedStatement.executeUpdate();
            if (res == 0) {
                throw this.exceptionFactory.createConflictException("Record could not be deleted (maybe somebody else updated or deleted it meanwhile)");
            }

        } catch (Exception e) {
            logger.error(sqlCommand);
            throw new RuntimeException("Can not insert record in table " + dao.getEntityDefinition().getName(), e);
        } finally {
            if (preparedStatement != null) {
                try {
                    preparedStatement.close();
                } catch (Exception ex) {
                    logger.error("Prepared statement can not be closed", ex);
                }
            }
        }
    }

    public void executeObject(GenericDao dao, String criteriaString, Object... criterias) {
        String name = "delete" + criteriaString;

        CommandMetaData commandMetaData = dao.metaDataAccess.getValue(name);
        if (commandMetaData == null) {
            commandMetaData = dao.metaDataAccess.getOrCreateValue(name, () -> {
                CommandMetaData value = new CommandMetaData(dao.getEntityDefinition());
                value.setCommand(this.commandBuilder.buildDeleteWhereCommand(dao.getEntityDefinition(), criteriaString));
                return value;
            });
        }

        //        CommandMetaData commandMetaData = dao.getCommandMetaDataMap().get(name);
        //        if (commandMetaData == null) {
        //            synchronized (dao.getCommandMetaDataMap()) {
        //                commandMetaData = dao.getCommandMetaDataMap().get(name);
        //                if (commandMetaData == null) {
        //                    commandMetaData = new CommandMetaData(dao.getEntityDefinition());
        //                    commandMetaData.setCommand(this.commandBuilder.buildDeleteWhereCommand(dao.getEntityDefinition(), criteriaString));
        //                    dao.getCommandMetaDataMap().put(name, commandMetaData);
        //                }
        //            }
        //        }
        String sqlCommand = commandMetaData.getCommand();

        PreparedStatement preparedStatement = null;
        try {
            preparedStatement = dao.getConnection().prepareStatement(sqlCommand);
            if (criteriaString != null) {
                for (int i = 0; i < criterias.length; i++) {
                    this.commandHelper.setParameter(preparedStatement, i + 1, criterias[i]);
                }
            }
            preparedStatement.executeUpdate();

        } catch (Exception e) {
            logger.error(sqlCommand);
            throw new RuntimeException("Can not read record from table " + dao.getEntityDefinition().getName(), e);
        } finally {
            if (preparedStatement != null) {
                try {
                    preparedStatement.close();
                } catch (Exception ex) {
                    logger.error("Prepared statement can not be closed", ex);
                }
            }
        }
    }

}
