package org.iworkz.habitat.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.iworkz.common.helper.CloneHelper;
import org.iworkz.common.helper.ReflectionHelper;
import org.iworkz.habitat.command.CommandMetaData;
import org.iworkz.habitat.command.DeleteExecutor;
import org.iworkz.habitat.command.FindExecutor;
import org.iworkz.habitat.command.InsertExecutor;
import org.iworkz.habitat.command.LoadExecutor;
import org.iworkz.habitat.command.SelectExecutor;
import org.iworkz.habitat.command.UpdateExecutor;
import org.iworkz.habitat.sync.SyncnronizedValueAcess;

@Singleton
public abstract class GenericDao extends EntityDao {

    protected final Map<String, CommandMetaData> commandMetaDataMap = new HashMap<>();

    public final SyncnronizedValueAcess<CommandMetaData> metaDataAccess;

    private IdGenerator<?> idGenerator;

    @Inject
    private ContextAccess contextAccess;

    @Inject
    protected ReflectionHelper reflectionHelper;

    @Inject
    protected ExceptionFactory exceptionFactory;

    @Inject
    protected CloneHelper beanHelper;

    @Inject
    protected InsertExecutor insertExecutor;

    @Inject
    protected UpdateExecutor updateExecutor;

    @Inject
    protected DeleteExecutor deleteExecutor;

    @Inject
    protected LoadExecutor loadExecutor;

    @Inject
    protected FindExecutor findExecutor;

    @Inject
    protected SelectExecutor selectExecutor;

    /**
     * Constructor.
     * Reads the entity definition.
     */
    public GenericDao() {
        this.metaDataAccess = new SyncnronizedValueAcess<CommandMetaData>(this.commandMetaDataMap) {
            @Override
            protected CommandMetaData createValue() {
                return new CommandMetaData(GenericDao.this.getEntityDefinition());
            }
        };
    }

    @SuppressWarnings("rawtypes")
    @Inject
    public void setIdGenerator(IdGenerator idGenerator) {
        this.idGenerator = idGenerator;
    }

    public IdGenerator<?> getIdGenerator() {
        return this.idGenerator;
    }

    public ConnectionProvider getConnectionProvider() {
        return this.contextAccess.getContext();
    }

    public Connection getConnection() {
        return getConnectionProvider().get();
    }

    /**
     * Insert a data and return the created object
     * 
     * @param obj
     */
    public <T> T create(T obj) {
        return create(null, null, obj);
    }

    /**
     * Insert a data and return the created object
     * 
     * @param connectionProvider
     * @param obj
     */
    public <T> T create(ConnectionProvider connectionProvider, CommandCustomizer statementAdapter, T obj) {
        return this.insertExecutor.executeCreate(connectionProvider, this, statementAdapter,obj);
    }
    
    /**
     * Insert a data (without returning the created object)
     * 
     * @param obj
     */
    public <T> void insert(T obj) {
        insert(null, null,obj);
    }
    
    public <T> void insert(CommandCustomizer statementAdapter, T obj) {
        insert(null, statementAdapter, obj);
    }

    /**
     * Insert a data (without returning the created object)
     * 
     * @param connectionProvider
     * @param obj
     */
    public <T> void insert(ConnectionProvider connectionProvider, CommandCustomizer statementAdapter, T obj) {
         this.insertExecutor.executeInsert(connectionProvider, this, statementAdapter, obj);
    }

    public <T> T update(T obj) {
        return update(null, obj);
    }

    public <T> T update(CommandCustomizer statementAdapter, T obj) {
        return this.updateExecutor.execute(this, statementAdapter, obj);
    }

    public void delete(Object... key) {
        this.delete((Context) null, key);
    }

    public void delete(ConnectionProvider connectionProvider, Object... key) {
        this.deleteExecutor.execute(connectionProvider, this, key);
    }

    public void deleteObject(Object obj) {
        this.deleteExecutor.executeObject(this, obj);
    }

    protected void delete(String criteriaString, Object... criterias) {
        this.deleteExecutor.executeObject(this, criteriaString, criterias);
    }

    public class CommandCustomizer {
    	
    	
    	private Map<String,String> customFieldValues;
    	private String[] customColumns;
    	
    	public CommandCustomizer() {
    		configure();
    	}
    	
    	protected void configure() {
    		// intended to be overwritten in case required
    	}
        
        protected void defineCustomFieldValue(String fieldName, String customValue) {
        	if (customFieldValues == null) {
        		customFieldValues = new HashMap<>();
        	}
        	customFieldValues.put(fieldName, customValue);
		}

        public String[] customColumns() {
        	if (customColumns == null && customFieldValues != null) {
        		Set<String> fieldNames = customFieldValues.keySet();
        		customColumns = fieldNames.toArray(new String[fieldNames.size()]);
        	}
            return customColumns;
        }

        public void customSetStatementParameter(PreparedStatement preparedStatement, FieldNavigator ii, Object obj) throws Exception {
        	// intended to be overwritten in case required
        }

        public String customValue(String fieldName) {
			if (customFieldValues != null) {
				return customFieldValues.get(fieldName);
			}
        	return null;
        }
        
        public void customReadResultSet(ResultSet rs, FieldNavigator ii, Object object) throws Exception {
        }

    }

    public <T> T load(Class<T> objectClass, Object... key) {
        T obj = this.reflectionHelper.createObject(objectClass);
        load(null, obj, key);
        return obj;
    }

    public <T> void load(T object, Object... key) {
        load(null, object, key);
    }

    protected <T> T load(CommandCustomizer statementAdapter, Class<T> objectClass, Object... key) {
        T obj = this.reflectionHelper.createObject(objectClass);
        load(statementAdapter, obj, key);
        return obj;
    }

    protected <T> void load(CommandCustomizer statementAdapter, T obj, Object... key) {
        this.loadExecutor.load(this, statementAdapter, obj, key);
    }

    public <T> Collection<T> findAll(Class<T> objectClass) {
        return find(objectClass, null, null);
    }

    public <T> Collection<T> find(Class<T> objectClass, String criteriaString, String orderByString, Object... criterias) {
        return findWithPaging(objectClass, criteriaString, orderByString, 0, 0, criterias);
    }

    public <T> Collection<T> findWithPaging(Class<T> objectClass, String criteriaString, String orderByString, int page, int pageSize, Object... criterias) {
        return findWithPaging(null, objectClass, criteriaString, orderByString, page, pageSize, criterias);
    }

    public <T> Collection<T> findWithPaging(ConnectionProvider connectionProvider, Class<T> objectClass, String criteriaString, String orderByString, int page, int pageSize, Object... criterias) {
        return this.findExecutor.findWithPaging(connectionProvider, this, objectClass, criteriaString, orderByString, page, pageSize, criterias);
    }

    protected <T> Collection<T> select(Class<T> objectClass, String selectSQL, Object... parameters) {
        return this.selectExecutor.select(this, objectClass, selectSQL, parameters);
    }

    protected <T> Collection<T> selectWithPaging(Class<T> objectClass, String insertTableSQL, int page, int pageSize, Object... parameters) {
        if (page > 0 && pageSize > 0) {
            int parameterLength = parameters.length;
            List<Object> paramterList = Arrays.asList(parameters);
            Object[] parametersWithPaging = paramterList.toArray(new Object[parameterLength + 2]);
            parametersWithPaging[parameterLength] = (page - 1) * pageSize;
            parametersWithPaging[parameterLength + 1] = pageSize;
            return select(objectClass, insertTableSQL + " OFFSET ? LIMIT ? ", parametersWithPaging);
        } else {
            return select(objectClass, insertTableSQL, parameters);
        }
    }

    protected ContextAccess getContextAccess() {
        return this.contextAccess;
    }

}
