package org.iworkz.habitat.sync;

import java.util.function.Consumer;
import java.util.function.Supplier;

/**
 * Convenient support for 'double-checked locking'-pattern for properties of an Object.
 *
 * @param <R>
 * @param <T>
 */
public class SynchronizedPropertyAccess<R, T> {

    protected final R object;

    protected final Supplier<T> getterInvocation;

    protected final Consumer<T> setterInvocation;

    public SynchronizedPropertyAccess(R object, Supplier<T> getterInvocation, Consumer<T> setterInvocation) {
        this.object = object;
        this.getterInvocation = getterInvocation;
        this.setterInvocation = setterInvocation;
    }

    public T get() {
        return this.getterInvocation.get();
    }

    public boolean isDefined() {
        return this.getterInvocation.get() != null;
    }

    public T getOrCreate(Supplier<T> valueSupplier) {
        T value = get();
        if (value == null) {
            synchronized (this) {
                value = get();
                if (value == null) {
                    value = valueSupplier.get();
                    this.setterInvocation.accept(value);
                }
            }
        }
        return value;
    }

    public T getOrCreate(Consumer<T> valueModifier) {
        T value = get();
        if (value == null) {
            synchronized (this) {
                value = get();
                if (value == null) {
                    value = create();
                    valueModifier.accept(value);
                    this.setterInvocation.accept(value);
                }
            }
        }
        return value;
    }

    public void createAndSet(Supplier<T> valueSupplier) {
        T value = get();
        if (value == null) {
            synchronized (this) {
                value = get();
                if (value == null) {
                    value = valueSupplier.get();
                    this.setterInvocation.accept(value);
                }
            }
        }
    }

    public void create(Consumer<R> valueConsumer) {
        T value = get();
        if (value == null) {
            synchronized (this) {
                value = get();
                if (value == null) {
                    valueConsumer.accept(this.object);
                }
            }
        }
    }

    protected T create() {
        throw new RuntimeException("Creation of value not implemented");
    }

}
