package org.iworkz.habitat.command;

import java.beans.PropertyDescriptor;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.iworkz.common.helper.ReflectionHelper;
import org.iworkz.habitat.command.CommandMetaData.ObjectMetaData;
import org.iworkz.habitat.dao.BlobHelper;
import org.iworkz.habitat.dao.FieldNavigator;
import org.iworkz.habitat.dao.GenericDao.CommandCustomizer;
import org.iworkz.habitat.entity.EntityDefinition;
import org.iworkz.habitat.entity.EntityFieldDefinition;

@Singleton
public class CommandHelper {

    @Inject
    protected ReflectionHelper reflectionHelper;

    public void applyTempFieldValues(Object record, Map<ObjectMetaData, Object> initializedFieldValues) throws Exception {
        if (initializedFieldValues != null) {
            for (ObjectMetaData fn : initializedFieldValues.keySet()) {
                fn.getWriteMethod().invoke(record, initializedFieldValues.get(fn));
            }
        }
    }

    public void setParameter(PreparedStatement preparedStatement, int index, Object parameter) throws Exception {
        if (parameter == null) {
            throw new RuntimeException("Parameter must not be null");
        }
        if (parameter instanceof Integer) {
            preparedStatement.setInt(index, (Integer) parameter);
        } else if (parameter instanceof String) {
            preparedStatement.setString(index, (String) parameter);
        } else if (parameter instanceof Long) {
            preparedStatement.setLong(index, (Long) parameter);
        } else {
            throw new RuntimeException("Unsupported type of parameter '" + parameter.getClass().getSimpleName() + "'");
        }
    }

    public <T> T readFieldsFromResultSet(T obj, ObjectMetaData[] objectMetaData, ResultSet resultSet, FieldNavigator index) throws Exception {
        for (int i = 0; i < objectMetaData.length; i++) {
            Class<?> type = objectMetaData[i].getType();
            Method writeMethod = objectMetaData[i].getWriteMethod();
            Object value = readValueFromResultSet(resultSet, index.next(), type);
            if (type.isAssignableFrom(OutputStream.class)) {
                OutputStream outputStream = (OutputStream) objectMetaData[i].getReadMethod().invoke(obj);
                BlobHelper.writeBlobToOutputStream((Blob) value, outputStream);
            } else {
                writeMethod.invoke(obj, value);
            }
        }
        return obj;
    }

    public Object readValueFromResultSet(ResultSet resultSet, int index, Class<? extends Object> fieldClass) throws Exception {
        if (fieldClass == String.class) {
            return resultSet.getString(index);
        } else if (fieldClass == Integer.class) {
            return resultSet.getInt(index);
        } else if (fieldClass == Date.class) {
            return resultSet.getDate(index);
        } else if (fieldClass == Timestamp.class) {
            return resultSet.getTimestamp(index);
        } else if (fieldClass == BigDecimal.class) {
        	return resultSet.getBigDecimal(index);
        } else if (fieldClass == byte[].class) {
            Blob blob = resultSet.getBlob(index);
            if (blob != null) {
                return blob.getBytes(0L, (int) blob.length());
            } else {
                return null;
            }
        } else if (fieldClass.isAssignableFrom(OutputStream.class)) {
            return resultSet.getBlob(index);
        }
        throw new RuntimeException("Unsupported field type '" + fieldClass.getSimpleName() + "'");
    }

    public void writeUpdateParameter(PreparedStatement statement, FieldNavigator index, Object obj, CommandMetaData commandMetaData) throws Exception {
        for (ObjectMetaData f : commandMetaData.getPrimaryKeyMetaData()) {
            Object value = f.getReadMethod().invoke(obj);
            statement.setObject(index.next(), value);
        }
        for (ObjectMetaData f : commandMetaData.getVersionMetaData()) {
            Object value = f.getReadMethod().invoke(obj);
            statement.setObject(index.next(), value);
        }
    }

    protected void writeRecordToPreparedStatement(PreparedStatement statement, FieldNavigator index, Object obj, CommandMetaData commandMetaData, Map<ObjectMetaData, Object> tempFieldValues, CommandCustomizer statementAdapter,
            boolean isNewRecord)
        throws Exception {
        for (ObjectMetaData objectMetaData : commandMetaData.getObjectMetaData()) {
            EntityFieldDefinition dbField = objectMetaData.getDatabaseField();
            boolean hasCustomValue = statementAdapter != null && statementAdapter.customValue(dbField.getName()) != null;
            if (!hasCustomValue) {
            	
            	Object value;
            	if (tempFieldValues != null && (dbField.isVersionField() || (isNewRecord && dbField.isPrimaryKeyField()))) {
            		value = tempFieldValues.get(objectMetaData);
            	} else {
            		value = objectMetaData.getReadMethod().invoke(obj);
            	}
            	
            	statement.setObject(index.next(), value);
            }
        }
    }

    protected <T> PropertyDescriptor[] createPropertyDescriptorsForClass(EntityDefinition entityDefinition, Class<T> objectClass, CommandCustomizer statementAdapter, ResultSet rs) {
        Field[] objectClassFields = this.reflectionHelper.getAllFields(objectClass);
        List<PropertyDescriptor> fieldList = new ArrayList<>();
        try {
            /* ResultSet available */
            if (rs != null) {
                ResultSetMetaData metaData = rs.getMetaData();
                for (int i = 0; i < metaData.getColumnCount(); i++) {
                    String recordFieldName = metaData.getColumnLabel(i + 1);
                    String tableName = metaData.getTableName(i + 1);
                    String propertyName = entityDefinition.fieldNameForColumn(recordFieldName, tableName);
                    Field recordField = entityDefinition.findField(objectClassFields, propertyName);
                    if (recordField == null) {
                        boolean isAdditionalColumn = false;
                        if (statementAdapter != null) {
                            for (String cn : statementAdapter.customColumns()) {
                                if (recordFieldName.equalsIgnoreCase(cn)) {
                                    isAdditionalColumn = true;
                                    break;
                                }
                            }
                        }
                        if (isAdditionalColumn == false) {
                            throw new RuntimeException("Property not found in bean '" + objectClass.getName() + "'. Expected property for field '" + recordFieldName + "' is '" + propertyName + "'");
                        }
                    }
                    if (recordField != null) {
                        PropertyDescriptor propertyDescriptor = new PropertyDescriptor(recordField.getName(), objectClass);
                        fieldList.add(propertyDescriptor);
                    }
                }
                /* No ResultSet available */
            } else {
                for (EntityFieldDefinition dbField : entityDefinition.getFields()) {
                    String recordFieldName = dbField.getName();
                    String tableName = entityDefinition.getName();
                    String propertyName = entityDefinition.fieldNameForColumn(recordFieldName, tableName);
                    Field recordField = entityDefinition.findField(objectClassFields, propertyName);
                    if (recordField != null) {
                        PropertyDescriptor propertyDescriptor = new PropertyDescriptor(propertyName, objectClass);
                        fieldList.add(propertyDescriptor);

                    }
                }
            }

            PropertyDescriptor[] resultSetFields = new PropertyDescriptor[fieldList.size()];
            fieldList.toArray(resultSetFields);

            return resultSetFields;

        } catch (Exception ex) {
            throw new RuntimeException("Can not create descriptor for class " + objectClass.getCanonicalName(), ex);
        }
    }

}
