package org.iworkz.habitat.command;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.Collection;

import javax.inject.Singleton;

import org.iworkz.habitat.dao.ConnectionProvider;
import org.iworkz.habitat.dao.FieldNavigator;
import org.iworkz.habitat.dao.GenericDao;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class FindExecutor extends AbstractExecutor {

    private static final Logger logger = LoggerFactory.getLogger(FindExecutor.class);

    public <T> Collection<T> findWithPaging(ConnectionProvider context, GenericDao dao, Class<T> objectClass, String criteriaString, String orderByString, int page, int pageSize, Object... criterias) {
        String name = "find" + objectClass.getSimpleName() + criteriaString + orderByString;
        if (page != 0) {
            name += "_paging";
        }

        CommandMetaData commandMetaData = dao.metaDataAccess.getValue(name);
        if (commandMetaData == null) {
            commandMetaData = dao.metaDataAccess.getOrCreateValue(name, (value) -> {
                value.setCommand(this.commandBuilder.buildFindCommand(dao.getEntityDefinition(), objectClass, null, criteriaString, orderByString, page != 0));
            });
        }

        String sqlCommand = commandMetaData.getCommand();
        FieldNavigator ii = new FieldNavigator();
        PreparedStatement preparedStatement = null;
        try {
            preparedStatement = getConnection(context, dao).prepareStatement(sqlCommand);
            if (criteriaString != null) {
                for (int i = 0; i < criterias.length; i++) {
                    this.commandHelper.setParameter(preparedStatement, i + 1, criterias[i]);
                }
            }
            if (page != 0) {
                int parameterLength = criterias.length;
                this.commandHelper.setParameter(preparedStatement, parameterLength + 1, (page - 1) * pageSize);
                this.commandHelper.setParameter(preparedStatement, parameterLength + 2, pageSize);
            }

            ResultSet rs = preparedStatement.executeQuery();

            if (!commandMetaData.objectMetaDataProperty.isDefined()) {
                commandMetaData.objectMetaDataProperty.create((object) -> {
                    object.setPropertyDescriptors(this.commandHelper.createPropertyDescriptorsForClass(dao.getEntityDefinition(), objectClass, null, rs), null);
                });
            }

            ArrayList<T> resultList = new ArrayList<T>();
            while (rs.next()) {
                ii.reset();
                T obj = this.reflectionHelper.createObject(objectClass);
                this.commandHelper.readFieldsFromResultSet(obj, commandMetaData.getObjectMetaData(), rs, ii);
                resultList.add(obj);
            }
            return resultList;
        } catch (Exception e) {
            logger.error(sqlCommand);
            throw new RuntimeException("Can not read record from table " + dao.getEntityDefinition().getName(), e);
        } finally {
            if (preparedStatement != null) {
                try {
                    preparedStatement.close();
                } catch (Exception ex) {
                    logger.error("Prepared statement can not be closed", ex);
                }
            }
        }
    }

}
