package org.iworkz.habitat.command;

import java.sql.PreparedStatement;
import java.util.Map;

import javax.inject.Singleton;

import org.iworkz.habitat.command.CommandMetaData.ObjectMetaData;
import org.iworkz.habitat.dao.FieldNavigator;
import org.iworkz.habitat.dao.GenericDao;
import org.iworkz.habitat.dao.GenericDao.CommandCustomizer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class UpdateExecutor extends AbstractExecutor {

    private static final Logger logger = LoggerFactory.getLogger(UpdateExecutor.class);

    public <T> T execute(GenericDao dao, T obj) {
        return execute(dao, null, obj);
    }

    public <T> T execute(GenericDao dao, CommandCustomizer statementAdapter, T obj) {

        Map<ObjectMetaData, Object> tempFieldValues;

        String name = "_updateObject" + obj.getClass().getName() + statementAdapter;

        CommandMetaData commandMetaData = dao.metaDataAccess.getValue(name);
        if (commandMetaData == null) {
            commandMetaData = dao.metaDataAccess.getOrCreateValue(name, (value) -> {
                value.setCommand(this.commandBuilder.buildUpdateCommand(dao.getEntityDefinition(), obj.getClass(), statementAdapter));
            });
        }

        String sqlCommand = commandMetaData.getCommand();

        PreparedStatement preparedStatement = null;
        RuntimeException conflictException = null;
        try {

            if (!commandMetaData.objectMetaDataProperty.isDefined()) {
                commandMetaData.objectMetaDataProperty.create((object) -> {
                    object.setPropertyDescriptors(this.commandHelper.createPropertyDescriptorsForClass(dao.getEntityDefinition(), obj.getClass(), statementAdapter, null),
                            statementAdapter);
                });
            }

            tempFieldValues = this.recordHelper.createUpdatedVersionInfo(obj, commandMetaData);

            preparedStatement = dao.getConnection().prepareStatement(sqlCommand);
            FieldNavigator ii = new FieldNavigator(commandMetaData.getTotalFieldCount());
            this.commandHelper.writeUpdateParameter(preparedStatement, ii, obj, commandMetaData);
            ii.reset();
            this.commandHelper.writeRecordToPreparedStatement(preparedStatement, ii, obj, commandMetaData, tempFieldValues, statementAdapter,false);
            if (statementAdapter != null) {
                statementAdapter.customSetStatementParameter(preparedStatement, ii, obj);
            }
            int res = preparedStatement.executeUpdate();
            if (res == 0) {
                conflictException = new RuntimeException("Record could not be updated (maybe somebody else updated or deleted it meanwhile)");
                throw conflictException;
            }
            /* return the persisted object (containing primary key and version values) */
            T persistentObject = cloneBean(obj);
            this.commandHelper.applyTempFieldValues(persistentObject, tempFieldValues);
            return persistentObject;
        } catch (Exception e) {
            logger.error(sqlCommand);
            if (e == conflictException) {
                throw this.exceptionFactory
                        .createConflictException("Can not update record in table " + dao.getEntityDefinition().getName() + " (maybe somebody else has updated or deleted it meanwhile)");
            } else {
                throw new RuntimeException("Can not update record in table " + dao.getEntityDefinition().getName(), e);
            }
        } finally {
            if (preparedStatement != null) {
                try {
                    preparedStatement.close();
                } catch (Exception ex) {
                    logger.error("Prepared statement can not be closed", ex);
                }
            }
        }

    }

}
