package org.iworkz.habitat.command;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.util.Collection;
import java.util.Map;

import javax.inject.Singleton;

import org.iworkz.habitat.command.CommandMetaData.ObjectMetaData;
import org.iworkz.habitat.dao.ConnectionProvider;
import org.iworkz.habitat.dao.FieldNavigator;
import org.iworkz.habitat.dao.GenericDao;
import org.iworkz.habitat.dao.GenericDao.CommandCustomizer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class InsertExecutor extends AbstractExecutor {
	
	private final int MAX_BATCH_SIZE_DEFAULT = Integer.parseInt(System.getProperty("org.habitat.batch.maxsize", "4000"));

    private static final Logger logger = LoggerFactory.getLogger(InsertExecutor.class);
    
    public <T> void executeInsert(ConnectionProvider task, GenericDao dao, CommandCustomizer statementAdapter, T obj) {

        Class<T> objectClass = (Class<T>) obj.getClass();

        String name = "_insertObject" + obj.getClass().getName();

        CommandMetaData commandMetaData = dao.metaDataAccess.getValue(name);
        if (commandMetaData == null) {
            commandMetaData = dao.metaDataAccess.getOrCreateValue(name, (value) -> {
                value.setCommand(this.commandBuilder.buildInsertCommand(dao.getEntityDefinition(), objectClass,statementAdapter));
            });
        }

        String sqlCommand = commandMetaData.getCommand();

        FieldNavigator ii = new FieldNavigator();
        PreparedStatement preparedStatement = null;
        try {

            if (!commandMetaData.objectMetaDataProperty.isDefined()) {
                commandMetaData.objectMetaDataProperty.create((object) -> {
                    object.setPropertyDescriptors(this.commandHelper.createPropertyDescriptorsForClass(dao.getEntityDefinition(), objectClass, statementAdapter, null), statementAdapter);
                });
            }

            Map<ObjectMetaData, Object> tempFieldValues;
            //Object record = setToRecord(obj);
            tempFieldValues = this.recordHelper.initializeNewRecord(obj, commandMetaData, dao.getIdGenerator());

            preparedStatement = getConnection(task, dao).prepareStatement(sqlCommand);

            this.commandHelper.writeRecordToPreparedStatement(preparedStatement, ii, obj, commandMetaData, tempFieldValues, statementAdapter, true);
            if (statementAdapter != null) {
                statementAdapter.customSetStatementParameter(preparedStatement, ii, obj);
            }
            //setObject(preparedStatement,ii,obj);
            preparedStatement.executeUpdate();

        } catch (Exception e) {
            logger.error(sqlCommand);
            throw new RuntimeException("Can not insert record in table " + dao.getEntityDefinition().getName(), e);
        } finally {
            if (preparedStatement != null) {
                try {
                    preparedStatement.close();
                } catch (Exception ex) {
                    logger.error("Prepared statement can not be closed", ex);
                }
            }
        }

    }
    
    public <T> void executeInsertAll(ConnectionProvider task, GenericDao dao, CommandCustomizer statementAdapter, Collection<T> objects, Integer maxBatchSize) {

    	if (maxBatchSize == null) {
    		maxBatchSize = MAX_BATCH_SIZE_DEFAULT;
    	}
    	
    	if (objects == null || objects.isEmpty()) {
    		return;
    	}
    	
    	T firstObj = objects.iterator().next();
    	
        Class<T> objectClass = (Class<T>) firstObj.getClass();

        String name = "_insertAllObjects" + firstObj.getClass().getName();

        CommandMetaData commandMetaData = dao.metaDataAccess.getValue(name);
        if (commandMetaData == null) {
            commandMetaData = dao.metaDataAccess.getOrCreateValue(name, (value) -> {
                value.setCommand(this.commandBuilder.buildInsertCommand(dao.getEntityDefinition(), objectClass,statementAdapter));
            });
        }

        String sqlCommand = commandMetaData.getCommand();
        PreparedStatement preparedStatement = null;
        Connection connection = getConnection(task, dao);
        boolean needToRestoreAutoCommitFlag = setAutoCommitFalse(connection);
        try {

            if (!commandMetaData.objectMetaDataProperty.isDefined()) {
                commandMetaData.objectMetaDataProperty.create((object) -> {
                    object.setPropertyDescriptors(this.commandHelper.createPropertyDescriptorsForClass(dao.getEntityDefinition(), objectClass, statementAdapter, null), statementAdapter);
                });
            }
           
            int insertCounter = 0;
            preparedStatement = connection.prepareStatement(sqlCommand);
            for (T obj : objects) {
            	
            	/* limit batch size */
            	insertCounter++;
            	if (maxBatchSize != -1 && insertCounter > maxBatchSize) {
                    preparedStatement.executeBatch();
                    insertCounter = 0;
            	}
            	
            	FieldNavigator ii = new FieldNavigator();
            	
            	Map<ObjectMetaData, Object> tempFieldValues;
            	tempFieldValues = this.recordHelper.initializeNewRecord(obj, commandMetaData, dao.getIdGenerator());
            	
            	this.commandHelper.writeRecordToPreparedStatement(preparedStatement, ii, obj, commandMetaData, tempFieldValues, statementAdapter, true);
            	if (statementAdapter != null) {
            		statementAdapter.customSetStatementParameter(preparedStatement, ii, obj);
            	}
            	preparedStatement.addBatch();
            }
            
            preparedStatement.executeBatch();
            connection.commit();

        } catch (Exception e) {
            logger.error(sqlCommand);
            throw new RuntimeException("Can not insert all records in table " + dao.getEntityDefinition().getName(), e);
        } finally {
            if (preparedStatement != null) {
                try {
                    preparedStatement.close();
     
                } catch (Exception ex) {
                    logger.error("Prepared statement can not be closed", ex);
                }
            }
        	if (connection != null && needToRestoreAutoCommitFlag) {
        		try {
        			connection.setAutoCommit(true);
        		} catch (Exception ex) {
        			logger.error("Auto commit flag cannot be restored", ex);
        		}
        	}
        }

    }
    
    protected boolean setAutoCommitFalse(Connection connection) {
    	try {
    		boolean prevAutoCommit = connection.getAutoCommit();
    		connection.setAutoCommit(false);
    		return prevAutoCommit;
    	} catch (Exception ex) {
    		throw new RuntimeException("Can not set auto commit to false", ex);
    	}
	}
    
    protected void executeBatchAndCommit() {
    	
    }

    public <T> T executeCreate(ConnectionProvider task, GenericDao dao, CommandCustomizer statementAdapter, T obj) {

        Class<T> objectClass = (Class<T>) obj.getClass();

        String name = "_insertObject" + obj.getClass().getName();

        CommandMetaData commandMetaData = dao.metaDataAccess.getValue(name);
        if (commandMetaData == null) {
            commandMetaData = dao.metaDataAccess.getOrCreateValue(name, (value) -> {
                value.setCommand(this.commandBuilder.buildInsertCommand(dao.getEntityDefinition(), objectClass,statementAdapter));
            });
        }

        String sqlCommand = commandMetaData.getCommand();

        FieldNavigator ii = new FieldNavigator();
        PreparedStatement preparedStatement = null;
        try {

            if (!commandMetaData.objectMetaDataProperty.isDefined()) {
                commandMetaData.objectMetaDataProperty.create((object) -> {
                    object.setPropertyDescriptors(this.commandHelper.createPropertyDescriptorsForClass(dao.getEntityDefinition(), objectClass, statementAdapter, null), statementAdapter);
                });
            }

            Map<ObjectMetaData, Object> tempFieldValues;
            //Object record = setToRecord(obj);
            tempFieldValues = this.recordHelper.initializeNewRecord(obj, commandMetaData, dao.getIdGenerator());

            preparedStatement = getConnection(task, dao).prepareStatement(sqlCommand);

            this.commandHelper.writeRecordToPreparedStatement(preparedStatement, ii, obj, commandMetaData, tempFieldValues, statementAdapter, true);
            if (statementAdapter != null) {
                statementAdapter.customSetStatementParameter(preparedStatement, ii, obj);
            }
            //setObject(preparedStatement,ii,obj);
            preparedStatement.executeUpdate();

            /* return the persisted object (containing primary key and version values) */

            T persistentObject = cloneBean(obj);
            this.commandHelper.applyTempFieldValues(persistentObject, tempFieldValues);
            return persistentObject;

        } catch (Exception e) {
            logger.error(sqlCommand);
            throw new RuntimeException("Can not insert record in table " + dao.getEntityDefinition().getName(), e);
        } finally {
            if (preparedStatement != null) {
                try {
                    preparedStatement.close();
                } catch (Exception ex) {
                    logger.error("Prepared statement can not be closed", ex);
                }
            }
        }

    }

}
