package org.iworkz.habitat.entity;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

import org.iworkz.common.helper.ReflectionHelper;
import org.iworkz.habitat.dao.GenericDao.CommandCustomizer;
import org.iworkz.habitat.dialect.DatabaseDialect;
import org.iworkz.habitat.helper.StringHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EntityDefinition {
	
    private static final Logger logger = LoggerFactory.getLogger(EntityDefinition.class);

    final private String name;

    private List<EntityFieldDefinition> fields = new ArrayList<EntityFieldDefinition>();

    private List<EntityFieldDefinition> primaryKeyFields = new ArrayList<EntityFieldDefinition>();

    private List<EntityFieldDefinition> versionFields = new ArrayList<EntityFieldDefinition>();

    protected StringHelper stringHelper = new StringHelper();
    
    private DatabaseDialect databaseDialect;

    protected ReflectionHelper reflectionHelper;// = new ReflectionHelper();

    public EntityDefinition(String name, DatabaseDialect databaseDialect, ReflectionHelper reflectionHelper) {
        this.name = name;
        this.databaseDialect = databaseDialect;
        this.reflectionHelper = reflectionHelper;
    }

    public String getName() {
        return this.name;
    }

    public List<EntityFieldDefinition> getFields() {
        return this.fields;
    }

    public List<EntityFieldDefinition> getPrimaryKey() {
        return this.primaryKeyFields;
    }

    public List<EntityFieldDefinition> getVersion() {
        return this.versionFields;
    }

    public <T> void appendFieldAssigmentSql(StringBuilder sql, Class<T> objectClass, boolean supportOnlyEntityFields) {
        if (objectClass != null) {
            Field[] objectFields = this.reflectionHelper.getAllFields(objectClass);
            List<EntityFieldDefinition> fields = getFields();
            int count = 0;
            for (int i = 0; i < fields.size(); i++) {
                EntityFieldDefinition f = fields.get(i);
                for (int j = 0; j < objectFields.length; j++) {
                    String columnName = columnNameForField(objectFields[j].getName(), supportOnlyEntityFields);
                    if (columnName != null && columnName.equals(f.getName())) {
                        if (count > 0) {
                            sql.append(", ");
                        }
                        sql.append(f.getName() + " = ?");
                        count++;
                        break;
                    }
                }
            }
        } else {
            for (int i = 0; i < this.fields.size(); i++) {
                EntityFieldDefinition f = this.fields.get(i);
                sql.append(f.getName() + " = ?");
                if (i != this.fields.size() - 1) {
                    sql.append(", ");
                }
            }
        }
    }

	public void appendAdditionalFieldAssigmentSql(StringBuilder sql, CommandCustomizer statementAdapter) {

		String[] columnNames = statementAdapter.customColumns();
		if (columnNames != null) {
			for (String cn : columnNames) {
				if (statementAdapter.customValue(cn) == null) {
					sql.append(", ");
					sql.append(cn + " = ?");
				}
			}
		}
	}

    public <T> void appendFieldSql(StringBuilder sql, Class<T> objectClass, boolean supportOnlyEntityFields) {
        if (objectClass != null) {
            Field[] objectFields = this.reflectionHelper.getAllFields(objectClass);
            List<EntityFieldDefinition> fields = getFields();
            int count = 0;
            for (int i = 0; i < fields.size(); i++) {
                EntityFieldDefinition f = fields.get(i);
                for (int j = 0; j < objectFields.length; j++) {
                    String columnName = columnNameForField(objectFields[j].getName(), supportOnlyEntityFields);
                    if (columnName != null && columnName.equals(f.getName())) {
                        if (count > 0) {
                            sql.append(", ");
                        }
                        sql.append(f.getName());
                        count++;
                        break;
                    }
                }
            }
        } else {
            List<EntityFieldDefinition> fields = getFields();
            for (int i = 0; i < fields.size(); i++) {
                EntityFieldDefinition f = fields.get(i);
                sql.append(f.getName());
                if (i != fields.size() - 1) {
                    sql.append(", ");
                }
            }
        }
    }

    public void appendAdditionalColumns(StringBuilder sql, String[] columnNames) {
        if (columnNames != null) {
            for (String cn : columnNames) {
                sql.append(", ");
                sql.append(cn);
            }

        }
    }

	public void appendFieldDefinitionsSql(StringBuilder sql) {
		databaseDialect.appendFieldDefinitionsSql(sql,this);
	}

    public <T> void appendFieldQuestionmarksSql(StringBuilder sql, Class<T> objectClass, CommandCustomizer statementAdapter) {
        if (objectClass != null) {

            Field[] objectFields = this.reflectionHelper.getAllFields(objectClass);
            List<EntityFieldDefinition> fields = getFields();
            int count = 0;
            for (int i = 0; i < fields.size(); i++) {
                EntityFieldDefinition f = fields.get(i);
                for (int j = 0; j < objectFields.length; j++) {
                    String columnName = columnNameForField(objectFields[j].getName(), false);
                    if (columnName != null && columnName.equals(f.getName())) {
                        if (count > 0) {
                            sql.append(", ");
                        }
                        if (statementAdapter != null) {
                            String customValue = statementAdapter.customValue(columnName);
                            if (customValue != null) {
                            	sql.append(customValue);
                            } else {
                            	sql.append("?");
                            }
                        } else {
                        	sql.append("?");
                        }
                        count++;
                        break;
                    }
                }
            }
        } else {
            for (int i = 0; i < this.fields.size(); i++) {
                sql.append("?");
                if (i != this.fields.size() - 1) {
                    sql.append(", ");
                }
            }
        }
    }

    public void appendPrimaryKeySqL(StringBuilder sql) {
    	databaseDialect.appendPrimaryKeySqL(sql, this);
    }

    public void addField(EntityFieldDefinition field) {
        this.fields.add(field);
    }

    public void setPrimaryKey(String key) {
        if (key == null) {
            throw new IllegalArgumentException("Key musst not be null");
        }
        boolean found = false;
        for (EntityFieldDefinition f : this.fields) {
            if (key.equals(f.getName())) {
                f.setPrimaryKeyField(true);
                this.primaryKeyFields.add(f);
                found = true;
                break;
            }
        }
        if (!found) {
            throw new RuntimeException("No field for key '" + key + "'found");
        }
    }

    public void setVersion(String version) {
        if (version == null) {
            throw new IllegalArgumentException("Version musst not be null");
        }
        boolean found = false;
        for (EntityFieldDefinition f : this.fields) {
            if (version.equals(f.getName())) {
                f.setVersionField(true);
                this.versionFields.add(f);
                found = true;
                break;
            }
        }
        if (!found) {
            throw new RuntimeException("No field for key '" + version + "'found");
        }
    }

    public void appendPrimaryKeyCriteriaSql(StringBuilder sql) {
        for (int i = 0; i < this.primaryKeyFields.size(); i++) {
            EntityFieldDefinition f = this.primaryKeyFields.get(i);
            sql.append(f.getName() + " = ?");
            if (i != this.primaryKeyFields.size() - 1) {
                sql.append(", ");
            }
        }
    }

    public void appendVersionCriteriaSql(StringBuilder sql) {
        if (this.versionFields.size() > 0) {
            sql.append(" AND ");
        }
        for (int i = 0; i < this.versionFields.size(); i++) {
            EntityFieldDefinition f = this.versionFields.get(i);
            sql.append(f.getName() + " = ?");
            if (i != this.versionFields.size() - 1) {
                sql.append(", ");
            }
        }
    }

    protected String columnNameForField(String fieldName, boolean supportOnlyEntityFields) {
        String columnName = columnNameForRecordField(fieldName);

        /* If field name not found in table definition use naming strategy */
        if (columnName == null && supportOnlyEntityFields == false) {
            columnName = this.stringHelper.camelCaseToUnderLineSeperated(fieldName);
            if (columnName != null) {
                columnName = columnName.toUpperCase();
            }
        }
        return columnName;
    }

    public String fieldNameForColumn(String columnName, String tableName) {
        String fieldName = null;
        /* When table name available, try name mapping from table definition */
        if (this.stringHelper.hasValue(tableName)) {
        	String originalTableName = tableName;
        	int lastDotIndex = tableName.lastIndexOf('.');
        	if (lastDotIndex >= 0) {
        		tableName = tableName.substring(lastDotIndex+1);
        	}
        	if (tableName.equalsIgnoreCase(getName())) {
        		fieldName = recordFieldNameForColumn(columnName);
        	} else {
        		logger.warn("Table name '"+originalTableName+"' of result set does not match the table name in the entity definition '"+getName()+"'");
        	}
        } else {
        	fieldName = recordFieldNameForColumn(columnName);
        }
        /* If field name not found in table definition use naming strategy */
        if (fieldName == null) {
            fieldName = this.stringHelper.underLineSeperatedToCamelCase(columnName);
        }
        return fieldName;
    }
    
    protected boolean containsField(String fieldName) {
    	if (fieldName != null) {
    		for (EntityFieldDefinition dbField : getFields()) {
    			if (fieldName.equals(dbField.getName())) {
    				return true;
    			}
    		}
    	}
        return false;
    }

    protected String columnNameForRecordField(String fieldName) {
        for (EntityFieldDefinition dbField : getFields()) {
            if (fieldName.equalsIgnoreCase(dbField.objectFieldName)) {
                return dbField.getName();
            }
        }
        return null;
    }

    protected EntityFieldDefinition databaseFieldForColumn(String columnName) {
        for (EntityFieldDefinition dbField : getFields()) {
            if (columnName.equalsIgnoreCase(dbField.getName())) {
                return dbField;
            }
        }
        return null;
    }

    protected String recordFieldNameForColumn(String columnName) {
        for (EntityFieldDefinition dbField : getFields()) {
            if (columnName.equalsIgnoreCase(dbField.getName())) {
                return dbField.objectFieldName;
            }
        }
        return null;
    }

    public <T> Field findField(Field[] fields, String fieldName) {
        for (Field f : fields) {
            if (f.getName().equals(fieldName)) {
                return f;
            }
        }
        return null;
    }

    public EntityFieldDefinition getDatabaseFieldForObjectFieldName(String fieldName) {
        for (EntityFieldDefinition databaseField : getFields()) {
            String columnName = columnNameForRecordField(fieldName);
            if (databaseField.getName().equalsIgnoreCase(columnName)) {
                return databaseField;
            }
        }
        throw new RuntimeException("Database field not found for field with name '" + fieldName + "' for entity '" + this.name + "'");
    }

}
