package org.iworkz.habitat.dao;

import java.beans.PropertyDescriptor;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.sql.Blob;
import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.apache.commons.beanutils.BeanUtils;
import org.iworkz.habitat.command.CommandBuilder;
import org.iworkz.habitat.command.CommandMetaData;
import org.iworkz.habitat.command.CommandMetaData.ObjectMetaData;
import org.iworkz.habitat.entity.EntityFieldDefinition;
import org.iworkz.habitat.helper.RecordHelper;
import org.iworkz.habitat.helper.ReflectionHelper;
import org.iworkz.habitat.helper.StringHelper;

@Singleton
public abstract class GenericDao extends EntityDao {


	protected final Map<String,CommandMetaData> commandMetaDataMap = new HashMap<String,CommandMetaData>();
	
	protected IdGenerator<?> idGenerator = createIdGenerator();
	
	protected StringHelper stringHelper = new StringHelper();
	
	@Inject
	protected ReflectionHelper reflectionHelper;
	
	@Inject
	RecordHelper recordHelper;
	
	@Inject
	CommandBuilder commandBuilder;
	
	@Inject
	protected ContextAccess contextAccess;
	
	@Inject
	protected ExceptionFactory exceptionFactory;
	

	/**
	 * Constructor.
	 * 
	 * Reads the entity definition.
	 */
	public GenericDao() {
	}
	
	protected Connection getConnection() {
		return contextAccess.getContext().getConnection();
	}
	
	public abstract IdGenerator createIdGenerator();
	

	
	protected synchronized void putCommandMetaData(String name, CommandMetaData commandMetaData) {
		commandMetaDataMap.put(name, commandMetaData);
	}
	
	protected synchronized CommandMetaData getCommandMetaData(String name) {
		return commandMetaDataMap.get(name);
	}
	
	protected void setParameter(PreparedStatement preparedStatement, int index, Object parameter) throws Exception {
		if (parameter == null) {
			throw new RuntimeException("Parameter must not be null");
		}
		if (parameter instanceof Integer) {
			preparedStatement.setInt(index,(Integer)parameter);
		} else if (parameter instanceof String) {
			preparedStatement.setString(index,(String)parameter);
		} else if (parameter instanceof Long) {
			preparedStatement.setLong(index,(Long)parameter);
		} else {
			throw new RuntimeException("Unsupported type of parameter '"+parameter.getClass().getSimpleName()+"'");
		}
	}
	
	protected void writeRecordToPreparedStatement(PreparedStatement statement, FieldNavigator index, Object obj, CommandMetaData commandMetaData, Map<ObjectMetaData,Object> tempFieldValues, boolean isNewRecord) throws Exception {
		for (ObjectMetaData objectMetaData : commandMetaData.getObjectMetaData()) {
			EntityFieldDefinition dbField = objectMetaData.getDatabaseField();
			Object value;
			if (tempFieldValues != null && (dbField.isVersionField() || (isNewRecord && dbField.isPrimaryKeyField()))) {
				value = tempFieldValues.get(objectMetaData);
			} else {
				value = objectMetaData.getReadMethod().invoke(obj);
			}

			statement.setObject(index.next(),value);
		}
	}
	
	protected Object readValueFromResultSet(ResultSet resultSet, int index, Class<? extends Object> fieldClass) throws Exception {
		if (fieldClass == String.class) {
			return resultSet.getString(index);
		} else if (fieldClass == Integer.class) {
			return resultSet.getInt(index);
		} else if (fieldClass == Date.class) {
			return resultSet.getDate(index);
		} else if (fieldClass == Timestamp.class) {
            return resultSet.getTimestamp(index);
		} else if (fieldClass == byte[].class) {
			Blob blob = resultSet.getBlob(index);
			if (blob != null) {
				return blob.getBytes(0L, (int)blob.length());
			} else {
				return null;
			}
		} else if (fieldClass.isAssignableFrom(OutputStream.class)) {
			return resultSet.getBlob(index);
		}
		throw new RuntimeException("Unsupported field type '"+fieldClass.getSimpleName()+"'");
	}
	
	protected<T> T readFieldsFromResultSet(T obj, ObjectMetaData[] objectMetaData, ResultSet resultSet, FieldNavigator index) throws Exception {
		for (int i=0;i<objectMetaData.length;i++) {
			Class<?> type = objectMetaData[i].getType();
			Method writeMethod = objectMetaData[i].getWriteMethod();
			Object value = readValueFromResultSet(resultSet,index.next(),type);
			if (type.isAssignableFrom(OutputStream.class)) {
				OutputStream outputStream = (OutputStream)objectMetaData[i].getReadMethod().invoke(obj);
				BlobHelper.writeBlobToOutputStream((Blob)value,outputStream);
			} else {
				writeMethod.invoke(obj,value);	
			}
		}
		return obj;
	}

	protected void writeUpdateParameter(PreparedStatement statement, FieldNavigator index, Object obj, CommandMetaData commandMetaData) throws Exception {
		for (ObjectMetaData f : commandMetaData.getPrimaryKeyMetaData()) {
			Object value = f.getReadMethod().invoke(obj);
			statement.setObject(index.next(),value);
		}
		for (ObjectMetaData f : commandMetaData.getVersionMetaData()) {
			Object value = f.getReadMethod().invoke(obj);
			statement.setObject(index.next(),value);
		}
	}

	public<T> T insert(T obj) {
		
		Class<T> objectClass = (Class<T>) obj.getClass();
		
		Connection conn = getConnection();
		
		String name = "_insertObject"+obj.getClass().getName();
		CommandMetaData commandMetaData = getCommandMetaData(name);
		if (commandMetaData == null) {
			commandMetaData = new CommandMetaData(getEntityDefinition());
			commandMetaData.setCommand(commandBuilder.buildInsertCommand(getEntityDefinition(),objectClass));
			putCommandMetaData(name,commandMetaData);
		}
		String sqlCommand = commandMetaData.getCommand();

		FieldNavigator ii = new FieldNavigator();
		PreparedStatement preparedStatement = null;
		try {
			
			if (commandMetaData.getObjectMetaData() == null) {	
				commandMetaData.setPropertyDescriptors(createPropertyDescriptorsForClass(objectClass,null,null),null);
			}

			Map<ObjectMetaData,Object> tempFieldValues;
			//Object record = setToRecord(obj);
			tempFieldValues = recordHelper.initializeNewRecord(obj,commandMetaData,idGenerator);
			
			preparedStatement = conn.prepareStatement(sqlCommand);

			writeRecordToPreparedStatement(preparedStatement,ii,obj,commandMetaData,tempFieldValues,true);
			//setObject(preparedStatement,ii,obj);
			preparedStatement.executeUpdate();
			
			/* return the persisted object (containing primary key and version values) */
			
			T persistentObject = cloneBean(obj);
			applyTempFieldValues(persistentObject,tempFieldValues);
			return persistentObject;
 
		} catch (Exception e) {
			System.out.println(sqlCommand);
			throw new RuntimeException("Can not insert record in table "+getEntityDefinition().getName(),e);
		} finally {
			if (preparedStatement != null) {
				try {
					preparedStatement.close();
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		}
	}
	
	protected <T> T cloneBean(T obj) {
		try {
			@SuppressWarnings("unchecked")
			T newBean = (T)reflectionHelper.createObject(obj.getClass());
			BeanUtils.copyProperties(newBean, obj);
			return newBean;
		} catch (Exception ex) {
			throw new RuntimeException("Can not clone object",ex);
		}
	}
	
	protected void applyTempFieldValues(Object record, Map<ObjectMetaData,Object> initializedFieldValues) throws Exception {
		if (initializedFieldValues != null) {
			for (ObjectMetaData fn : initializedFieldValues.keySet()) {
				fn.getWriteMethod().invoke(record,initializedFieldValues.get(fn));	
			}
		}
	}
		
	public<T> T update(T obj) {
		return update(null,obj);	
	}

	public<T> T update(CommandCustomizer statementAdapter, T obj) {
		
		Connection conn = getConnection();
		
		Map<ObjectMetaData,Object> tempFieldValues;
		
		String name = "_updateObject"+obj.getClass().getName()+statementAdapter;
		CommandMetaData commandMetaData = getCommandMetaData(name);
		if (commandMetaData == null) {
			commandMetaData = new CommandMetaData(getEntityDefinition());
			commandMetaData.setCommand(commandBuilder.buildUpdateCommand(getEntityDefinition(),obj.getClass(),statementAdapter));
			putCommandMetaData(name,commandMetaData);
		}
		String sqlCommand = commandMetaData.getCommand();

		PreparedStatement preparedStatement = null;
		RuntimeException conflictException = null;
		try {
			
			if (commandMetaData.getObjectMetaData() == null) {	
				commandMetaData.setPropertyDescriptors(createPropertyDescriptorsForClass(obj.getClass(),statementAdapter,null),statementAdapter);
			}
			
			tempFieldValues = recordHelper.createUpdatedVersionInfo(obj,commandMetaData);
			
			preparedStatement = conn.prepareStatement(sqlCommand);
			FieldNavigator ii = new FieldNavigator(commandMetaData.getTotalFieldCount());
			writeUpdateParameter(preparedStatement,ii,obj,commandMetaData);
			ii.reset();
			writeRecordToPreparedStatement(preparedStatement,ii,obj,commandMetaData,tempFieldValues,false);
			if (statementAdapter != null) {
				statementAdapter.customSetStatementParameter(preparedStatement,ii,obj);
			}
			int res = preparedStatement.executeUpdate();
			if (res == 0) {
				conflictException = new RuntimeException("Record could not be updated (maybe somebody else updated or deleted it meanwhile)");
				throw conflictException;
			}
			/* return the persisted object (containing primary key and version values) */
			T persistentObject = cloneBean(obj);
			applyTempFieldValues(persistentObject,tempFieldValues);
			return persistentObject;
		} catch (Exception e) {
			System.out.println(sqlCommand);
			if (e == conflictException) {
				throw exceptionFactory.createConflictException("Can not update record in table "+getEntityDefinition().getName()+" (maybe somebody else has updated or deleted it meanwhile)");
			} else {
				throw new RuntimeException("Can not update record in table "+getEntityDefinition().getName(),e);
			}
		} finally {
			if (preparedStatement != null) {
				try {
					preparedStatement.close();
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		}
	}
	
	   public void delete(Object... key) {
	        
	        Connection conn = getConnection();
	        
	        String name = "_deleteObject";
	        CommandMetaData commandMetaData = getCommandMetaData(name);
	        if (commandMetaData == null) {
	            commandMetaData = new CommandMetaData(getEntityDefinition());
	            commandMetaData.setCommand(commandBuilder.buildDeleteCommand(getEntityDefinition()));
	            putCommandMetaData(name,commandMetaData);
	        }
	        String sqlCommand = commandMetaData.getCommand();
	        
	        FieldNavigator ii = new FieldNavigator();
	        PreparedStatement preparedStatement = null;
	        try {
	            
	            preparedStatement = conn.prepareStatement(sqlCommand);
	            for (int i=0;i<key.length;i++) {
	                setParameter(preparedStatement,i+1,key[i]);
	            }

	            int res = preparedStatement.executeUpdate();
	            if (res == 0) {
	            	throw exceptionFactory.createConflictException("Record could not be deleted (maybe somebody else updated or deleted it meanwhile)");
	            }
	 
	        } catch (Exception e) {
	            System.out.println(sqlCommand);
	            throw new RuntimeException("Can not insert record in table "+getEntityDefinition().getName(),e);
	        } finally {
	            if (preparedStatement != null) {
	                try {
	                    preparedStatement.close();
	                } catch (Exception ex) {
	                    ex.printStackTrace();
	                }
	            }
	        }
	    }

	
	public void deleteObject(Object obj) {
		
		Connection conn = getConnection();
		
		String name = "_deleteObject";
		CommandMetaData commandMetaData = getCommandMetaData(name);
		if (commandMetaData == null) {
			commandMetaData = new CommandMetaData(getEntityDefinition());
			commandMetaData.setCommand(commandBuilder.buildDeleteObjectCommand(getEntityDefinition()));
			putCommandMetaData(name,commandMetaData);
		}
		String sqlCommand = commandMetaData.getCommand();
		
		FieldNavigator ii = new FieldNavigator();
		PreparedStatement preparedStatement = null;
		try {
			
			preparedStatement = conn.prepareStatement(sqlCommand);
			
			if (commandMetaData.getObjectMetaData() == null) {	
				commandMetaData.setPropertyDescriptors(createPropertyDescriptorsForClass(obj.getClass(),null,null),null);
			}
			writeUpdateParameter(preparedStatement,ii,obj,commandMetaData);
			int res = preparedStatement.executeUpdate();
			if (res == 0) {
				throw exceptionFactory.createConflictException("Record could not be deleted (maybe somebody else updated or deleted it meanwhile)");
			}
 
		} catch (Exception e) {
			System.out.println(sqlCommand);
			throw new RuntimeException("Can not insert record in table "+getEntityDefinition().getName(),e);
		} finally {
			if (preparedStatement != null) {
				try {
					preparedStatement.close();
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		}
	}
	
	protected void delete(String criteriaString, Object... criterias) {
		
		String name = "delete"+criteriaString;
		
		Connection conn = getConnection();
		
		CommandMetaData commandMetaData = getCommandMetaData(name);
		if (commandMetaData == null) {
			commandMetaData = new CommandMetaData(getEntityDefinition());
			commandMetaData.setCommand(commandBuilder.buildDeleteWhereCommand(getEntityDefinition(),criteriaString));
			putCommandMetaData(name,commandMetaData);
		}
		String sqlCommand = commandMetaData.getCommand();

		PreparedStatement preparedStatement = null;
		try {
			preparedStatement = conn.prepareStatement(sqlCommand);
			if (criteriaString != null) {
				for (int i=0;i<criterias.length;i++) {
					setParameter(preparedStatement,i+1,criterias[i]);
				}
			}
			preparedStatement.executeUpdate();

		} catch (Exception e) {
			System.out.println(sqlCommand);
			throw new RuntimeException("Can not read record from table "+getEntityDefinition().getName(),e);
		} finally {
			if (preparedStatement != null) {
				try {
					preparedStatement.close();
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		}
	}
	
	public class CommandCustomizer {
		
		public String[] customColumns() {
			return null;
		}
		
		public void customSetStatementParameter(PreparedStatement preparedStatement, FieldNavigator ii, Object obj) throws Exception {	
		}
		
		public void customReadResultSet(ResultSet rs, FieldNavigator ii, Object object) throws Exception {
		}
		
	}
	
	public<T> T load(Class<T> objectClass, Object... key) {
		T obj = reflectionHelper.createObject(objectClass);					
		load(null,obj,key);
		return obj;
	}
	
	public<T> void load(T object, Object... key) {
		load(null,object,key);	
	}
	
	protected<T> T load(CommandCustomizer statementAdapter, Class<T> objectClass, Object... key) {
		T obj = reflectionHelper.createObject(objectClass);					
		load(statementAdapter,obj,key);
		return obj;
	}
	
	protected<T> void load(CommandCustomizer statementAdapter, T obj, Object... key) {	
		if (obj == null) {
			throw new IllegalArgumentException("Object to be loaded into must not be null");
		}
		
		Class<T> objectClass = (Class<T>)obj.getClass();
	
		String name = "load"+objectClass.getSimpleName()+statementAdapter;
		
		Connection conn = getConnection();
		
		CommandMetaData commandMetaData = getCommandMetaData(name);
		if (commandMetaData == null) {
			commandMetaData = new CommandMetaData(getEntityDefinition());
			commandMetaData.setCommand(commandBuilder.buildLoadCommand(getEntityDefinition(),objectClass,statementAdapter));
			putCommandMetaData(name,commandMetaData);
		}
		String sqlCommand = commandMetaData.getCommand();

		FieldNavigator ii = new FieldNavigator();
		PreparedStatement preparedStatement = null;
		RuntimeException notFoundException = null;
		try {
			preparedStatement = conn.prepareStatement(sqlCommand);
			for (int i=0;i<key.length;i++) {
				setParameter(preparedStatement,i+1,key[i]);
			}
			ResultSet rs = preparedStatement.executeQuery();
			if (commandMetaData.getObjectMetaData() == null) {	
				commandMetaData.setPropertyDescriptors(createPropertyDescriptorsForClass(objectClass,statementAdapter,rs),statementAdapter);
			}
			if (rs.next()) {
				ii.reset();
				readFieldsFromResultSet(obj,commandMetaData.getObjectMetaData(),rs,ii);
				if (statementAdapter != null) {
					statementAdapter.customReadResultSet(rs,ii,obj);
				}
				return;
			}
			notFoundException = new RuntimeException("Object to load not found");
			throw notFoundException;
			
		} catch (Exception e) {
			System.out.println(sqlCommand);
			if (e == notFoundException) {
				throw exceptionFactory.createNotFoundException("Object not found in table "+getEntityDefinition().getName());
			} else {
				throw new RuntimeException("Can not read record from table "+getEntityDefinition().getName(),e);
			}
		} finally {
			if (preparedStatement != null) {
				try {
					preparedStatement.close();
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		}
	}
	
	public<T> Collection<T> findAll(Class<T> objectClass) {
	    return find(objectClass,null,null);
	}

	public<T> Collection<T> find(Class<T> objectClass, String criteriaString, String orderByString, Object... criterias) {
	    return findWithPaging(objectClass,criteriaString,orderByString,0,0,criterias);
	}
	
    public<T> Collection<T> findWithPaging(Class<T> objectClass, String criteriaString, String orderByString, int page, int pageSize, Object... criterias) {
        
        String name = "find"+objectClass.getSimpleName()+criteriaString+orderByString;
        if (page != 0) {
            name+="_paging";
        }
        CommandMetaData commandMetaData = getCommandMetaData(name);
        if (commandMetaData == null) {
            commandMetaData = new CommandMetaData(getEntityDefinition());
            commandMetaData.setCommand(commandBuilder.buildFindCommand(getEntityDefinition(),objectClass,null,criteriaString,orderByString,page != 0));
            putCommandMetaData(name,commandMetaData);
        }
        String sqlCommand = commandMetaData.getCommand();
        FieldNavigator ii = new FieldNavigator();
        Connection conn = getConnection();
        PreparedStatement preparedStatement = null;
        try {
            preparedStatement = conn.prepareStatement(sqlCommand);
            if (criteriaString != null) {
                for (int i=0;i<criterias.length;i++) {
                    setParameter(preparedStatement,i+1,criterias[i]);
                }
            }
            if (page != 0) {
                int parameterLength = criterias.length;
                setParameter(preparedStatement,parameterLength+1,(page-1)*pageSize);
                setParameter(preparedStatement,parameterLength+2,pageSize);
            }

            ResultSet rs = preparedStatement.executeQuery();
            if (commandMetaData.getObjectMetaData() == null) {
                commandMetaData.setPropertyDescriptors(createPropertyDescriptorsForClass(objectClass,null,rs),null);
            }
            
            ArrayList<T> resultList = new ArrayList<T>();
            while (rs.next()) {
                ii.reset();
                T obj = reflectionHelper.createObject(objectClass);
                readFieldsFromResultSet(obj,commandMetaData.getObjectMetaData(),rs,ii);
                resultList.add(obj);
            }
            return resultList;
        } catch (Exception e) {
            System.out.println(sqlCommand);
            throw new RuntimeException("Can not read record from table "+getEntityDefinition().getName(),e);
        } finally {
            if (preparedStatement != null) {
                try {
                    preparedStatement.close();
                } catch (Exception ex) {
                    ex.printStackTrace();
                }
            }
        }
    }

	protected<T> Collection<T> select(Class<T> objectClass, String selectSQL, Object... parameters) {
		
		String name = "select"+objectClass.getSimpleName()+selectSQL;
		CommandMetaData commandMetaData = getCommandMetaData(name);
		if (commandMetaData == null) {
			commandMetaData = new CommandMetaData(getEntityDefinition());
			commandMetaData.setCommand(selectSQL);
			putCommandMetaData(name,commandMetaData);
		}
		
		FieldNavigator ii = new FieldNavigator();
		Connection conn = getConnection();
		PreparedStatement preparedStatement = null;
		try {
			preparedStatement = conn.prepareStatement(selectSQL.toString());
			for (int i=0;i<parameters.length;i++) {
				setParameter(preparedStatement,i+1,parameters[i]);
			}

			ResultSet rs = preparedStatement.executeQuery();
			if (commandMetaData.getObjectMetaData() == null) {	
				commandMetaData.setPropertyDescriptors(createPropertyDescriptorsForClass(objectClass,null,rs),null);
			}
			
			ArrayList<T> resultList = new ArrayList<T>();
			while (rs.next()) {
				ii.reset();
				T obj = reflectionHelper.createObject(objectClass);
				readFieldsFromResultSet(obj,commandMetaData.getObjectMetaData(),rs,ii);
				resultList.add(obj);
			}
			return resultList;
		} catch (Exception e) {
			System.out.println(selectSQL);
			throw new RuntimeException("Can not read record from table "+getEntityDefinition().getName(),e);
		} finally {
			if (preparedStatement != null) {
				try {
					preparedStatement.close();
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		}
	}
	
   protected<T> Collection<T> selectWithPaging(Class<T> objectClass, String insertTableSQL, int page, int pageSize, Object... parameters) {
        if (page > 0 && pageSize > 0) {
            int parameterLength = parameters.length;
            List<Object> paramterList = Arrays.asList(parameters);
            Object[] parametersWithPaging = paramterList.toArray(new Object[parameterLength+2]);
            parametersWithPaging[parameterLength] = (page-1)*pageSize;
            parametersWithPaging[parameterLength+1] = pageSize;
            return select(objectClass,insertTableSQL+" OFFSET ? LIMIT ? ",parametersWithPaging);
        } else {
            return select(objectClass,insertTableSQL,parameters);
        }
    }
	
	protected<T> PropertyDescriptor[] createPropertyDescriptorsForClass(Class<T> objectClass, CommandCustomizer statementAdapter, ResultSet rs) throws Exception {
		Field[] objectClassFields = reflectionHelper.getAllFields(objectClass);
		List<PropertyDescriptor> fieldList = new ArrayList<PropertyDescriptor>();
		/* ResultSet available */
		if (rs != null) {
			ResultSetMetaData metaData = rs.getMetaData();
			for (int i=0;i<metaData.getColumnCount();i++) {
				String recordFieldName = metaData.getColumnLabel(i+1);
				String tableName = metaData.getTableName(i+1);
				String propertyName = getEntityDefinition().fieldNameForColumn(recordFieldName,tableName);
				Field recordField = getEntityDefinition().findField(objectClassFields,propertyName);
				if (recordField == null) {
					boolean isAdditionalColumn = false;
					if (statementAdapter != null) {
						for (String cn : statementAdapter.customColumns()) {
							if (recordFieldName.equalsIgnoreCase(cn)) {
								isAdditionalColumn = true;
								break;
							}
						}
					}
					if (isAdditionalColumn == false) {
						throw new RuntimeException("Property not found in bean '"+objectClass.getName()+"'. Expected property for field '"+recordFieldName+"' is '"+propertyName+"'" );
					}
				}
				if (recordField != null) {
					PropertyDescriptor propertyDescriptor = new PropertyDescriptor(recordField.getName(), objectClass);
					fieldList.add(propertyDescriptor);
				}
			}	
		/* No ResultSet available */
		} else {
			for (EntityFieldDefinition dbField : getEntityDefinition().getFields()) {
				String recordFieldName = dbField.getName();
				String tableName = getEntityDefinition().getName();
				String propertyName = getEntityDefinition().fieldNameForColumn(recordFieldName,tableName);
				Field recordField = getEntityDefinition().findField(objectClassFields,propertyName);
				if (recordField != null) {
					PropertyDescriptor propertyDescriptor = new PropertyDescriptor(propertyName, objectClass);
					fieldList.add(propertyDescriptor);
					
				}
			}
		}
		
		PropertyDescriptor[] resultSetFields = new PropertyDescriptor[fieldList.size()];
		fieldList.toArray(resultSetFields);
		
		
		return resultSetFields;
	}

}


