package org.iworkz.habitat.entity;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

import org.iworkz.habitat.helper.ReflectionHelper;
import org.iworkz.habitat.helper.StringHelper;


public class EntityDefinition {
	
	final private String name;
	
	private List<EntityFieldDefinition> fields = new ArrayList<EntityFieldDefinition>();
	private List<EntityFieldDefinition> primaryKeyFields = new ArrayList<EntityFieldDefinition>();
	private List<EntityFieldDefinition> versionFields = new ArrayList<EntityFieldDefinition>();
	
	protected StringHelper stringHelper = new StringHelper();
	protected ReflectionHelper reflectionHelper;// = new ReflectionHelper();
	
	public EntityDefinition(String name, ReflectionHelper reflectionHelper) {
		this.name = name;
		this.reflectionHelper = reflectionHelper;
	}
	
	public String getName() {
		return name;
	}

	public List<EntityFieldDefinition> getFields() {
		return fields;
	}
	
	public List<EntityFieldDefinition> getPrimaryKey() {
		return primaryKeyFields;
	}
	
	public List<EntityFieldDefinition> getVersion() {
		return versionFields;
	}
	


	public<T> void appendFieldAssigmentSql(StringBuilder sql, Class<T> objectClass, boolean supportOnlyEntityFields) {
		if (objectClass != null) {
          Field[] objectFields = reflectionHelper.getAllFields(objectClass);
            List<EntityFieldDefinition> fields = getFields();
            int count = 0;
            for (int i=0;i<fields.size();i++) {
                EntityFieldDefinition f = fields.get(i);
                for (int j=0;j<objectFields.length;j++) {
                    String columnName = columnNameForField(objectFields[j].getName(),supportOnlyEntityFields);
                    if (columnName != null && columnName.equals(f.getName())) {
                        if (count > 0 ) {
                            sql.append(", ");
                        }
                        sql.append(f.getName()+" = ?");
                        count++;
                        break;
                    }
                }
            }
		} else {
			for (int i=0;i<fields.size();i++) {
				EntityFieldDefinition f = fields.get(i);
				sql.append(f.getName()+" = ?");
				if (i != fields.size()-1 ) {
					sql.append(", ");
				}
			}
		}
	}
	
	public void appendAdditionalFieldAssigmentSql(StringBuilder sql, String[] columnNames) {
		if (columnNames != null) {
			for (String cn : columnNames) {
				sql.append(", ");
				sql.append(cn+" = ?");
			}
			
		}
	}
	
	public<T> void appendFieldSql(StringBuilder sql, Class<T> objectClass, boolean supportOnlyEntityFields) {
		if (objectClass != null) {
		    Field[] objectFields = reflectionHelper.getAllFields(objectClass);
		    List<EntityFieldDefinition> fields = getFields();
		    int count = 0;
            for (int i=0;i<fields.size();i++) {
                EntityFieldDefinition f = fields.get(i);
                for (int j=0;j<objectFields.length;j++) {
                    String columnName = columnNameForField(objectFields[j].getName(),supportOnlyEntityFields);
                    if (columnName != null && columnName.equals(f.getName())) {
                        if (count > 0 ) {
                            sql.append(", ");
                        }
                        sql.append(f.getName());
                        count++;
                        break;
                    }
                }
            }
		} else {
			List<EntityFieldDefinition> fields = getFields();
			for (int i=0;i<fields.size();i++) {
				EntityFieldDefinition f = fields.get(i);
				sql.append(f.getName());
				if (i != fields.size()-1 ) {
					sql.append(", ");
				}
			}
		}
	}
	
	public void appendAdditionalColumns(StringBuilder sql, String[] columnNames) {
		if (columnNames != null) {
			for (String cn : columnNames) {
				sql.append(", ");
				sql.append(cn);
			}
			
		}
	}

	public void appendFieldDefinitionsSql(StringBuilder sql) {
		for (EntityFieldDefinition f : getFields()) {
			sql.append(f.getName()+" "+f.getTypeName()+f.getConstraintsSql()+f.getNotNullSql()+", ");
		}
	}

	public<T> void appendFieldQuestionmarksSql(StringBuilder sql, Class<T> objectClass) {
		if (objectClass != null) {
		    
		    Field[] objectFields = reflectionHelper.getAllFields(objectClass);
            List<EntityFieldDefinition> fields = getFields();
            int count = 0;
            for (int i=0;i<fields.size();i++) {
                EntityFieldDefinition f = fields.get(i);
                for (int j=0;j<objectFields.length;j++) {
                    String columnName = columnNameForField(objectFields[j].getName(),false);
                    if (columnName != null && columnName.equals(f.getName())) {
                        if (count > 0 ) {
                            sql.append(", ");
                        }
                        sql.append("?");
                        count++;
                        break;
                    }
                }
            }
		} else {
			for (int i=0;i<fields.size();i++) {
				sql.append("?");
				if (i != fields.size()-1 ) {
					sql.append(", ");
				}
			}
		}
	}
	
	public void appendPrimaryKeySqL(StringBuilder sql) {
		if (primaryKeyFields.size() > 0) {
			sql.append("PRIMARY KEY (");
			for (EntityFieldDefinition p : getPrimaryKey()) {
				sql.append(p.getName());
			}
			sql.append(")");
		}
	}
	
	public void addField(EntityFieldDefinition field) {
		fields.add(field);
	}
	
	public void setPrimaryKey(String key) {
		if (key == null) {
			throw new IllegalArgumentException("Key musst not be null");
		}
		boolean found = false;
		for (EntityFieldDefinition f : fields) {
			if (key.equals(f.getName())) {
				f.setPrimaryKeyField(true);
				primaryKeyFields.add(f);
				found = true;
				break;
			}
		}
		if (!found) {
			throw new RuntimeException("No field for key '"+key+"'found");
		}
	}
	
	public void setVersion(String version) {
		if (version == null) {
			throw new IllegalArgumentException("Version musst not be null");
		}
		boolean found = false;
		for (EntityFieldDefinition f : fields) {
			if (version.equals(f.getName())) {
				f.setVersionField(true);
				versionFields.add(f);
				found = true;
				break;
			}
		}
		if (!found) {
			throw new RuntimeException("No field for key '"+version+"'found");
		}
	}

	public void appendPrimaryKeyCriteriaSql(StringBuilder sql) {
		for (int i=0;i<primaryKeyFields.size();i++) {
			EntityFieldDefinition f = primaryKeyFields.get(i);
			sql.append(f.getName()+" = ?");
			if (i != primaryKeyFields.size()-1 ) {
				sql.append(", ");
			}
		}
	}
	
	public void appendVersionCriteriaSql(StringBuilder sql) {
		if (versionFields.size() > 0) {
			sql.append(" AND ");
		}
		for (int i=0;i<versionFields.size();i++) {
			EntityFieldDefinition f = versionFields.get(i);
			sql.append(f.getName()+" = ?");
			if (i != versionFields.size()-1 ) {
				sql.append(", ");
			}
		}
	}
	
	protected String columnNameForField(String fieldName, boolean supportOnlyEntityFields) {
		String columnName = columnNameForRecordField(fieldName);

		/* If field name not found in table definition use naming strategy */
		if (columnName == null && supportOnlyEntityFields == false) {
			columnName =  stringHelper.camelCaseToUnderLineSeperated(fieldName);
			if (columnName != null) {
				columnName = columnName.toUpperCase();
			}
		}
		return columnName;
	}
	
	public String fieldNameForColumn(String columnName, String tableName) {
		String fieldName = null;
		/* When table name available, try name mapping from table definition */
		if (stringHelper.hasValue(tableName) && tableName.equalsIgnoreCase(getName())) {
			fieldName = recordFieldNameForColumn(columnName);
		}
		/* If field name not found in table definition use naming strategy */
		if (fieldName == null) {
			fieldName =  stringHelper.underLineSeperatedToCamelCase(columnName);		
		}
		return fieldName;
	}
	
	protected String columnNameForRecordField(String fieldName) {
		for (EntityFieldDefinition dbField : getFields()) {
			if (fieldName.equalsIgnoreCase(dbField.objectFieldName)) {
				return dbField.getName();
			}
		}
		return null;
	}
	
	protected EntityFieldDefinition databaseFieldForColumn(String columnName) {
		for (EntityFieldDefinition dbField : getFields()) {
			if (columnName.equalsIgnoreCase(dbField.getName())) {
				return dbField;
			}
		}
		return null;
	}
	
	protected String recordFieldNameForColumn(String columnName) {
		for (EntityFieldDefinition dbField : getFields()) {
			if (columnName.equalsIgnoreCase(dbField.getName())) {
				return dbField.objectFieldName;
			}
		}
		return null;
	}

	public <T> Field findField(Field[] fields, String fieldName) {
		for (Field f : fields) {
			if (f.getName().equals(fieldName)) {
				return f;
			}
		}
		return null;
	}

	public EntityFieldDefinition getDatabaseFieldForObjectFieldName(String name) {
		for (EntityFieldDefinition databaseField : getFields()) {
			String columnName = columnNameForRecordField(name);
			if (databaseField.getName().equalsIgnoreCase(columnName)) {
				return databaseField;
			}
		};
		return null;
	}


}
