package org.iworkz.habitat.helper;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Array;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import javax.inject.Inject;
import javax.inject.Singleton;

@Singleton
public class BeanHelper {
	
	@Inject
	protected ReflectionHelper reflectionHelper;
	
	public <T> T cloneBean(T source) {
		Map<Object,Object> cloneStack = new HashMap<Object,Object>();
		return cloneBean(source,cloneStack);
	}
	
	@SuppressWarnings("unchecked")
	protected <T> T cloneBean(T source, Map<Object, Object> cloneMap) {
		if (source != null) {
			if (cloneMap.containsKey(this)) {
				return (T)cloneMap.get(this);
			} else {
				T destination = (T)reflectionHelper.createObject(source.getClass());
				cloneBean(source, destination,cloneMap);
				cloneMap.put(this, destination);
				return destination;
			}
		}
		return null;
	}
	
	@SuppressWarnings("unchecked")
	protected <T> void cloneBean(T source, T destination, Map<Object,Object> cloneMap) {
		if (source != null) {
			if (destination == null) {
				throw new IllegalArgumentException("The destination bean is null");
			}
			try {
				BeanInfo beanInfo = Introspector.getBeanInfo(source.getClass());
				PropertyDescriptor[] propertyDescriptors = beanInfo.getPropertyDescriptors();
				for (PropertyDescriptor pd : propertyDescriptors) {
					Method readMethod = pd.getReadMethod();
					if (readMethod != null) {
						Method writeMethod = pd.getWriteMethod();
						if (writeMethod != null) {
							Object value = readMethod.invoke(source);
							if (value != null) {
								Class<?> valueClass = value.getClass();
								if (valueClass.isArray()) {
									writeMethod.invoke(destination,cloneArray(value,cloneMap));
								} else if (Collection.class.isAssignableFrom(valueClass)) {
									writeMethod.invoke(destination,cloneCollection((Collection<?>)value,cloneMap));
								} else if (shouldClonePropertyOfType(valueClass)) {
									writeMethod.invoke(destination,cloneBean(value,cloneMap));
								} else {
									writeMethod.invoke(destination, value);
								}
							}
						}
					}
				}
			} catch (Exception e) {
				throw new RuntimeException("Can not clone bean '"+source.getClass().getCanonicalName()+"'",e);
			}
		} 
	}
	

	protected boolean shouldClonePropertyOfType(Class<?> propertyClass) {
		if (propertyClass.isPrimitive()) {
			return false;
		}
		Package propertyPackage = propertyClass.getPackage();
		String packageName = propertyPackage.getName();
		if (packageName.startsWith("java")) {
			return false;
		}
		return true;
	}
	
	protected Object cloneArray(Object value, Map<Object,Object> cloneMap) {
		Class<?> valueClass = value.getClass().getComponentType();
		int length = Array.getLength(value);
		Object destination = Array.newInstance(valueClass,length);
		if (shouldClonePropertyOfType(valueClass)) {
			for (int i=0;i<length;i++) {
				Object component = Array.get(value, i);
				Array.set(destination, i, cloneBean(component,cloneMap));
			}
		} else {
			 System.arraycopy(value,0,destination,0,length);
		}
		return destination;
	}

	@SuppressWarnings("unchecked")
	protected Object cloneCollection(Collection<?> value, Map<Object,Object> cloneMap) {
		Collection<Object> destination = reflectionHelper.createObject(value.getClass());
		for (Object sourceItem : value) {
			Class<?> valueClass = value.getClass();
			if (shouldClonePropertyOfType(valueClass)) {
				destination.add(cloneBean(sourceItem,cloneMap));
			} else {
				destination.add(sourceItem);
			}
		}
		return destination;
	}
}
