package org.iworkz.common.helper;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.io.Closeable;
import java.lang.reflect.Array;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import javax.inject.Inject;
import javax.inject.Singleton;

@Singleton
public class CloneHelper {
	
	private static final Object PROPERTIES_CACHE_LOCK = new Object();
	private static final Map<Class<?>,PropertyInfo[]> PROPERTIES_CACHE = new IdentityHashMap<>();
	
	protected static final Set<Class<?>> FINAL_IMMUTABLES = new HashSet<>();
	
	static {
		FINAL_IMMUTABLES.add(String.class);
		FINAL_IMMUTABLES.add(Boolean.class);
		FINAL_IMMUTABLES.add(Character.class);
		FINAL_IMMUTABLES.add(Byte.class);
		FINAL_IMMUTABLES.add(Short.class);
		FINAL_IMMUTABLES.add(Integer.class);
		FINAL_IMMUTABLES.add(Long.class);
		FINAL_IMMUTABLES.add(Float.class);
		FINAL_IMMUTABLES.add(Double.class);
		FINAL_IMMUTABLES.add(Locale.class);
		FINAL_IMMUTABLES.add(UUID.class);
		FINAL_IMMUTABLES.add(BigInteger.class);
		FINAL_IMMUTABLES.add(BigDecimal.class);
	}
	
	@Inject
	protected ReflectionHelper reflectionHelper;
	
	/**
	 * Clone the source bean and the properties of the source bean recursively (deep copy).
	 * Cyclic dependencies are detected and handled correctly.
	 * 
	 * @param source
	 * @return
	 */
	public <T> T cloneBean(T source) {
		Map<Object,Object> instanceMap = new IdentityHashMap<>();
		return cloneBean(source,instanceMap);
	}
	
	@SuppressWarnings("unchecked")
	protected <T> T cloneBean(final T source, final Map<Object, Object> instanceMap) {
		if (source != null) {
			if (instanceMap.containsKey(source)) {
				return (T)instanceMap.get(source);
			} else {
				Class<?> sourceClass = source.getClass();
				T destination;
				if (sourceClass.isArray()) {
					final Class<?> componentType = sourceClass.getComponentType();
					boolean finalImmutuable = isImmutable(componentType);
					destination = (T)cloneArray(source,componentType,finalImmutuable,instanceMap);
				} else if (Collection.class.isAssignableFrom(sourceClass)) {
					destination = (T)cloneCollection((Collection<?>)source,instanceMap);
				} else if (Map.class.isAssignableFrom(sourceClass)) {
					destination = (T)cloneMap((Map<?,?>)source,instanceMap);
				} else if (isCloneRequired(sourceClass)) {
					destination = clone(source,instanceMap);
				} else {
					destination = source;
				}
				return destination;
			}
		}
		return null;
	}
	
	protected <T> T clone(final T source, final Map<Object, Object> instanceMap) {
		T destination = createCustomClone(source,instanceMap);
		if (destination != null) {
			instanceMap.put(source, destination);
		} else {
			/* create generic clone */
			destination = createClone(source);
			instanceMap.put(source, destination);
			cloneProperties(source, destination, instanceMap);
		}
		return destination;
	}
	
	@SuppressWarnings("unchecked")
	protected <T> T createClone(final T source) {
		return (T)reflectionHelper.createObject(source.getClass());
	}
	
	@SuppressWarnings("unchecked")
	protected <T> T createCustomClone(final T source, final Map<Object, Object> instanceMap) {
		T destination = null;
		Class<?> sourceClass = source.getClass();
		if (Date.class.isAssignableFrom(sourceClass)) {
			if (java.sql.Date.class == sourceClass) {
				destination = (T)new java.sql.Date(((java.sql.Date)source).getTime());
			} else if (java.util.Date.class == sourceClass) {
				destination = (T)new java.util.Date(((java.util.Date)source).getTime());
			} else if (java.sql.Time.class == sourceClass) {
				destination = (T)new java.sql.Time(((java.sql.Time)source).getTime());
			} else if (java.sql.Timestamp.class == sourceClass) {
				destination = (T)new java.sql.Timestamp(((java.sql.Timestamp)source).getTime());
			} else {
				destination = null;
			}
		}
		return destination;
	}
	
	protected <T> void cloneProperties(final T source, final T destination, final Map<Object,Object> instanceMap) {
		if (source != null) {
			if (destination == null) {
				throw new IllegalArgumentException("The destination bean is null");
			}
			try {
				PropertyInfo[] propertyInfos = getPropertyInfos(source.getClass());
				for (PropertyInfo propertyInfo : propertyInfos) {
					Object value = propertyInfo.readMethod.invoke(source);
					if (value != null) {
						if (propertyInfo.immutable) {
							propertyInfo.writeMethod.invoke(destination,value);
						} else if (propertyInfo.map) {
							propertyInfo.writeMethod.invoke(destination,cloneMap((Map<?,?>)value,instanceMap));
						} else if (propertyInfo.collection) {
							propertyInfo.writeMethod.invoke(destination,cloneCollection((Collection<?>)value,instanceMap));
						} else if (propertyInfo.array) {
							final Class<?> componentType = value.getClass().getComponentType();
							propertyInfo.writeMethod.invoke(destination,cloneArray(value,componentType,propertyInfo.componentImmutable,instanceMap));
						} else {
							propertyInfo.writeMethod.invoke(destination,cloneBean(value,instanceMap));
						}
					}
				}
			} catch (Exception e) {
				throw new RuntimeException("Can not clone bean '"+source.getClass().getCanonicalName()+"'",e);
			}
		} 
	}
	
	@SuppressWarnings("unchecked")
	protected <T> Collection<T> createCollection(final Collection<T> sourceCollection) {
		return (Collection<T>)reflectionHelper.createObject(sourceCollection.getClass());
	}
	
	@SuppressWarnings("unchecked")
	protected <T,R> Map<T,R> createMap(final Map<T,R> sourceMap) {
		return (Map<T,R>)reflectionHelper.createObject(sourceMap.getClass());
	}

	protected boolean isCloneRequired(Class<?> sourceClass) {
		if (isImmutable(sourceClass)) {
			return false;
		}
		if (Closeable.class.isAssignableFrom(sourceClass)) {
			return false;
		}
		return true;
	}
	
	protected boolean isImmutable(Class<?> valueClass) {
		if (valueClass.isPrimitive()) {
			return false;
		}
		if (valueClass.isEnum()) {
			return false;
		}
		return FINAL_IMMUTABLES.contains(valueClass);
	}
	
	protected Object cloneArray(Object source, Class<?> sourceClass, final boolean finalImmutable, final Map<Object,Object> instanceMap) {
		if (instanceMap.containsKey(source)) {
			return instanceMap.get(source);
		} else {
			int length = Array.getLength(source);
			Object destination = Array.newInstance(sourceClass,length);
			instanceMap.put(source, destination);
			if (finalImmutable) {
				System.arraycopy(source,0,destination,0,length);
			} else {
				for (int i=0;i<length;i++) {
					Object component = Array.get(source, i);
					Array.set(destination, i, cloneBean(component,instanceMap));
				}
			}
			return destination;
		}
	}

	protected <T> Collection<T> cloneCollection(final Collection<T> collection, final Map<Object,Object> instanceMap) {
		if (instanceMap.containsKey(collection)) {
			return (Collection<T>)instanceMap.get(collection);
		} else {
			Collection<T> destination = createCollection(collection);
			instanceMap.put(collection, destination);
			for (T sourceItem : collection) {
				destination.add(cloneBean(sourceItem,instanceMap));
			}
			return destination;
		}
	}
	
	protected <T,R> Map<T,R> cloneMap(final Map<T,R> map, final Map<Object,Object> instanceMap) {
		if (instanceMap.containsKey(map)) {
			return (Map<T,R>)instanceMap.get(map);
		} else {
			Map<T,R> destination = createMap(map);
			instanceMap.put(map, destination);
			for (T sourceKey : map.keySet()) {
				R sourceItem = map.get(sourceKey);
				destination.put(sourceKey,cloneBean(sourceItem,instanceMap));
			}
			return destination;
		}
	}
	
	protected PropertyInfo[] getPropertyInfos(final Class<?> sourceClass) {
		PropertyInfo[] propertyInfos = PROPERTIES_CACHE.get(sourceClass);
		if (propertyInfos == null) {
			synchronized (PROPERTIES_CACHE_LOCK) {
				try {
					/* get propertyInfos again but this time synchronized */
					propertyInfos = PROPERTIES_CACHE.get(sourceClass);
					if (propertyInfos == null) {
						List<PropertyInfo> propertyInfoList = new ArrayList<>();
						BeanInfo beanInfo = Introspector.getBeanInfo(sourceClass);
						PropertyDescriptor[] propertyDescriptors = beanInfo.getPropertyDescriptors();
						for (PropertyDescriptor pd : propertyDescriptors) {
							Method readMethod = pd.getReadMethod();
							if (readMethod != null) {
								Method writeMethod = pd.getWriteMethod();
								if (writeMethod != null) {
									Class<?> valueClass = pd.getPropertyType();
									propertyInfoList.add(createPropertyInfo(valueClass,readMethod,writeMethod));
								}
							}
						}
						propertyInfos = propertyInfoList.toArray(new PropertyInfo[propertyInfoList.size()]);
						PROPERTIES_CACHE.put(sourceClass, propertyInfos);
					}
				} catch (Exception ex) {
					throw new RuntimeException("Can not create property infos",ex);
				}
			}
		}
		return propertyInfos;
	}
	
	protected PropertyInfo createPropertyInfo(final Class<?> valueClass, Method readMethod, Method writeMethod) {
		PropertyInfo propertyInfo = new PropertyInfo();
		propertyInfo.readMethod = readMethod;
		propertyInfo.writeMethod = writeMethod;
		if (valueClass.isArray()) {
			propertyInfo.array = true;
			propertyInfo.componentImmutable=isImmutable(valueClass.getComponentType());
		} else if (Collection.class.isAssignableFrom(valueClass)) {
			propertyInfo.collection = true;
		} else if (Map.class.isAssignableFrom(valueClass)) {
			propertyInfo.map = true;
		}  else if (isImmutable(valueClass)) {
			propertyInfo.immutable=isImmutable(valueClass);
		}
		return propertyInfo;
	}
	
	protected static class PropertyInfo {
		
		Method readMethod;
		Method writeMethod;

		boolean collection;
		boolean map;
		boolean array;
		boolean immutable;
		boolean componentImmutable;
		
	}

}
