package org.iworkz.provenience;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.Stack;

import javax.inject.Inject;

import org.iworkz.common.helper.ReflectionHelper;

public class Injector {
	
	ReflectionHelper reflectionHelper = new ReflectionHelper();
	
	protected final Map<Class<?>,Provider<?>> providers;
	protected final Map<Class<?>,Binding<?>> bindings;
	
	
	public Injector(Module... modules) {
		
		bindings = new HashMap<Class<?>,Binding<?>>();
		providers = new HashMap<Class<?>,Provider<?>>();
		
		providers.put(Injector.class, new Provider<Injector>(this));
		
		if (modules != null) {
			for (Module module : modules) {
				module.configure();
				if (module.getBindings() != null) {
					for (Class<?> singletonClass : module.getBindings().keySet()) {
						Binding<?> binding = module.getBindings().get(singletonClass);
						bindings.put(singletonClass, binding);
					}
				}
			}
		}
	}
	
	public void injectMembers(Object instance) {
		if (instance != null) {
			Stack<Class<?>> creationStack = new Stack<Class<?>>();
			injectMembers(instance,instance.getClass(),creationStack);
		}
	}
	
	protected void injectMembers(Object instance, Class<?> classWithMembers, Stack<Class<?>> creationStack) {
		/* inject fields */
		for (Field field : reflectionHelper.getAllFields(classWithMembers)) {
			Annotation injectAnnotation = field.getAnnotation(Inject.class);
			if (injectAnnotation != null) {
				try {
					field.setAccessible(true);
					field.set(instance, get(field.getType(),creationStack));
				} catch (Exception e) {
					throw new RuntimeException("Can not inject field '"+field.getName()+"' of class '"+instance.getClass().getCanonicalName()+"'",e);
				}
			}
		}
		/* inject setters */
		try {
			BeanInfo beanInfo = Introspector.getBeanInfo(classWithMembers);
			for (PropertyDescriptor pd : beanInfo.getPropertyDescriptors()) {
//				System.out.println("Property name = "+beanInfo.getBeanDescriptor().getName()+"."+ pd.getName());
				Method setter = pd.getWriteMethod();
				if (setter != null) {
					setter.setAccessible(true);
					if (setter.isAnnotationPresent(Inject.class)) {
						setter.invoke(instance, get(setter.getParameterTypes()[0],creationStack));
					}
				}
			}
		} catch (Exception e) {
			throw new RuntimeException("Can not inject setter of class '"+instance.getClass().getCanonicalName()+"'",e);
		}
	}
	
	public <T> T get(Class<T> instanceClass) {
		Stack<Class<?>> creationStack = new Stack<Class<?>>();
		return get(instanceClass,creationStack);
	}
	
	@SuppressWarnings("unchecked")
	protected <T> T get(Class<T> instanceClass, Stack<Class<?>> creationStack) {
		if (creationStack.contains(instanceClass)) {
			throw new RuntimeException("Circular dependency detected during creation of '"+instanceClass+"', stack = "+creationStack.toString());
		}
		Provider<T> provider = (Provider<T>)providers.get(instanceClass);
		if (provider == null) {
			Binding<T> binding = (Binding<T>)bindings.get(instanceClass);
			if (binding != null) {
				provider = new Provider<T>(instanceClass,binding.implementationClass,binding.getInstance());
			} else {
				provider = new Provider<T>(instanceClass,instanceClass,null);
			}
			providers.put(instanceClass, provider);
		}
		return (T) provider.getInstance(this,creationStack);
	}

}
