package org.iworkz.provenience;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Qualifier;

import org.iworkz.common.helper.ReflectionHelper;

public class Injector {
	
	ReflectionHelper reflectionHelper = new ReflectionHelper();
	
	protected final Map<Class<?>,Provider<?>> providers;
	protected final Map<Class<?>,Binding<?>> bindings;
	
	protected final Set<Module> modules;
	
	
	public Injector(Module... modules) {
		
		if (modules != null && modules.length > 0) {
			this.modules = new HashSet<>();
			for (Module module : modules) {
				this.modules.add(module);
			}
		} else {
			this.modules = null;
		}
		
		bindings = new HashMap<>();
		providers = new HashMap<>();
		
		providers.put(Injector.class, new Provider<Injector>(this));
		
		if (modules != null) {
			for (Module module : modules) {
				module.configure();
				if (module.getBindings() != null) {
					for (Class<?> singletonClass : module.getBindings().keySet()) {
						Binding<?> binding = module.getBindings().get(singletonClass);
						bindings.put(singletonClass, binding);
						// TODO merge annotation + annotationClass bindings
					}
				}
			}
		}
	}
	
	public void injectMembers(Object instance) {
		if (instance != null) {
			InjectionContext ctx = new InjectionContext();
			injectMembers(instance,instance.getClass(),ctx);
			postProcess(ctx);
		}
	}
	
	protected void postProcess(InjectionContext ctx) {
		if (modules != null) {
			for (Module module : modules) {
				module.postProcess(ctx.createdInstances);
			}
		}
	}
	
	protected void injectMembers(Object instance, Class<?> classWithMembers, InjectionContext ctx) {
		/* inject fields */
		for (Field field : reflectionHelper.getAllFields(classWithMembers)) {
			Annotation injectAnnotation = field.getAnnotation(Inject.class);
			if (injectAnnotation != null) {
				try {
					field.setAccessible(true);
					Object injectedObject = get(field.getType(),field,ctx);
					ctx.putInjectedObject(instance, injectedObject);
					field.set(instance, injectedObject);
				} catch (Exception e) {
					throw new RuntimeException("Can not inject field '"+field.getName()+"' of class '"+instance.getClass().getCanonicalName()+"'",e);
				}
			}
		}
		/* inject setters */
		try {
			BeanInfo beanInfo = Introspector.getBeanInfo(classWithMembers);
			for (PropertyDescriptor pd : beanInfo.getPropertyDescriptors()) {
//				System.out.println("Property name = "+beanInfo.getBeanDescriptor().getName()+"."+ pd.getName());
				Method setter = pd.getWriteMethod();
				if (setter != null) {
					setter.setAccessible(true);
					if (setter.isAnnotationPresent(Inject.class)) {
						Object injectedObject = get(setter.getParameterTypes()[0],setter,ctx);
						ctx.putInjectedObject(instance, injectedObject);
						setter.invoke(instance, injectedObject);
					}
				}
			}
		} catch (Exception e) {
			throw new RuntimeException("Can not inject setter of class '"+instance.getClass().getCanonicalName()+"'",e);
		}
	}
	
	public <T> T get(Class<T> instanceClass) {
		InjectionContext ctx = new InjectionContext();
		T t = get(instanceClass,null,ctx);
		postProcess(ctx);
		return t;
	}
	
	@SuppressWarnings("unchecked")
	protected <T> T get(Class<T> instanceClass, AnnotatedElement annotatedElement, InjectionContext ctx) {
		Annotation annotation = null;
		if (annotatedElement != null) {
			if (annotatedElement.isAnnotationPresent(Named.class)) {
				annotation = annotatedElement.getAnnotation(Named.class);
			} else {
				for (Annotation elementAnnotation : annotatedElement.getAnnotations()) {
					Class<?> annotationType = elementAnnotation.annotationType();
					if (annotationType.isAnnotationPresent(Qualifier.class)) {
						annotation = elementAnnotation;
						break;
					}
				}
			}
		}
		if (ctx.creationStack.contains(instanceClass)) {
			throw new RuntimeException("Circular dependency detected during creation of '"+instanceClass+"', stack = "+ctx.creationStack.toString());
		}
		Provider<T> provider = (Provider<T>)providers.get(instanceClass);
		if (provider == null) {
			Binding<T> binding = (Binding<T>)bindings.get(instanceClass);
			if (binding != null) {
				/* standard provider */
				provider = new Provider<T>(instanceClass,binding.implementationClass,binding.getInstance());
				/* annotation providers */
				if (binding.annotationBindings != null) {
					provider.annotationProviders = new HashMap<>();
					for (Annotation annotationInstance : binding.annotationBindings.keySet()) {
						Binding<T> annotationBinding = binding.annotationBindings.get(annotationInstance);
						provider.annotationProviders.put(annotationInstance, new Provider<T>(instanceClass,annotationBinding.implementationClass,annotationBinding.getInstance()));
					}
				}
				/* annotation class providers */
				if (binding.annotationClassBindings != null) {
					provider.annotationClassProviders = new HashMap<>();
					for (Class<? extends Annotation> annotationClass : binding.annotationClassBindings.keySet()) {
						Binding<T> annotationClassBinding = binding.annotationClassBindings.get(annotationClass);
						provider.annotationClassProviders.put(annotationClass, new Provider<T>(instanceClass,annotationClassBinding.implementationClass,annotationClassBinding.getInstance()));
					}
				}
			} else {
				/* default provider */
				provider = new Provider<T>(instanceClass,null,null);
			}
			providers.put(instanceClass, provider);
		}
		return (T) provider.getInstance(this,annotation,ctx);
	}

}
