package org.jackstaff.grpc.generator;

import com.google.common.base.CaseFormat;
import com.squareup.javapoet.*;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.TypeElement;
import java.io.IOException;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Collectors;

class Utils {

    public static String packageName(TypeElement typeElement){
        return typeElement.getEnclosingElement()+"";
    }

    public static void write(ProcessingEnvironment env, TypeElement packRef, TypeSpec.Builder builder) {
        write(env, packageName(packRef), builder);
    }

    public static void write(ProcessingEnvironment env, String packageName, TypeSpec.Builder builder) {
        builder.addJavadoc("Generated by org.jackstaff.grpc.generator.ProtocolProcessor;  DO NOT EDIT!");
        JavaFile javaFile = JavaFile.builder(packageName, builder.build()).
                skipJavaLangImports(true).indent("    ").build();
        try {
            javaFile.writeTo(env.getFiler());
        } catch (IOException ex) {
            ex.printStackTrace();
        }
    }

    public static ParameterSpec parameter(Element el){
        return parameter(typeName(el));
    }

    public static ParameterSpec parameter(TypeName tn){
        return ParameterSpec.builder(tn, lowerCamelName(tn)).build();
    }

    public static ParameterSpec responseParameter(MethodInfo method) {
        String suffix;
        switch (method.getMethodType()) {
            case UNARY:
            case CLIENT_STREAMING:
                suffix = "Result";
                break;
            case BIDI_STREAMING:
            case SERVER_STREAMING:
            default:
                suffix = "Stream";
                break;
        }
        switch (method.getMethodType()) {
            case CLIENT_STREAMING:
            case BIDI_STREAMING:
                return streamingParameter(method.response(), suffix);
            case UNARY:
            case SERVER_STREAMING:
            default:
                if (method.request().toString().endsWith(suffix)){
                    return streamingParameter(method.response(), "Consumer");
                }
                return streamingParameter(method.response(), suffix);
        }
    }

    private static ParameterSpec streamingParameter(TypeName tn, String suffix){
        return ParameterSpec.builder(consumerTypeName(tn), lowerCamelName(tn)+suffix).build();
    }

    public static TypeName typeName(Element el){
        return TypeName.get(el.asType());
    }

    public static TypeName consumerTypeName(TypeName tn){
        return ParameterizedTypeName.get(ClassName.get(Consumer.class), tn);
    }

    public static TypeName listTypeName(TypeName tn){
        return ParameterizedTypeName.get(ClassName.get(List.class), tn);
    }

    public static String lowerCamelName(TypeName tn){
        if (tn instanceof ClassName){
            return lowerCamelName(((ClassName) tn).simpleName());
        }
        return lowerCamelName(tn.toString());
    }

    public static String lowerCamelName(String name){
        return CaseFormat.UPPER_CAMEL.to(CaseFormat.LOWER_CAMEL, name.indexOf(".")>0 ? name.substring(name.lastIndexOf(".")+1): name);
    }

    public static List<String> enumNames(TypeElement typeElement){
        return typeElement.getEnclosedElements().stream().filter(ele->ele.getKind()== ElementKind.ENUM_CONSTANT).
                map(Object::toString).collect(Collectors.toList());
    }

}
