/*
 *
 *
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "[]"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright [2017] [Andy Moncsek]
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package org.jacpfx.vertx.event.eventbus.blocking;

import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import io.vertx.core.eventbus.DeliveryOptions;
import io.vertx.core.eventbus.Message;
import io.vertx.core.shareddata.Counter;
import io.vertx.core.shareddata.Lock;
import io.vertx.core.shareddata.SharedData;
import org.jacpfx.common.throwable.ThrowableFunction;
import org.jacpfx.common.throwable.ThrowableSupplier;
import org.jacpfx.common.encoder.Encoder;
import org.jacpfx.vertx.event.interfaces.blocking.RecursiveBlockingExecutor;
import org.jacpfx.vertx.event.interfaces.blocking.RetryBlockingExecutor;
import org.jacpfx.vertx.event.response.basic.ResponseExecution;

import java.util.Optional;
import java.util.function.Consumer;

/**
 * Created by Andy Moncsek on 05.04.16.
 * Handles event-bus call and blocking execution of the message to create an event-bus response
 */
public class EventbusBridgeBlockingExecution {

    public static final long LOCK_VALUE = -1l;
    public static final int DEFAULT_LOCK_TIMEOUT = 2000;
    public static final long NO_TIMEOUT = 0l;


    /**
     * Send event-bus message and process the result in the passed function for blocking execution chain
     * @param methodId the method identifier
     * @param targetId the event-bus target id
     * @param message the message to send
     * @param function the function to process the result message
     * @param deliveryOptions the event-bus delivery options
     * @param vertx  the vertx instance
     * @param errorMethodHandler the error-method handler
     * @param requestMessage the request message to respond after chain execution
     * @param encoder the encoder to serialize the response object
     * @param errorHandler the error handler
     * @param onFailureRespond the function that takes a Future with the alternate response value in case of failure
     * @param responseDeliveryOptions the delivery options for the event response
     * @param retryCount the amount of retries before failure execution is triggered
     * @param timeout the amount of time before the execution will be aborted
     * @param delay the delay time in ms between an execution error and the retry
     * @param circuitBreakerTimeout the amount of time before the circuit breaker closed again
     * @param executor the typed executor to process the chain
     * @param retryExecutor the typed retry executor of the chain
     * @param <T>  the type of response
     */
    public static <T> void sendMessageAndSupplyHandler(String methodId,
                                                       String targetId,
                                                       Object message,
                                                       ThrowableFunction<AsyncResult<Message<Object>>, T> function,
                                                       DeliveryOptions deliveryOptions,
                                                       Vertx vertx,
                                                       Consumer<Throwable> errorMethodHandler,
                                                       Message<Object> requestMessage,
                                                       Encoder encoder,
                                                       Consumer<Throwable> errorHandler,
                                                       ThrowableFunction<Throwable, T> onFailureRespond,
                                                       DeliveryOptions responseDeliveryOptions,
                                                       int retryCount, long timeout, long delay, long circuitBreakerTimeout,
                                                       RecursiveBlockingExecutor executor, RetryBlockingExecutor retryExecutor) {

        if (circuitBreakerTimeout == 0l) {
            executeDefaultState(targetId,
                    message,
                    function,
                    deliveryOptions,
                    methodId,
                    vertx,
                    errorMethodHandler,
                    requestMessage,
                    encoder,
                    errorHandler,
                    onFailureRespond,
                    responseDeliveryOptions,
                    retryCount, timeout, delay, circuitBreakerTimeout, executor, retryExecutor, null);
        } else {
            executeStateful(targetId,
                    message,
                    function,
                    deliveryOptions,
                    methodId,
                    vertx,
                    errorMethodHandler,
                    requestMessage,
                    encoder,
                    errorHandler,
                    onFailureRespond,
                    responseDeliveryOptions,
                    retryCount, timeout, delay, circuitBreakerTimeout, executor, retryExecutor);
        }
    }

    private static <T> void executeStateful(String targetId,
                                            Object message,
                                            ThrowableFunction<AsyncResult<Message<Object>>, T> byteFunction,
                                            DeliveryOptions deliveryOptions,
                                            String methodId,
                                            Vertx vertx,
                                            Consumer<Throwable> errorMethodHandler,
                                            Message<Object> requestMessage,
                                            Encoder encoder,
                                            Consumer<Throwable> errorHandler,
                                            ThrowableFunction<Throwable, T> onFailureRespond,
                                            DeliveryOptions responseDeliveryOptions,
                                            int retryCount, long timeout, long delay, long circuitBreakerTimeout,
                                            RecursiveBlockingExecutor executor, RetryBlockingExecutor retry) {

        executeLocked(((lock, counter) ->
                counter.get(counterHandler -> {
                    long currentVal = counterHandler.result();
                    if (currentVal == 0) {
                        executeInitialState(targetId,
                                message,
                                byteFunction,
                                deliveryOptions,
                                methodId,
                                vertx,
                                errorMethodHandler,
                                requestMessage,
                                encoder,
                                errorHandler,
                                onFailureRespond,
                                responseDeliveryOptions,
                                retryCount, timeout, delay,
                                circuitBreakerTimeout, executor,
                                retry, lock, counter);
                    } else if (currentVal > 0) {
                        executeDefaultState(targetId,
                                message,
                                byteFunction,
                                deliveryOptions,
                                methodId,
                                vertx,
                                errorMethodHandler,
                                requestMessage,
                                encoder,
                                errorHandler,
                                onFailureRespond,
                                responseDeliveryOptions,
                                retryCount, timeout, delay,
                                circuitBreakerTimeout, executor, retry, lock);
                    } else {
                        executeErrorState(methodId,
                                vertx,
                                errorMethodHandler,
                                requestMessage,
                                encoder,
                                errorHandler,
                                onFailureRespond,
                                responseDeliveryOptions,
                                retryCount, timeout, delay,
                                circuitBreakerTimeout,executor, lock);
                    }
                })), methodId, vertx, errorMethodHandler, requestMessage, encoder, errorHandler, onFailureRespond, responseDeliveryOptions, retryCount, timeout, delay, circuitBreakerTimeout,executor);
    }


    private static <T> void executeInitialState(String targetId,
                                                Object message,
                                                ThrowableFunction<AsyncResult<Message<Object>>, T> byteFunction,
                                                DeliveryOptions deliveryOptions,
                                                String methodId,
                                                Vertx vertx,
                                                Consumer<Throwable> errorMethodHandler,
                                                Message<Object> requestMessage,
                                                Encoder encoder,
                                                Consumer<Throwable> errorHandler,
                                                ThrowableFunction<Throwable, T> onFailureRespond,
                                                DeliveryOptions responseDeliveryOptions,
                                                int retryCount, long timeout, long delay, long circuitBreakerTimeout,
                                                RecursiveBlockingExecutor executor, RetryBlockingExecutor retry, Lock lock, Counter counter) {
        int incrementCounter = retryCount + 1;
        counter.addAndGet(Integer.valueOf(incrementCounter).longValue(), rHandler ->
                executeDefaultState(targetId,
                        message,
                        byteFunction,
                        deliveryOptions,
                        methodId,
                        vertx,
                        errorMethodHandler,
                        requestMessage,
                        encoder,
                        errorHandler,
                        onFailureRespond,
                        responseDeliveryOptions,
                        retryCount, timeout, delay,
                        circuitBreakerTimeout, executor, retry, lock));
    }

    private static <T> void executeDefaultState(String targetId,
                                                Object message,
                                                ThrowableFunction<AsyncResult<Message<Object>>, T> function,
                                                DeliveryOptions deliveryOptions,
                                                String methodId,
                                                Vertx vertx,
                                                Consumer<Throwable> errorMethodHandler,
                                                Message<Object> requestMessage,
                                                Encoder encoder,
                                                Consumer<Throwable> errorHandler,
                                                ThrowableFunction<Throwable, T> onFailureRespond,
                                                DeliveryOptions responseDeliveryOptions,
                                                int retryCount, long timeout, long delay, long circuitBreakerTimeout,
                                                RecursiveBlockingExecutor executor, RetryBlockingExecutor retry, Lock lock) {

        Optional.ofNullable(lock).ifPresent(Lock::release);
        vertx.
                eventBus().
                send(targetId, message, deliveryOptions,
                        event ->
                                createSupplierAndExecute(targetId,
                                        message,
                                        function,
                                        deliveryOptions,
                                        methodId,
                                        vertx,
                                        errorMethodHandler,
                                        requestMessage,
                                        encoder,
                                        errorHandler,
                                        onFailureRespond,
                                        responseDeliveryOptions,
                                        retryCount, timeout, delay, circuitBreakerTimeout,executor, retry, event));
    }

    private static <T> void executeErrorState(String methodId,
                                              Vertx vertx,
                                              Consumer<Throwable> errorMethodHandler,
                                              Message<Object> requestMessage,
                                              Encoder encoder,
                                              Consumer<Throwable> errorHandler,
                                              ThrowableFunction<Throwable, T> onFailureRespond,
                                              DeliveryOptions responseDeliveryOptions,
                                              int retryCount, long timeout, long delay, long circuitBreakerTimeout,
                                              RecursiveBlockingExecutor executor, Lock lock) {
        final Throwable cause = Future.failedFuture("circuit open").cause();
        handleError(methodId,
                vertx,
                errorMethodHandler,
                requestMessage,
                encoder,
                errorHandler,
                onFailureRespond,
                responseDeliveryOptions,
                retryCount, timeout, delay,
                circuitBreakerTimeout, executor, lock, cause);
    }

    private static <T> void createSupplierAndExecute(String targetId,
                                                     Object message,
                                                     ThrowableFunction<AsyncResult<Message<Object>>, T> function,
                                                     DeliveryOptions deliveryOptions,
                                                     String methodId,
                                                     Vertx vertx,
                                                     Consumer<Throwable> errorMethodHandler,
                                                     Message<Object> requestMessage,
                                                     Encoder encoder,
                                                     Consumer<Throwable> errorHandler,
                                                     ThrowableFunction<Throwable, T> onFailureRespond,
                                                     DeliveryOptions responseDeliveryOptions,
                                                     int retryCount, long timeout, long delay, long circuitBreakerTimeout,
                                                     RecursiveBlockingExecutor executor, RetryBlockingExecutor retry,
                                                     AsyncResult<Message<Object>> event) {
        final ThrowableSupplier<T> supplier = createSupplier(targetId,
                message,
                function,
                deliveryOptions,
                methodId,
                vertx,
                errorMethodHandler,
                requestMessage,
                encoder,
                errorHandler,
                onFailureRespond,
                responseDeliveryOptions,
                retryCount, timeout, delay, circuitBreakerTimeout, retry, event);

        if (circuitBreakerTimeout == NO_TIMEOUT) {
            statelessExecution(targetId,
                    message,
                    function,
                    deliveryOptions,
                    methodId,
                    vertx,
                    errorMethodHandler,
                    requestMessage,
                    encoder,
                    errorHandler,
                    onFailureRespond,
                    responseDeliveryOptions,
                    retryCount, timeout, delay, circuitBreakerTimeout,executor,retry, event, supplier);
        } else {
            statefulExecution(targetId,
                    message,
                    function,
                    deliveryOptions,
                    methodId,
                    vertx,
                    errorMethodHandler,
                    requestMessage,
                    encoder,
                    errorHandler,
                    onFailureRespond,
                    responseDeliveryOptions,
                    retryCount, timeout, delay, circuitBreakerTimeout,executor,retry, event, supplier);
        }
    }

    private static <T> void statelessExecution(String targetId,
                                               Object message,
                                               ThrowableFunction<AsyncResult<Message<Object>>, T> function,
                                               DeliveryOptions deliveryOptions,
                                               String methodId,
                                               Vertx vertx,
                                               Consumer<Throwable> errorMethodHandler,
                                               Message<Object> requestMessage,
                                               Encoder encoder,
                                               Consumer<Throwable> errorHandler,
                                               ThrowableFunction<Throwable, T> onFailureRespond,
                                               DeliveryOptions responseDeliveryOptions,
                                               int retryCount, long timeout, long delay, long circuitBreakerTimeout,
                                               RecursiveBlockingExecutor executor, RetryBlockingExecutor retry,
                                               AsyncResult<Message<Object>> event, ThrowableSupplier<T> byteSupplier) {
        if (event.succeeded() || (event.failed() && retryCount <= 0)) {
            executor.execute(methodId, vertx, event.cause(), errorMethodHandler, requestMessage, byteSupplier, encoder,
                    errorHandler, onFailureRespond, responseDeliveryOptions, retryCount, timeout, delay, circuitBreakerTimeout);
        } else if (event.failed() && retryCount > 0) {
            retryFunction(targetId,
                    message,
                    function,
                    deliveryOptions,
                    methodId, vertx,
                    event.cause(),
                    errorMethodHandler,
                    requestMessage,encoder,
                    errorHandler, onFailureRespond,
                    responseDeliveryOptions,
                    retryCount, timeout,
                    delay, circuitBreakerTimeout,retry);
        }
    }

    private static <T> void statefulExecution(String targetId,
                                              Object message,
                                              ThrowableFunction<AsyncResult<Message<Object>>, T> function,
                                              DeliveryOptions deliveryOptions,
                                              String methodId,
                                              Vertx vertx,
                                              Consumer<Throwable> errorMethodHandler,
                                              Message<Object> requestMessage,
                                              Encoder encoder,
                                              Consumer<Throwable> errorHandler,
                                              ThrowableFunction<Throwable, T> onFailureRespond,
                                              DeliveryOptions responseDeliveryOptions,
                                              int retryCount, long timeout, long delay, long circuitBreakerTimeout,
                                              RecursiveBlockingExecutor executor, RetryBlockingExecutor retry,
                                              AsyncResult<Message<Object>> event, ThrowableSupplier<T> supplier) {
        if (event.succeeded()) {
            executor.execute(methodId, vertx, event.cause(), errorMethodHandler, requestMessage, supplier, encoder,
                    errorHandler, onFailureRespond, responseDeliveryOptions, retryCount, timeout, delay, circuitBreakerTimeout);
        } else {
            statefulErrorHandling(targetId, message, function, deliveryOptions,
                    methodId, vertx, errorMethodHandler, requestMessage, encoder, errorHandler,
                    onFailureRespond, responseDeliveryOptions, retryCount, timeout, delay, circuitBreakerTimeout,executor,retry, event);
        }
    }

    private static <T> void statefulErrorHandling(String targetId,
                                                  Object message,
                                                  ThrowableFunction<AsyncResult<Message<Object>>, T> function,
                                                  DeliveryOptions deliveryOptions,
                                                  String methodId,
                                                  Vertx vertx,
                                                  Consumer<Throwable> errorMethodHandler,
                                                  Message<Object> requestMessage,
                                                  Encoder encoder,
                                                  Consumer<Throwable> errorHandler,
                                                  ThrowableFunction<Throwable, T> onFailureRespond,
                                                  DeliveryOptions responseDeliveryOptions,
                                                  int retryCount, long timeout, long delay, long circuitBreakerTimeout,
                                                  RecursiveBlockingExecutor executor, RetryBlockingExecutor retry, AsyncResult<Message<Object>> event) {

        executeLocked((lock, counter) ->
                counter.decrementAndGet(valHandler -> {
                    if (valHandler.succeeded()) {
                        long count = valHandler.result();
                        if (count <= 0) {
                            openCircuitAndHandleError(methodId, vertx, errorMethodHandler, requestMessage, encoder, errorHandler,
                                    onFailureRespond, responseDeliveryOptions, retryCount, timeout, delay, circuitBreakerTimeout, executor, event, lock, counter);
                        } else {
                            lock.release();
                            final Throwable cause = event.cause();
                            retryFunction(targetId, message, function, deliveryOptions, methodId, vertx, cause, errorMethodHandler, requestMessage, encoder,errorHandler,
                                    onFailureRespond, responseDeliveryOptions, retryCount, timeout, delay, circuitBreakerTimeout, retry);
                        }
                    } else {
                        final Throwable cause = valHandler.cause();
                        handleError(methodId, vertx, errorMethodHandler, requestMessage, encoder,
                                errorHandler, onFailureRespond, responseDeliveryOptions, retryCount, timeout, delay,
                                circuitBreakerTimeout, executor, lock, cause);
                    }
                }), methodId, vertx, errorMethodHandler, requestMessage, encoder, errorHandler, onFailureRespond, responseDeliveryOptions, retryCount, timeout, delay, circuitBreakerTimeout, executor);


    }

    private static <T> void openCircuitAndHandleError(String methodId,
                                                      Vertx vertx,
                                                      Consumer<Throwable> errorMethodHandler,
                                                      Message<Object> requestMessage,
                                                      Encoder encoder,
                                                      Consumer<Throwable> errorHandler,
                                                      ThrowableFunction<Throwable, T> onFailureRespond,
                                                      DeliveryOptions responseDeliveryOptions,
                                                      int retryCount, long timeout, long delay, long circuitBreakerTimeout,
                                                      RecursiveBlockingExecutor executor,
                                                      AsyncResult<Message<Object>> event, Lock lock, Counter counter) {
        resetLockTimer(vertx, retryCount, circuitBreakerTimeout, counter);
        lockAndHandle(counter, val -> {
            final Throwable cause = event.cause();
            handleError(methodId,
                    vertx,
                    errorMethodHandler,
                    requestMessage,
                    encoder,
                    errorHandler,
                    onFailureRespond,
                    responseDeliveryOptions,
                    retryCount,
                    timeout, delay,
                    circuitBreakerTimeout, executor,
                    lock, cause);
        });
    }

    private static void lockAndHandle(Counter counter, Handler<AsyncResult<Long>> asyncResultHandler) {
        counter.addAndGet(LOCK_VALUE, asyncResultHandler);
    }

    private static void resetLockTimer(Vertx vertx, int retryCount, long circuitBreakerTimeout, Counter counter) {
        vertx.setTimer(circuitBreakerTimeout, timer -> counter.addAndGet(Integer.valueOf(retryCount + 1).longValue(), val -> {
        }));
    }

    private static <T> void handleError(String methodId,
                                        Vertx vertx,
                                        Consumer<Throwable> errorMethodHandler,
                                        Message<Object> requestMessage,
                                        Encoder encoder,
                                        Consumer<Throwable> errorHandler,
                                        ThrowableFunction<Throwable, T> onFailureRespond,
                                        DeliveryOptions responseDeliveryOptions,
                                        int retryCount, long timeout, long delay, long circuitBreakerTimeout, RecursiveBlockingExecutor executor, Lock lock, Throwable cause) {
        Optional.ofNullable(lock).ifPresent(Lock::release);
        final ThrowableSupplier<T> failConsumer = () -> {
            assert cause != null;
            throw cause;
        };
        executor.execute(methodId, vertx, cause, errorMethodHandler, requestMessage, failConsumer, encoder,
                errorHandler, onFailureRespond, responseDeliveryOptions, retryCount, timeout, delay, circuitBreakerTimeout);
    }

    private static <T> void retryFunction(String targetId,
                                          Object message,
                                          ThrowableFunction<AsyncResult<Message<Object>>, T> function,
                                          DeliveryOptions requestDeliveryOptions,
                                          String methodId,
                                          Vertx vertx,
                                          Throwable failure,
                                          Consumer<Throwable> errorMethodHandler,
                                          Message<Object> requestMessage,
                                          Encoder encoder,
                                          Consumer<Throwable> errorHandler,
                                          ThrowableFunction<Throwable, T> onFailureRespond,
                                          DeliveryOptions responseDeliveryOptions,
                                          int retryCount, long timeout, long delay,
                                          long circuitBreakerTimeout, RetryBlockingExecutor retry) {
        ResponseExecution.handleError(errorHandler, failure);
        retry.execute(methodId,
                targetId,
                message,
                function,
                requestDeliveryOptions,
                vertx,
                failure,
                errorMethodHandler,
                requestMessage,
                null,
                encoder,
                errorHandler,
                onFailureRespond,
                responseDeliveryOptions,
                retryCount,
                timeout, delay, circuitBreakerTimeout);
    }


    private static <T> ThrowableSupplier<T> createSupplier(String targetId,
                                                           Object message,
                                                           ThrowableFunction<AsyncResult<Message<Object>>, T> function,
                                                           DeliveryOptions deliveryOptions,
                                                           String methodId,
                                                           Vertx vertx,
                                                           Consumer<Throwable> errorMethodHandler,
                                                           Message<Object> requestMessage,
                                                           Encoder encoder,
                                                           Consumer<Throwable> errorHandler,
                                                           ThrowableFunction<Throwable, T> onFailureRespond,
                                                           DeliveryOptions responseDeliveryOptions,
                                                           int retryCount, long timeout, long delay, long circuitBreakerTimeout,
                                                           RetryBlockingExecutor retry, AsyncResult<Message<Object>> event) {
        return () -> {
            T resp = null;
            if (event.failed()) {
                if (retryCount > 0) {
                    retryFunction(targetId,
                            message,
                            function,
                            deliveryOptions,
                            methodId, vertx,
                            event.cause(),
                            errorMethodHandler,
                            requestMessage,
                            encoder, errorHandler,
                            onFailureRespond,
                            responseDeliveryOptions,
                            retryCount, timeout, delay,
                            circuitBreakerTimeout, retry);
                } else {
                    throw event.cause();
                }
            } else {
                resp = function.apply(event);
            }

            return resp;
        };
    }

    private static <T> void executeLocked(LockedConsumer consumer,
                                          String methodId,
                                          Vertx vertx,
                                          Consumer<Throwable> errorMethodHandler,
                                          Message<Object> requestMessage,
                                          Encoder encoder,
                                          Consumer<Throwable> errorHandler,
                                          ThrowableFunction<Throwable, T> onFailureRespond,
                                          DeliveryOptions responseDeliveryOptions,
                                          int retryCount, long timeout, long delay,
                                          long circuitBreakerTimeout, RecursiveBlockingExecutor executor) {
        final SharedData sharedData = vertx.sharedData();
        sharedData.getLockWithTimeout(methodId, DEFAULT_LOCK_TIMEOUT, lockHandler -> {
            if (lockHandler.succeeded()) {
                final Lock lock = lockHandler.result();
                sharedData.getCounter(methodId, resultHandler -> {
                    if (resultHandler.succeeded()) {
                        consumer.execute(lock, resultHandler.result());
                    } else {
                        final Throwable cause = resultHandler.cause();
                        handleError(methodId,
                                vertx,
                                errorMethodHandler,
                                requestMessage, encoder,
                                errorHandler,
                                onFailureRespond,
                                responseDeliveryOptions,
                                retryCount, timeout, delay,
                                circuitBreakerTimeout,
                                executor, lock, cause);
                    }
                });
            } else {
                final Throwable cause = lockHandler.cause();
                handleError(methodId,
                        vertx,
                        errorMethodHandler,
                        requestMessage, encoder,
                        errorHandler,
                        onFailureRespond,
                        responseDeliveryOptions,
                        retryCount, timeout, delay,
                        circuitBreakerTimeout, executor,
                        null, cause);
            }

        });
    }


    private interface LockedConsumer {
        void execute(Lock lock, Counter counter);
    }


}
