/*
 *
 *
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "[]"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright [2017] [Andy Moncsek]
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package org.jacpfx.vertx.event.response.blocking;

import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.Vertx;
import io.vertx.core.shareddata.Counter;
import io.vertx.core.shareddata.Lock;
import io.vertx.core.shareddata.SharedData;
import org.jacpfx.common.ExecutionResult;
import org.jacpfx.common.throwable.ThrowableFunction;
import org.jacpfx.common.throwable.ThrowableSupplier;
import org.jacpfx.vertx.event.response.basic.ResponseExecution;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;

/**
 * Created by Andy Moncsek on 19.01.16.
 * Performs blocking Executions and prepares response
 */
public class ResponseBlockingExecution {

    private static final int DEFAULT_VALUE = 0;
    private static final long DEFAULT_LONG_VALUE = 0;
    private static final int DEFAULT_LOCK_TIMEOUT = 2000;
    private static final int STOP_CONDITION = -1;
    private static final long LOCK_VALUE = -1l;

    /**
     * Executes the response creation and handles failures
     *
     * @param methodId              the method name/id to be executed
     * @param supplier              the user defined supply method to be executed (mapToStringResponse, mapToByteResponse, mapToObjectResponse)
     * @param blockingHandler       the result handler, that takes the result
     * @param errorHandler          the intermediate error method, executed on each error
     * @param onFailureRespond      the method to be executed on failure
     * @param errorMethodHandler    the fallback method
     * @param vertx                 the vertx instance
     * @param failure               last thrown Exception
     * @param retry                 the amount of retries
     * @param timeout               the max timeout time for the method execution
     * @param circuitBreakerTimeout the stateful circuit breaker release time
     * @param delay                 the delay time between retry
     * @param <T>                   the type of response (String, byte, Object)
     */
    public static <T> void createResponseBlocking(String methodId,
                                                  ThrowableSupplier<T> supplier,
                                                  Future<ExecutionResult<T>> blockingHandler,
                                                  Consumer<Throwable> errorHandler,
                                                  ThrowableFunction<Throwable, T> onFailureRespond,
                                                  Consumer<Throwable> errorMethodHandler,
                                                  Vertx vertx,
                                                  Throwable failure,
                                                  int retry,
                                                  long timeout,
                                                  long circuitBreakerTimeout,
                                                  long delay) {
        if (circuitBreakerTimeout > DEFAULT_LONG_VALUE) {
            executeLocked((lock, counter) -> counter.get(counterHandler -> {
                        long currentVal = counterHandler.result();
                        if (currentVal == DEFAULT_LONG_VALUE) {
                            executeInitialState(methodId,
                                    supplier,
                                    blockingHandler,
                                    errorHandler,
                                    onFailureRespond,
                                    errorMethodHandler,
                                    vertx,
                                    failure,
                                    retry,
                                    timeout,
                                    circuitBreakerTimeout,
                                    delay,
                                    lock,
                                    counter);
                        } else if (currentVal > DEFAULT_LONG_VALUE) {
                            executeDefault(methodId,
                                    supplier,
                                    blockingHandler,
                                    errorHandler,
                                    onFailureRespond,
                                    errorMethodHandler,
                                    vertx,
                                    failure,
                                    retry,
                                    timeout,
                                    circuitBreakerTimeout,
                                    delay,
                                    lock);
                        } else {
                            executeErrorState(blockingHandler,
                                    errorHandler,
                                    onFailureRespond,
                                    errorMethodHandler,
                                    failure,
                                    lock);
                        }
                    }), methodId,
                    vertx,
                    blockingHandler,
                    errorHandler,
                    onFailureRespond,
                    errorMethodHandler,
                    null);
        } else {
            executeStateless(supplier,
                    blockingHandler,
                    errorHandler,
                    onFailureRespond,
                    errorMethodHandler,
                    vertx,
                    retry,
                    timeout,
                    delay);
        }
    }

    private static <T> void executeErrorState(Future<ExecutionResult<T>> _blockingHandler,
                                              Consumer<Throwable> _errorHandler,
                                              ThrowableFunction<Throwable, T> _onFailureRespond,
                                              Consumer<Throwable> _errorMethodHandler,
                                              Throwable failure,
                                              Lock lock) {
        Optional.ofNullable(lock).ifPresent(Lock::release);
        handleErrorExecution(_blockingHandler,
                _errorHandler,
                _onFailureRespond,
                _errorMethodHandler,
                Optional.ofNullable(failure).
                        orElse(Future.failedFuture("circuit open").cause()));
    }

    private static <T> void executeDefault(String _methodId,
                                           ThrowableSupplier<T> _supplier,
                                           Future<ExecutionResult<T>> _blockingHandler,
                                           Consumer<Throwable> _errorHandler,
                                           ThrowableFunction<Throwable, T> _onFailureRespond,
                                           Consumer<Throwable> _errorMethodHandler,
                                           Vertx vertx,
                                           Throwable _failure,
                                           int _retry,
                                           long _timeout,
                                           long _circuitBreakerTimeout,
                                           long _delay,
                                           Lock lock) {
        Optional.ofNullable(lock).ifPresent(Lock::release);
        vertx.executeBlocking(bhandler -> {
            try {
                executeDefaultState(_supplier, _blockingHandler, vertx, _timeout);
                bhandler.complete();
            } catch (Throwable e) {
                executeLocked((lck, counter) ->
                                counter.decrementAndGet(valHandler -> {
                                    if (valHandler.succeeded()) {
                                        handleStatefulError(_methodId,
                                                _supplier,
                                                _blockingHandler,
                                                _errorHandler,
                                                _onFailureRespond,
                                                _errorMethodHandler,
                                                vertx,
                                                _failure,
                                                _retry,
                                                _timeout,
                                                _circuitBreakerTimeout,
                                                _delay,
                                                e,
                                                lck,
                                                counter,
                                                valHandler);
                                        bhandler.complete();
                                    } else {
                                        releaseLockAndHandleError(_blockingHandler,
                                                _errorHandler,
                                                _onFailureRespond,
                                                _errorMethodHandler,
                                                valHandler.cause(),
                                                lck);
                                        bhandler.complete();
                                    }
                                }), _methodId,
                        vertx,
                        _blockingHandler,
                        _errorHandler,
                        _onFailureRespond,
                        _errorMethodHandler,
                        bhandler);
            }
        }, false, res -> {

        });
    }


    private static <T> void executeInitialState(String _methodId,
                                                ThrowableSupplier<T> _supplier,
                                                Future<ExecutionResult<T>> _blockingHandler,
                                                Consumer<Throwable> _errorHandler,
                                                ThrowableFunction<Throwable, T> _onFailureRespond,
                                                Consumer<Throwable> _errorMethodHandler,
                                                Vertx vertx,
                                                Throwable _t,
                                                int _retry,
                                                long _timeout,
                                                long _circuitBreakerTimeout,
                                                long _delay,
                                                Lock lock,
                                                Counter counter) {
        final long initialRetryCounterValue = (long) (_retry + 1);
        counter.addAndGet(initialRetryCounterValue, rHandler -> executeDefault(_methodId,
                _supplier,
                _blockingHandler,
                _errorHandler,
                _onFailureRespond,
                _errorMethodHandler,
                vertx,
                _t,
                _retry,
                _timeout,
                _circuitBreakerTimeout,
                _delay,
                lock));
    }

    private static <T> void releaseLockAndHandleError(Future<ExecutionResult<T>> _blockingHandler,
                                                      Consumer<Throwable> _errorHandler,
                                                      ThrowableFunction<Throwable, T> _onFailureRespond,
                                                      Consumer<Throwable> _errorMethodHandler,
                                                      Throwable cause,
                                                      Lock lock) {
        Optional.ofNullable(lock).ifPresent(Lock::release);
        handleErrorExecution(_blockingHandler, _errorHandler, _onFailureRespond, _errorMethodHandler, cause);
    }

    private static <T> void handleErrorExecution(Future<ExecutionResult<T>> _blockingHandler,
                                                 Consumer<Throwable> _errorHandler,
                                                 ThrowableFunction<Throwable, T> _onFailureRespond,
                                                 Consumer<Throwable> _errorMethodHandler,
                                                 Throwable cause) {
        T result = null;
        result = handleError(result, _errorHandler, _onFailureRespond, _errorMethodHandler, cause);
        if (!_blockingHandler.isComplete()) _blockingHandler.complete(new ExecutionResult<>(result, true, true, null));
    }

    private static <T> void handleStatefulError(String _methodId, ThrowableSupplier<T> _supplier,
                                                Future<ExecutionResult<T>> _blockingHandler,
                                                Consumer<Throwable> _errorHandler,
                                                ThrowableFunction<Throwable, T> _onFailureRespond,
                                                Consumer<Throwable> _errorMethodHandler,
                                                Vertx vertx,
                                                Throwable _t,
                                                int _retry,
                                                long _timeout,
                                                long _circuitBreakerTimeout,
                                                long _delay,
                                                Throwable e,
                                                Lock lck,
                                                Counter counter,
                                                AsyncResult<Long> valHandler) {
        //////////////////////////////////////////
        long count = valHandler.result();
        if (count <= DEFAULT_VALUE) {
            setCircuitBreakerReleaseTimer(vertx, _retry, _circuitBreakerTimeout, counter);
            openCircuitBreakerAndHandleError(_blockingHandler, _errorHandler, _onFailureRespond, _errorMethodHandler, vertx, e, lck, counter);
        } else {
            lck.release();
            ResponseExecution.handleError(_errorHandler, e);
            handleDelay(_delay);
            createResponseBlocking(_methodId, _supplier, _blockingHandler, _errorHandler,
                    _onFailureRespond, _errorMethodHandler, vertx, _t, _retry, _timeout, _circuitBreakerTimeout, _delay);
        }
        ////////////////////////////////////////
    }

    private static <T> void openCircuitBreakerAndHandleError(Future<ExecutionResult<T>> _blockingHandler,
                                                             Consumer<Throwable> _errorHandler,
                                                             ThrowableFunction<Throwable, T> _onFailureRespond,
                                                             Consumer<Throwable> _errorMethodHandler,
                                                             Vertx vertx,
                                                             Throwable e,
                                                             Lock lck,
                                                             Counter counter) {
        counter.addAndGet(LOCK_VALUE, val -> {
            lck.release();
            vertx.executeBlocking(bhandler -> {
                T result = null;
                result = handleError(result, _errorHandler, _onFailureRespond, _errorMethodHandler, e);
                if (!_blockingHandler.isComplete())
                    _blockingHandler.complete(new ExecutionResult<>(result, true, true, null));
            }, false, res -> {

            });
        });
    }

    private static void setCircuitBreakerReleaseTimer(Vertx vertx, int _retry, long _circuitBreakerTimeout, Counter counter) {
        // TODO should the counter executed with lock?
        final long initialRetryCounterValue = (long) (_retry + 1);
        vertx.setTimer(_circuitBreakerTimeout, timer -> counter.addAndGet(initialRetryCounterValue, val -> {
        }));
    }

    private static <T> void executeDefaultState(ThrowableSupplier<T> _supplier,
                                                Future<ExecutionResult<T>> _blockingHandler,
                                                Vertx vertx,
                                                long _timeout) throws Throwable {
        T result;
        if (_timeout > DEFAULT_LONG_VALUE) {
            result = executeWithTimeout(_supplier, vertx, _timeout);
        } else {
            result = _supplier.get();
        }
        if (!_blockingHandler.isComplete()) _blockingHandler.complete(new ExecutionResult<>(result, true, false, null));
    }

    private static <T> T executeWithTimeout(ThrowableSupplier<T> _supplier, Vertx vertx, long _timeout) throws Throwable {
        T result;
        final CompletableFuture<T> timeoutFuture = new CompletableFuture<>();
        vertx.executeBlocking((innerHandler) -> {
            try {
                timeoutFuture.complete(_supplier.get());
            } catch (Throwable throwable) {
                timeoutFuture.obtrudeException(throwable);
            }
        }, false, (val) -> {
        });

        try {
            result = timeoutFuture.get(_timeout, TimeUnit.MILLISECONDS);
        } catch (TimeoutException timeout) {
            throw new TimeoutException("operation _timeout");
        }

        return result;
    }

    private static <T> void executeStateless(ThrowableSupplier<T> _supplier,
                                             Future<ExecutionResult<T>> _blockingHandler,
                                             Consumer<Throwable> errorHandler,
                                             ThrowableFunction<Throwable, T> onFailureRespond,
                                             Consumer<Throwable> errorMethodHandler,
                                             Vertx vertx,
                                             int _retry,
                                             long timeout,
                                             long delay) {
        T result = null;
        boolean errorHandling = false;
        while (_retry >= DEFAULT_VALUE) {
            errorHandling = false;
            try {
                if (timeout > DEFAULT_LONG_VALUE) {
                    result = executeWithTimeout(_supplier, vertx, timeout);
                    _retry = STOP_CONDITION;
                } else {
                    result = _supplier.get();
                    _retry = STOP_CONDITION;
                }

            } catch (Throwable e) {
                _retry--;
                if (_retry < DEFAULT_VALUE) {
                    try {
                        result = handleError(result, errorHandler, onFailureRespond, errorMethodHandler, e);
                        errorHandling = true;
                    } catch (Exception ee) {
                        _blockingHandler.fail(ee);
                    }

                } else {
                    ResponseExecution.handleError(errorHandler, e);
                    handleDelay(delay);
                }
            }
        }
        if (!errorHandling || errorHandling && result != null) {
            if (!_blockingHandler.isComplete())
                _blockingHandler.complete(new ExecutionResult<>(result, true, errorHandling, null));
        }
    }


    private static void handleDelay(long delay) {
        try {
            if (delay > DEFAULT_LONG_VALUE) Thread.sleep(delay);
        } catch (InterruptedException e1) {
            e1.printStackTrace();
        }
    }


    private static <T> T handleError(T result, Consumer<Throwable> errorHandler,
                                     ThrowableFunction<Throwable, T> onFailureRespond,
                                     Consumer<Throwable> errorMethodHandler,
                                     Throwable e) {
        try {
            if (errorHandler != null) {
                errorHandler.accept(e);
            }
            if (onFailureRespond != null) {
                result = onFailureRespond.apply(e);
            }
            if (errorHandler == null && onFailureRespond == null) {
                errorMethodHandler.accept(e); // TODO switch to function to return true if an error method was executed, no if no error method is available
                return null;

            }
        } catch (Throwable throwable) {
            errorMethodHandler.accept(throwable);
        }
        return result;
    }

    private static <T, U> void executeLocked(LockedConsumer consumer,
                                             String _methodId,
                                             Vertx vertx,
                                             Future<ExecutionResult<T>> _blockingHandler,
                                             Consumer<Throwable> _errorHandler,
                                             ThrowableFunction<Throwable, T> _onFailureRespond,
                                             Consumer<Throwable> _errorMethodHandler,
                                             Future<U> blockingCodeHandler) {
        final SharedData sharedData = vertx.sharedData();
        sharedData.getLockWithTimeout(_methodId, DEFAULT_LOCK_TIMEOUT, lockHandler -> {
            final Lock lock = lockHandler.result();
            if (lockHandler.succeeded()) {
                sharedData.getCounter(_methodId, resultHandler -> {
                    if (resultHandler.succeeded()) {
                        consumer.execute(lock, resultHandler.result());
                    } else {
                        releaseLockAndHandleError(_blockingHandler, _errorHandler, _onFailureRespond, _errorMethodHandler, resultHandler.cause(), lock);
                        Optional.ofNullable(blockingCodeHandler).ifPresent(Future::complete);
                    }
                });
            } else {
                handleErrorExecution(_blockingHandler, _errorHandler, _onFailureRespond, _errorMethodHandler, lockHandler.cause());
                Optional.ofNullable(blockingCodeHandler).ifPresent(Future::complete);
            }

        });
    }


    private interface LockedConsumer {
        void execute(Lock lock, Counter counter);
    }
}
