/*
 * Copyright [2018] [Andy Moncsek]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jacpfx.vxms.event.util;

import io.vertx.core.AsyncResult;
import io.vertx.core.eventbus.DeliveryOptions;
import io.vertx.core.eventbus.Message;
import java.io.Serializable;
import java.util.Optional;
import java.util.function.Consumer;
import org.jacpfx.vxms.common.VxmsShared;
import org.jacpfx.vxms.common.encoder.Encoder;
import org.jacpfx.vxms.common.throwable.ThrowableFunction;
import org.jacpfx.vxms.common.throwable.ThrowableSupplier;
import org.jacpfx.vxms.event.eventbus.blocking.EventbusBridgeExecution;
import org.jacpfx.vxms.event.interfaces.blocking.ExecuteEventbusObjectCall;
import org.jacpfx.vxms.event.interfaces.blocking.RecursiveExecutor;
import org.jacpfx.vxms.event.interfaces.blocking.RetryExecutor;
import org.jacpfx.vxms.event.response.blocking.ExecuteEventbusObjectResponse;

/**
 * Created by Andy Moncsek on 05.04.16. Typed execution of event-bus calls and blocking object
 * response
 */
public class EventbusObjectExecutionBlockingUtil {

  /**
   * create execution chain for event-bus request and reply to the event
   *
   * @param _methodId the method identifier
   * @param _targetId the event-bus target id
   * @param _message the message to send
   * @param _objectFunction the function to process the result message
   * @param _requestDeliveryOptions the event-bus delivery serverOptions
   * @param _vxmsShared the vxmsShared instance, containing the Vertx instance and other shared
   *     objects per instance
   * @param _failure the failure thrown while task execution
   * @param _errorMethodHandler the error-method handler
   * @param _requestMessage the event-bus request serverOptions
   * @param _objectSupplier the supplier, producing the object response
   * @param _encoder the encoder to serialize the result object
   * @param _errorHandler the error handler
   * @param _onFailureRespond the consumer that takes a Future with the alternate response value in
   *     case of failure
   * @param _responseDeliveryOptions the delivery serverOptions
   * @param _retryCount the amount of retries before failure execution is triggered
   * @param _timeout the amount of time before the execution will be aborted
   * @param _delay the delay between an error and the retry
   * @param _circuitBreakerTimeout the amount of time before the circuit breaker closed again
   * @return the execution chain {@link ExecuteEventbusObjectResponse}
   */
  public static ExecuteEventbusObjectResponse mapToObjectResponse(
      String _methodId,
      String _targetId,
      Object _message,
      ThrowableFunction<AsyncResult<Message<Object>>, Serializable> _objectFunction,
      DeliveryOptions _requestDeliveryOptions,
      VxmsShared _vxmsShared,
      Throwable _failure,
      Consumer<Throwable> _errorMethodHandler,
      Message<Object> _requestMessage,
      ThrowableSupplier<Serializable> _objectSupplier,
      Encoder _encoder,
      Consumer<Throwable> _errorHandler,
      ThrowableFunction<Throwable, Serializable> _onFailureRespond,
      DeliveryOptions _responseDeliveryOptions,
      int _retryCount,
      long _timeout,
      long _delay,
      long _circuitBreakerTimeout) {
    final DeliveryOptions deliveryOptions =
        Optional.ofNullable(_requestDeliveryOptions).orElse(new DeliveryOptions());
    final RetryExecutor<Serializable> retry =
        (methodId,
            targetId,
            message,
            function,
            requestDeliveryOptions,
            vxmsShared,
            t,
            errorMethodHandler,
            requestMessage,
            supplier,
            encoder,
            errorHandler,
            onFailureRespond,
            responseDeliveryOptions,
            retryCount,
            timeout,
            delay,
            circuitBreakerTimeout) -> {
          int retryValue = retryCount - 1;
          mapToObjectResponse(
                  methodId,
                  targetId,
                  message,
                  function,
                  requestDeliveryOptions,
                  vxmsShared,
                  t,
                  errorMethodHandler,
                  requestMessage,
                  null,
                  encoder,
                  errorHandler,
                  onFailureRespond,
                  responseDeliveryOptions,
                  retryValue,
                  timeout,
                  delay,
                  circuitBreakerTimeout)
              .execute();
        };

    final RecursiveExecutor<Serializable> executor =
        (methodId,
            vxmsShared,
            t,
            errorMethodHandler,
            requestMessage,
            supplier,
            encoder,
            errorHandler,
            onFailureRespond,
            responseDeliveryOptions,
            retryCount,
            timeout,
            delay,
            circuitBreakerTimeout) ->
            new ExecuteEventbusObjectResponse(
                    methodId,
                    vxmsShared,
                    t,
                    errorMethodHandler,
                    requestMessage,
                    null,
                    supplier,
                    null,
                    encoder,
                    errorHandler,
                    onFailureRespond,
                    responseDeliveryOptions,
                    retryCount,
                    timeout,
                    delay,
                    circuitBreakerTimeout)
                .execute();

    final ExecuteEventbusObjectCall excecuteEventBusAndReply =
        (methodId,
            vxmsShared,
            errorMethodHandler,
            requestMessage,
            encoder,
            errorHandler,
            errorHandlerObject,
            responseDeliveryOptions,
            retryCount,
            timeout,
            delay,
            circuitBreakerTimeout) ->
            EventbusBridgeExecution.sendMessageAndSupplyHandler(
                methodId,
                _targetId,
                _message,
                _objectFunction,
                deliveryOptions,
                vxmsShared,
                errorMethodHandler,
                requestMessage,
                encoder,
                errorHandler,
                errorHandlerObject,
                responseDeliveryOptions,
                retryCount,
                timeout,
                delay,
                circuitBreakerTimeout,
                executor,
                retry);

    return new ExecuteEventbusObjectResponse(
        _methodId,
        _vxmsShared,
        _failure,
        _errorMethodHandler,
        _requestMessage,
        null,
        _objectSupplier,
        excecuteEventBusAndReply,
        _encoder,
        _errorHandler,
        _onFailureRespond,
        _responseDeliveryOptions,
        _retryCount,
        _timeout,
        _delay,
        _circuitBreakerTimeout);
  }
}
