/**
 * Licensed to Jasig under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Jasig licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a
 * copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.jasig.portlet.notice.service;

import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.IOException;

import javax.portlet.PortletRequest;

import org.jasig.portlet.notice.response.NotificationResponse;
import org.jasig.portlet.notice.service.exceptions.NotificationServiceException;

import net.sf.json.JSON;
import net.sf.json.JSONException;
import net.sf.json.JSONSerializer;

/**
 * This is a simple demo service provider. It reads data from
 * a file and returns it.
 */
public class DemoNotificationService extends AbstractNotificationService {

	private String demoFilename = "DemoNoticationResponse.dat";
	
	/**
	 * Returns the name of the service.
	 * @return String.
	 */
	public String getName()
	{
		return "DemoService";
	}

	/**
	 * Set the filename of the demo data.
	 * @param filename is the demo filename.
	 */
	public void setFilename(String filename)
	{
		demoFilename = filename;
	}
    
    /**
     * Retrieves all available service requests for requester
     * 
     * @param partyNumber
     * @param username
     * @return List of service requests or an empty list
     */
    public NotificationResponse fetchNotificationsFromSource(PortletRequest req)
    throws NotificationServiceException
    {
    	return readFromFile(demoFilename);
    }

	/**
	 * Serialize the given instance to JSON data and write it to a file.
	 *
	 * @param request is the NotificationRequest instance to be serialized to a file.
	 * @param filename is the path and name of the file to be written.
	 * @return boolean, false if the data write fails.
	 */
	public boolean writeToFile(NotificationResponse request, String filename)
	{
		try
		{
			JSON json = JSONSerializer.toJSON(request.toMap());
			String data = json.toString(1);
			
			FileOutputStream fos = new FileOutputStream(filename);
			fos.write(data.getBytes());
			fos.close();
			return true;
		}
		catch(JSONException je)
		{
			je.printStackTrace();
			return false;
		}
		catch(IOException ioe)
		{
			ioe.printStackTrace();
			return false;
		}
	}

	/**
	 * De-serialize the given JSON formatted file back into a object.
	 *
	 * @param filename is the path and name of the file to be read.
	 * @return NotificationRequest, null if the de-serialization fails.
	 */
	public NotificationResponse readFromFile(String filename)
	{
		try
		{
			InputStream fis = getClass().getClassLoader().getResourceAsStream(filename);
			int available = fis.available();
			byte[] bytes = new byte[available];
			fis.read(bytes);
			fis.close();

			return NotificationResponse.fromJson(new String(bytes));
		}
		catch(JSONException je)
		{
			je.printStackTrace();
			return null;
		}
		catch(IOException ioe)
		{
			ioe.printStackTrace();
			return null;
		}
	}
}
