/**
 * Licensed to Jasig under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Jasig licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a
 * copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jasig.portlet.emailpreview.dao.demo;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import javax.portlet.PortletRequest;
import javax.portlet.PortletSession;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jasig.portlet.emailpreview.AccountSummary;
import org.jasig.portlet.emailpreview.EmailMessage;
import org.jasig.portlet.emailpreview.EmailMessageContent;
import org.jasig.portlet.emailpreview.dao.IEmailAccountService;
import org.springframework.stereotype.Component;

/**
 * A Demo implementation of {@link IEmailAccountService}.
 *
 * @author Landis Casner
 * @author Drew Wills, drew@unicon.net
 */
@Component
public final class DemoAccountService implements IEmailAccountService {

    private final Log log = LogFactory.getLog(getClass());

    private static final String ACCOUNT_SUMMARY_KEY = "DemoAccountService.ACCOUNT_SUMMARY_KEY";
    private static final String INBOX_URL = "http://www.jasig.org/";
    private static final int DEFAULT_BATCH_SIZE = 20;

    private String jsonLocation = "/SampleJSON.json";

    /*
     * Public API
     */

    public void setJsonLocation(String jsonLocation) {
        this.jsonLocation = jsonLocation;
    }

    public AccountSummary getAccountSummary(PortletRequest req, int start,
            int max, boolean refresh) {

        // Try PortletSession first
        PortletSession session = req.getPortletSession();
        AccountSummary rslt = (AccountSummary) session.getAttribute(ACCOUNT_SUMMARY_KEY);

        if (rslt == null) {
            List<EmailMessage> messages = getEmailMessages(req);
            rslt = new AccountSummary(INBOX_URL, messages, getUnreadMessageCount(messages),
                                                messages.size(), start, max, true);
            req.getPortletSession().setAttribute(ACCOUNT_SUMMARY_KEY, rslt);
        }

        return rslt;

    }

    public EmailMessage getMessage(PortletRequest req, int messageNum) {

        AccountSummary summary = (AccountSummary) req.getPortletSession().getAttribute(ACCOUNT_SUMMARY_KEY);
        if (summary == null) {
            // Probably shouldn't happen...
            summary = getAccountSummary(req, 0, DEFAULT_BATCH_SIZE, false);
        }

        List<EmailMessage> messages = summary.getMessages();
        if (!(messages.size() > messageNum)) {
            throw new RuntimeException("No such message:  " + messageNum);
        }
        return messages.get(messageNum);

    }

    public boolean deleteMessages(PortletRequest req, long[] uids) {

        List<Long> excluded = Arrays.asList(ArrayUtils.toObject(uids));

        PortletSession session = req.getPortletSession(true);
        AccountSummary summary = (AccountSummary) session.getAttribute(ACCOUNT_SUMMARY_KEY);
        if (summary == null) {
            // Probably shouldn't happen...
            summary = getAccountSummary(req, 0, DEFAULT_BATCH_SIZE, false);
        }
        List<EmailMessage> messages = summary.getMessages();
        List<EmailMessage> newList = new ArrayList<EmailMessage>();

        for (EmailMessage m : messages) {
            if (!excluded.contains(m.getUid())) {
                newList.add(m);
            }
        }

        session.setAttribute(ACCOUNT_SUMMARY_KEY, new AccountSummary(INBOX_URL,
                newList, getUnreadMessageCount(newList), newList.size(),
                0, DEFAULT_BATCH_SIZE, true));

        return true;  // Indicate success

    }

    public boolean setSeenFlag(PortletRequest req, long[] uids, boolean seenValue) {

        List<Long> changed = Arrays.asList(ArrayUtils.toObject(uids));

        PortletSession session = req.getPortletSession(true);
        AccountSummary summary = (AccountSummary) session.getAttribute(ACCOUNT_SUMMARY_KEY);
        if (summary == null) {
            // Probably shouldn't happen...
            summary = getAccountSummary(req, 0, DEFAULT_BATCH_SIZE, false);
        }
        List<EmailMessage> messages = summary.getMessages();
        List<EmailMessage> newList = new ArrayList<EmailMessage>();

        for (EmailMessage m : messages) {
            EmailMessage msg = !changed.contains(m.getUid()) ? m
                    : new EmailMessage(newList.size(), m.getUid(), m.getSender(), m.getSubject(),
                        m.getSentDate(), !seenValue, m.isAnswered(), m.isDeleted(),
                        m.isMultipart(), m.getContentType(), m.getContent());
            newList.add(msg);
        }

        session.setAttribute(ACCOUNT_SUMMARY_KEY, new AccountSummary(INBOX_URL,
                newList, getUnreadMessageCount(newList), newList.size(),
                0, DEFAULT_BATCH_SIZE, true));

        return true;  // success

    }

    /**
     * Builds a fresh collection of messages.
     */
    private List<EmailMessage> getEmailMessages(PortletRequest req) {

        File jsonFile = new File(getClass().getResource(jsonLocation).getFile());
        List<EmailMessage> messages = new ArrayList<EmailMessage>();

        InputStream inpt = null;
        try {

            inpt = new FileInputStream(jsonFile);

            JSONArray returnedFiles = JSONArray.fromObject(IOUtils.toString(inpt));

            // Creates a Mime Message because Email Message depends on a "message" variable.
            for (int i=0; i < returnedFiles.size(); i++) {

                JSONObject msg = returnedFiles.getJSONObject(i);

                long uid = msg.getLong("uid");
                String sender = msg.getString("from");
                String subject = msg.getString("subject");
                Date sentDate = new Date(msg.getLong("sentDate"));
                boolean unread = msg.getBoolean("unread");
                boolean answered = false; // didn't consider to change this
                boolean deleted = false; // more testing is available here
                EmailMessageContent content = new EmailMessageContent(msg.getString("body"), true);

                messages.add(new EmailMessage(messages.size(), uid,
                        sender, subject, sentDate, unread, answered, deleted,
                        false, "text/plain", content));

            }
        } catch (Exception e) {
            log.error("Failed to load messages collection", e);
        } finally {
            try {
                inpt.close();
            } catch (IOException ioe) {
                log.error("Failed to close input stream", ioe);
            }
        }

        return messages;

    }

    /**
     * replacement for 'inbox.getUnreadMessageCount()
     */
    private int getUnreadMessageCount(List<EmailMessage> messages)
    {
        int unreadCount = 0;

        for (EmailMessage email : messages)
        {
            if (email.isUnread())
            {
                unreadCount++;
            }
        }

        return unreadCount;
    }

}
