/**
 * Licensed to Jasig under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Jasig licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a
 * copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.jasig.portlet.emailpreview.security;

import org.jasig.portlet.emailpreview.EmailPreviewException;
import org.jasypt.encryption.pbe.PBEStringEncryptor;
import org.jasypt.exceptions.EncryptionInitializationException;

/**
 * JasyptPBEStringEncryptionServiceImpl is an implementation of 
 * IStringEncryptionService that uses a configurable Jasypt PBEStringEncryptor
 * to perform string encryption and decryption.
 * 
 * @author Jen Bourey
 */
public class JasyptPBEStringEncryptionServiceImpl implements IStringEncryptionService {

	PBEStringEncryptor encryptor = null;
	
	/**
	 * Set the PBEStringEncryptor to be used
	 * 
	 * @param encryptor
	 */
	public void setStringEncryptor(PBEStringEncryptor encryptor) {
		this.encryptor = encryptor;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public String encrypt(String plaintext) {
        try {
		    return this.encryptor.encrypt(plaintext);
        } catch (EncryptionInitializationException e) {
            throw new EmailPreviewException("Encryption error. Verify an encryption password"
                    + " is configured in the email preview portlet's"
                    + " stringEncryptionService bean in applicationContent.xml", e);
        }
	}
	
	/**
	 * {@inheritDoc}
	 */
	public String decrypt(String cryptotet) {
        try {
		    return this.encryptor.decrypt(cryptotet);
        } catch (EncryptionInitializationException e) {
            throw new EmailPreviewException("Decryption error. Was encryption password"
                    + " changed in the email preview portlet's"
                    + " stringEncryptionService bean in applicationContent.xml?", e);
        }
    }
	
}
