/**
 * Licensed to Apereo under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Apereo licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License.  You may obtain a
 * copy of the License at the following location:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jasig.portlet.notice;

import java.io.Serializable;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;

import org.codehaus.jackson.map.annotate.JsonSerialize;

/**
 * This class is used report errors when a INotificationService
 * tries to retrieve notifications but is unable to due to
 * some error.
 */
@JsonSerialize(include=JsonSerialize.Inclusion.NON_EMPTY)
@XmlAccessorType(XmlAccessType.FIELD)
public class NotificationError implements Serializable, Cloneable {
    private static final long serialVersionUID = 1L;

    private String error;
    private String source;
    private int key = -1;  // Generated by error & source

    public NotificationError() {}

    public NotificationError(String error, String source) {
        this.error = error;
        this.source = source;
        updateKey(source, error);
    }

    public String getSource() {
        return source;
    }

    public void setSource(String source) {
        this.source = source;
        updateKey(source, error);
    }

    public String getError() {
        return error;
    }

    public void setError(String error) {
        this.error = error;
        updateKey(source, error);
    }

    public int getKey() {
        return key;
    }

    private void updateKey(String source, String error) {
        this.key = new String(source + ":" + error).hashCode();
    }

    /**
     * Implements deep-copy clone.
     * 
     * @throws CloneNotSupportedException Not really, but it's on the method 
     * signature we're overriding.
     */
    @Override
    public Object clone() throws CloneNotSupportedException {

        // Start with superclass impl (handles immutables and primitives)
        final NotificationError rslt = (NotificationError) super.clone();

        // Adjust to satisfy deep-copy strategy

        /*
         * NB:  Nothing to do here...
         */

        return rslt;

    }


    public boolean equals(Object object) {
        if(object instanceof NotificationError) {
            NotificationError temp = (NotificationError)object;
            return (temp.getKey() == this.getKey());
        }
        return false;
    }

    @Override
    /**
     * Returns a string representation of this class' data.
     * 
     * @return String.
     */
    public String toString() {
        return "org.jasig.portlet.notice.serverresponse.NotificationError\n"
                + "\tSource            = " + source  + "\n"
                + "\tError             = " + error  + "\n"
                + "\tKey               = " + key  + "\n";
    }

}
