/*
 * Copyright(c) Brigham Young University, 2008.  All rights reserved.
 * (See licensing and redistribution disclosures in licensing project directory.)
 * 
 */
package org.jasig.portlet.search.ldap;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.naming.directory.SearchControls;

import org.springframework.ldap.core.LdapTemplate;
import org.springframework.ldap.filter.AndFilter;
import org.springframework.ldap.filter.EqualsFilter;
import org.springframework.ldap.filter.WhitespaceWildcardsFilter;

public class LdapSearchServiceImpl {

	private String queryAttribute = "cn";
	private LdapTemplate ldapTemplate;
	private PersonAttributesMapper personAttributesMapper;
	private List<String> guestAttributes;
	private List<String> loggedInAttributes;
	private int returnedLimit = 10;
	private String searchBase = "";

	/**
	 * Search the configured LDAP directory with the specified search term
	 * 
	 * @param searchterm	user's search term
	 * @param loggedIn		<code>true</code> for logged in users
	 * @return				List of matching Person objects
	 */
	public List<Person> searchDirectory(String searchterm, boolean loggedIn) {
		
		// create a list of permitted LDAP attributes to be returned
		List<String> allowedAttributes = null;
		if (loggedIn)
			allowedAttributes = this.loggedInAttributes;
		else
			allowedAttributes = this.guestAttributes;
		Set<String> allowedMappedAttributes = new HashSet<String>();
		Map<String,String> attrMap = personAttributesMapper.getAttributesMap();
		for (String attr : allowedAttributes) {
			allowedMappedAttributes.add(attrMap.get(attr));
		}
		allowedMappedAttributes.add(personAttributesMapper.getFullNameAttribute());

		// construct a new search controls object for our search
		SearchControls searchControls = new SearchControls();
		searchControls.setCountLimit(returnedLimit);
		searchControls.setReturningAttributes(allowedMappedAttributes.toArray(new String[]{}));
		searchControls.setSearchScope(SearchControls.SUBTREE_SCOPE);

		// create a filter to search for a person matching
		// the search term the user entered
		AndFilter filter = new AndFilter();
		filter.and(new EqualsFilter("objectclass", "person"));
		filter.and(new WhitespaceWildcardsFilter(this.queryAttribute, searchterm));
				
		// perform the search
		@SuppressWarnings("unchecked")
		List<Person> people = (List<Person>) ldapTemplate.search(searchBase, filter
					.encode(), searchControls, personAttributesMapper);
		
		return people;
	}

	/**
	 * Set the LDAP attribute to run queries against.  This query 
	 * is generally set to the full name attribute.
	 * 
	 * @param queryAttribute
	 */
	public void setQueryAttribute(String queryAttribute) {
		this.queryAttribute = queryAttribute;
	}

	public void setLdapTemplate(LdapTemplate ldapTemplate) {
		this.ldapTemplate = ldapTemplate;
	}

	public void setPersonAttributesMapper(
			PersonAttributesMapper personAttributesMapper) {
		this.personAttributesMapper = personAttributesMapper;
	}

	/**
	 * Set the list of allowed LDAP attributes to display to guest users.  The
	 * values in this list should match keys in the PersonAttributeMapper's
	 * attribute map.
	 * 
	 * @param guestAttributes
	 */
	public void setGuestAttributes(List<String> guestAttributes) {
		this.guestAttributes = guestAttributes;
	}

	/**
	 * Set the list of allowed LDAP attributes to display to logged in users.  The
	 * values in this list should match keys in the PersonAttributeMapper's
	 * attribute map.
	 * 
	 * @param loggedInAttributes
	 */
	public void setLoggedInAttributes(List<String> loggedInAttributes) {
		this.loggedInAttributes = loggedInAttributes;
	}

	/**
	 * Set the limit for the number of Person objects that should be returned
	 * from an LDAP search.
	 * 
	 * @param returnedLimit
	 */
	public void setReturnedLimit(int returnedLimit) {
		this.returnedLimit = returnedLimit;
	}
	
	/**
	 * Set the search base for LDAP directory search.  If this is not required, 
	 * it may be left as an empty string.
	 * 
	 * @param searchBase
	 */
	public void setSearchBase(String searchBase) {
		this.searchBase = searchBase;
	}
	
}
