/*
 * Copyright(c) Brigham Young University, 2008.  All rights reserved.
 * (See licensing and redistribution disclosures in licensing project directory.)
 * 
 */
package org.jasig.portlet.search.ldap;

import java.util.LinkedHashMap;

import javax.naming.NamingException;
import javax.naming.directory.Attributes;

import org.springframework.ldap.core.AttributesMapper;

/**
 * PersonAttributesMapper constructs a Person object from an Attributes
 * object returned from LDAP.  The mapper requires an attribute name
 * that represents the LDAP attribute for a full display name, as well
 * as a map of attribute names and keys.  In this provided map, keys should
 * equal more consistent, useful names for Person attributes, while the values
 * should be LDAP attribute names.
 * 
 * @author Jen Bourey
 * @version $Revision: 44239 $
 */
public class PersonAttributesMapper implements AttributesMapper {

	private final String fullNameAttribute;
	private final LinkedHashMap<String,String> attributesMap;
	
	/**
	 * Construct a new PersonAttributesMapper
	 * 
	 * @param fullNameAttribute		key for the full name attribute
	 * @param attributesMap			map of key/value attribute pairs
	 */
	public PersonAttributesMapper(String fullNameAttribute, LinkedHashMap<String,String> attributesMap) {
		// set the full name for this Person
		this.fullNameAttribute = fullNameAttribute;
		// set the attributes for this Person
		this.attributesMap = attributesMap;
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.springframework.ldap.core.AttributesMapper#mapFromAttributes(javax.naming.directory.Attributes)
	 */
	public Object mapFromAttributes(Attributes attrs)
			throws NamingException {
		Person person = new Person();
		if (attrs.get(this.fullNameAttribute) != null)
			person.setFullName((String) attrs.get(this.fullNameAttribute).get());
		for (String key : attributesMap.keySet()) {
			String value = attributesMap.get(key);
			if (attrs.get(value) != null)
				person.put(key, (String) attrs.get(value).get());
		}
		return person;
	}

	/**
	 * Get the key that maps the full name attribute
	 * 
	 * @return
	 */
	public String getFullNameAttribute() {
		return fullNameAttribute;
	}

	/**
	 * Return the map of attribute key/value pairs.  
	 * 
	 * @return
	 */
	public LinkedHashMap<String, String> getAttributesMap() {
		return attributesMap;
	}

}
