/*
 * Copyright(c) Yale University, 2008.  All rights reserved.
 * (See licensing and redistribution disclosures in licensing project directory.)
 * 
 */
package org.jasig.portlet.search.mvc;

import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.View;
import org.springframework.web.servlet.mvc.AbstractController;

/**
 * GsaSearchController performs searches against a Google Search Appliance.
 * 
 * @author Jen Bourey
 * @version $Revision: 44240 $
 */
public class GsaSearchController extends AbstractController {

	private View view;
	private static Log log = LogFactory.getLog(GsaSearchController.class);
	public String googleBaseUrl;
	public String googleStylesheetName;
	public String queryParameterName = "q";
	public String siteParameterName = "site";
	public String numberLimitParameterName = "num";
	public String googleSearchServlet = "search";
	public Map<String,String> googleParameters = new HashMap<String,String>();
	
	/*
	 * (non-Javadoc)
	 * @see org.springframework.web.servlet.mvc.AbstractController#handleRequestInternal(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	@Override
	protected ModelAndView handleRequestInternal(HttpServletRequest request,
			HttpServletResponse response) throws Exception {
		Map<String,Object> model = new HashMap<String,Object>();
		
		// collect the user's submitted form data
		String query = request.getParameter("q");
		String site = request.getParameter("site");
		String num = request.getParameter("num");
		
		/*
		 * construct the google query URL
		 */

		StringBuffer googleUrl = new StringBuffer();
		googleUrl.append(googleBaseUrl);
		googleUrl.append("/");
		googleUrl.append(googleSearchServlet);
		
		// add the search query
		addParameter(googleUrl, queryParameterName, query, true);
		if (site != null && !site.equals(""))
			addParameter(googleUrl, siteParameterName, site, false);
		
		// add the limit for the number of results to be returned
		addParameter(googleUrl, numberLimitParameterName, num, false);
		
		// add each configured parameter
		for (String paramName : googleParameters.keySet()) {
			addParameter(googleUrl, paramName, googleParameters.get(paramName), false);
		}
		
		String url = googleUrl.toString();
		log.debug("Searching GSA with URL " + url);

		// retrieve an input stream for the above URL and construct 
		// a Source from it
		final HttpClient client = new HttpClient();
		final GetMethod get = new GetMethod(url);
		int rc = client.executeMethod(get);
		if (rc != HttpStatus.SC_OK) {
			log.warn("HttpStatus for " + url + ":" + rc);
		}
		final InputStream is = get.getResponseBodyAsStream();
		Source source = new StreamSource(is);
		
		// add the source to the model and return our view
		model.put("source", source);
		model.put("googleStylesheet", googleStylesheetName);
		model.put("googleBaseUrl", googleBaseUrl);
		return new ModelAndView(view, "model", model);
	}
	
	private void addParameter(StringBuffer url, String paramName, String paramValue, boolean firstParam) throws UnsupportedEncodingException {
		if (firstParam)
			url.append("?");
		else
			url.append("&");
		url.append(paramName);
		url.append("=");
		url.append(URLEncoder.encode(paramValue, "UTF-8"));
	}
	
	/**
	 * Set the View for the GSA search controller.
	 * 
	 * @param view
	 */
	public void setView(View view) {
		this.view = view;
	}

	/**
	 * Set the base URL for Google searches.  This URL should include the
	 * protocol and server.  Example: http://google.school.edu
	 * 
	 * @param googleBaseUrl
	 */
	public void setGoogleBaseUrl(String googleBaseUrl) {
		this.googleBaseUrl = googleBaseUrl;
	}

	/**
	 * Set the name of the google stylesheet to use in the "more results"
	 * link.
	 * 
	 * @param googleStylesheetName
	 */
	public void setGoogleStylesheetName(String googleStylesheetName) {
		this.googleStylesheetName = googleStylesheetName;
	}

	/**
	 * Set the name of the query parameter.  "q" by default
	 * 
	 * @param queryParameterName
	 */
	public void setQueryParameterName(String queryParameterName) {
		this.queryParameterName = queryParameterName;
	}

	/**
	 * Set the name of the site parameter. "site" by default
	 * 
	 * @param siteParameterName
	 */
	public void setSiteParameterName(String siteParameterName) {
		this.siteParameterName = siteParameterName;
	}

	/**
	 * Set the name of the number limit parameter.  "num" by default
	 * 
	 * @param numberLimitParameterName
	 */
	public void setNumberLimitParameterName(String numberLimitParameterName) {
		this.numberLimitParameterName = numberLimitParameterName;
	}

	/**
	 * Set a map of parameter names and values for google searches.
	 * These parameters may include instructions regarding encoding, 
	 * client, etc.
	 * 
	 * @param googleParameters
	 */
	public void setGoogleParameters(Map<String, String> googleParameters) {
		this.googleParameters = googleParameters;
	}

	
}
