/**
 * Licensed to Jasig under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Jasig licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a
 * copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jasig.portlet.search.ldap;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.naming.directory.SearchControls;
import javax.portlet.PortletRequest;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.ldap.core.LdapTemplate;
import org.springframework.ldap.filter.AndFilter;
import org.springframework.ldap.filter.EqualsFilter;
import org.springframework.ldap.filter.WhitespaceWildcardsFilter;

public class LdapSearchServiceImpl {

	private String queryAttribute = "cn";
	private LdapTemplate ldapTemplate;
	private PersonAttributesMapper personAttributesMapper;
	private List<PersonAttribute> attributes;
	private Map<String,String> attributeTypeMap;
	private int returnedLimit = 10;
	private String searchBase = "";

	/**
	 * Search the configured LDAP directory with the specified search term
	 * 
	 * @param searchterm	user's search term
	 * @param loggedIn		<code>true</code> for logged in users
	 * @return				List of matching Person objects
	 */
	public List<Person> searchDirectory(String searchterm, PortletRequest request) {
		
		// create a list of permitted LDAP attributes to be returned
		List<String> allowedAttributes = getAllowedAttributes(request);
		allowedAttributes.add(personAttributesMapper.getFullNameAttribute());

		// construct a new search controls object for our search
		SearchControls searchControls = new SearchControls();
		searchControls.setCountLimit(returnedLimit);
		searchControls.setReturningAttributes(allowedAttributes.toArray(new String[]{}));
		searchControls.setSearchScope(SearchControls.SUBTREE_SCOPE);

		// create a filter to search for a person matching
		// the search term the user entered
		AndFilter filter = new AndFilter();
		filter.and(new EqualsFilter("objectclass", "person"));
		filter.and(new WhitespaceWildcardsFilter(this.queryAttribute, searchterm));
				
		// perform the search
		@SuppressWarnings("unchecked")
		List<Person> people = (List<Person>) ldapTemplate.search(searchBase, filter
					.encode(), searchControls, personAttributesMapper);
		
		return people;
	}
	
	public Map<String,String> getAttributeTypeMap() {
		return attributeTypeMap;
	}
	
	protected List<String> getAllowedAttributes(PortletRequest request) {
		List<String> allowed = new ArrayList<String>();
		Map<String,Boolean> roles = new HashMap<String,Boolean>();
		for (PersonAttribute attribute : attributes) {
			for (String role : attribute.getRoles()) {
				if (roles.containsKey(role)) {
					if (roles.get(role)) {
						allowed.add(attribute.getKey());
					}
					allowed.add(attribute.getKey());
					break;
				} else if (request.isUserInRole(role)) {
					allowed.add(attribute.getKey());
					roles.put(role, true);
					break;
				} else {
					roles.put(role, false);
				}
			}
		}
		allowed.add(personAttributesMapper.getFullNameAttribute());
		return allowed;
	}

	/**
	 * Set the LDAP attribute to run queries against.  This query 
	 * is generally set to the full name attribute.
	 * 
	 * @param queryAttribute
	 */
	@Required
	public void setQueryAttribute(String queryAttribute) {
		this.queryAttribute = queryAttribute;
	}

	@Autowired(required = true)
	public void setLdapTemplate(LdapTemplate ldapTemplate) {
		this.ldapTemplate = ldapTemplate;
	}

	@Autowired(required = true)
	public void setPersonAttributesMapper(
			PersonAttributesMapper personAttributesMapper) {
		this.personAttributesMapper = personAttributesMapper;
		if (!attributeTypeMap.containsKey(this.personAttributesMapper.getFullNameAttribute())) {
			attributeTypeMap.put(this.personAttributesMapper.getFullNameAttribute(), "none");
		}
	}
	
	@Required
	@Resource(name = "directoryAttributes")
	public void setAttributes(List<PersonAttribute> attributes) {
		this.attributes = attributes;
		
		// initialize the key -> type mapping
		attributeTypeMap = new HashMap<String,String>();
		for (PersonAttribute attr : attributes) {
			attributeTypeMap.put(attr.getKey(), attr.getDisplay());
		}
		if (this.personAttributesMapper != null && this.personAttributesMapper.getFullNameAttribute() != null) {
			attributeTypeMap.put(this.personAttributesMapper.getFullNameAttribute(), "none");
		}
	}

	/**
	 * Set the limit for the number of Person objects that should be returned
	 * from an LDAP search.
	 * 
	 * @param returnedLimit
	 */
	public void setReturnedLimit(int returnedLimit) {
		this.returnedLimit = returnedLimit;
	}
	
	/**
	 * Set the search base for LDAP directory search.  If this is not required	 
	 * by your LDAP server it may be left as an empty string.
	 * 
	 * @param searchBase
	 */
	public void setSearchBase(String searchBase) {
		this.searchBase = searchBase;
	}
	
}
