/**
 * Licensed to Jasig under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Jasig licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a
 * copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jasig.portlet.search.ldap;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;

import org.springframework.ldap.core.AttributesMapper;

/**
 * PersonAttributesMapper constructs a Person object from an Attributes
 * object returned from LDAP.  The mapper requires an attribute name
 * that represents the LDAP attribute for a full display name, as well
 * as a map of attribute names and keys.  In this provided map, keys should
 * equal more consistent, useful names for Person attributes, while the values
 * should be LDAP attribute names.
 * 
 * @author Jen Bourey
 * @version $Revision: 46837 $
 */
public class PersonAttributesMapper implements AttributesMapper {

	private final String fullNameAttribute;
	
	/**
	 * Construct a new PersonAttributesMapper
	 * 
	 * @param fullNameAttribute		key for the full name attribute
	 * @param attributesMap			map of key/value attribute pairs
	 */
	public PersonAttributesMapper(String fullNameAttribute) {
		// set the full name for this Person
		this.fullNameAttribute = fullNameAttribute;
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.springframework.ldap.core.AttributesMapper#mapFromAttributes(javax.naming.directory.Attributes)
	 */
	public Object mapFromAttributes(Attributes attrs)
			throws NamingException {
		Person person = new Person();
		if (attrs.get(this.fullNameAttribute) != null)
			person.setFullName((String) attrs.get(this.fullNameAttribute).get());
		
		NamingEnumeration<? extends Attribute> attributes = attrs.getAll();
		while (attributes.hasMoreElements()) {
			Attribute attr = attributes.next();
			if (attr.get() != null) {
				person.put(attr.getID(), String.valueOf(attr.get()));
			}
		}
		return person;
	}

	/**
	 * Get the key that maps the full name attribute
	 * 
	 * @return
	 */
	public String getFullNameAttribute() {
		return fullNameAttribute;
	}

}
