/**
 * Licensed to Jasig under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Jasig licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a
 * copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jasig.portlet.search.mvc;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.View;

/**
 * GsaSearchController performs searches against a Google Search Appliance.
 * 
 * @author Jen Bourey
 * @version $Revision: 47825 $
 */
@Controller
@RequestMapping("/gsaSearch")
public class GsaSearchController {

	private static Log log = LogFactory.getLog(GsaSearchController.class);

	private static final String QUERY_PARAMETER_NAME = "q";
	private static final String SITE_PARAMETER_NAME = "site";
	private static final String NUMBER_LIMIT_PARAMETER_NAME = "num";

	@RequestMapping(method = RequestMethod.GET)
	public ModelAndView searchGsa(HttpServletRequest request) throws HttpException, IOException {
		
		// collect the user's submitted form data
		String query = request.getParameter("q");
		String site = request.getParameter("site");
		String num = request.getParameter("num");
		
		/*
		 * construct the google query URL
		 */

		String url = buildGoogleSearchUrl(query, site, num);
		log.debug("Searching GSA with URL " + url);

		// retrieve an input stream for the above URL and construct 
		// a Source from it
		final HttpClient client = new HttpClient();
		final GetMethod get = new GetMethod(url);
		int rc = client.executeMethod(get);
		if (rc != HttpStatus.SC_OK) {
			log.warn("HttpStatus for " + url + ":" + rc);
		}
		final InputStream is = get.getResponseBodyAsStream();
		
		// add the source to the model and return our view
		return new ModelAndView(view, "source", is);
	}
	
	protected String buildGoogleSearchUrl(String query, String site, String num) throws UnsupportedEncodingException {
		StringBuffer googleUrl = new StringBuffer();
		googleUrl.append(googleBaseUrl);
		googleUrl.append("/");
		googleUrl.append(googleSearchServlet);
		
		// add the search query
		addParameter(googleUrl, QUERY_PARAMETER_NAME, query, true);
		if (site != null && !site.equals(""))
			addParameter(googleUrl, SITE_PARAMETER_NAME, site, false);
		
		// add the limit for the number of results to be returned
		addParameter(googleUrl, NUMBER_LIMIT_PARAMETER_NAME, num, false);
		
		// add each configured parameter
		for (String paramName : googleParameters.keySet()) {
			addParameter(googleUrl, paramName, googleParameters.get(paramName), false);
		}
		
		return googleUrl.toString();
	}
	
	protected void addParameter(StringBuffer url, String paramName, String paramValue, boolean firstParam) throws UnsupportedEncodingException {
		if (firstParam)
			url.append("?");
		else
			url.append("&");
		url.append(paramName);
		url.append("=");
		url.append(URLEncoder.encode(paramValue, "UTF-8"));
	}

	
	private View view;
	
	/**
	 * Set the View for the GSA search controller.
	 * 
	 * @param view
	 */
	public void setView(View view) {
		this.view = view;
	}
	
	public String googleSearchServlet = "search";

	public String googleBaseUrl;
	
	/**
	 * Set the base URL for Google searches.  This URL should include the
	 * protocol and server.  Example: http://google.school.edu
	 * 
	 * @param googleBaseUrl
	 */
	public void setGoogleBaseUrl(String googleBaseUrl) {
		this.googleBaseUrl = googleBaseUrl;
	}

	public String googleStylesheetName;
	/**
	 * Set the name of the google stylesheet to use in the "more results"
	 * link.
	 * 
	 * @param googleStylesheetName
	 */
	public void setGoogleStylesheetName(String googleStylesheetName) {
		this.googleStylesheetName = googleStylesheetName;
	}

	public Map<String,String> googleParameters = new HashMap<String,String>();
	
	/**
	 * Set a map of parameter names and values for google searches.
	 * These parameters may include instructions regarding encoding, 
	 * client, etc.
	 * 
	 * @param googleParameters
	 */
	@Resource(name="gsaParameters")
	public void setGoogleParameters(Map<String, String> googleParameters) {
		this.googleParameters = googleParameters;
	}

	
}
