/**
 * Licensed to Jasig under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Jasig licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a
 * copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jasig.portlet.search.mvc;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.portlet.PortletSession;
import javax.portlet.RenderRequest;

import org.springframework.beans.factory.annotation.Required;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.portlet.ModelAndView;

/**
 * TabbedSearchController is the main portlet controller for the tabbed search
 * portlet.
 * 
 * @author Jen Bourey
 */
@Controller
@RequestMapping("VIEW")
public class TabbedSearchController {

	@RequestMapping()
	public ModelAndView getSearchView(RenderRequest request) {

		Map<String, Object> model = new HashMap<String, Object>();
		
		// if the user is currently logged in, save a logged in flag in the session
		PortletSession session = request.getPortletSession();
		if (request.getRemoteUser() != null)
			session.setAttribute("loggedIn", true, PortletSession.APPLICATION_SCOPE);
		
		List<ConfiguredSearch> allowedSearches = new ArrayList<ConfiguredSearch>();
		for (ConfiguredSearch search : this.configuredSearches) {
			for (String role : search.getRoles()) {
				if (request.isUserInRole(role)) {
					allowedSearches.add(search);
					break;
				}
			}
		}
		session.setAttribute("allowedSearches", allowedSearches, PortletSession.APPLICATION_SCOPE);
		
		model.put("configuredSearches", allowedSearches);
		model.put("gsaSearchEngines", this.gsaSearchEngines);
		
		return new ModelAndView("/index", "model", model);
	}

	private List<String> gsaSearchEngines;
	
	/**
	 * Set the list of GSA search sites.
	 * 
	 * @param gsaSearchEngines
	 */
	@Resource(name="gsaSearchEngines")
	public void setGsaSearchEngines(List<String> gsaSearchEngines) {
		this.gsaSearchEngines = gsaSearchEngines;
	}
	
	private List<ConfiguredSearch> configuredSearches;
	
	/**
	 * Set the list of configured search engines.  Search tabs will be displayed
	 * in the order represented in this list.
	 * 
	 * @param configuredSearches
	 */
	@Resource(name="configuredSearches")
	@Required
	public void setConfiguredSearches(List<ConfiguredSearch> configuredSearches) {
		this.configuredSearches = configuredSearches;
	}
	
	
}
