/* 
 * Copyright 2010-2018 Jan de Jongh <jfcmdejongh@gmail.com>, TNO.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package org.javades.jqueues.r5.util.predictor;

import java.io.PrintStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.NavigableMap;
import java.util.Set;
import java.util.TreeMap;
import org.javades.jqueues.r5.entity.jq.SimJQEvent;
import org.javades.jqueues.r5.entity.jq.job.SimJob;
import org.javades.jqueues.r5.entity.jq.job.visitslogging.JobQueueVisitLog;
import org.javades.jqueues.r5.entity.jq.queue.SimQueue;
import org.javades.jqueues.r5.entity.jq.queue.composite.tandem.Tandem;
import org.javades.jqueues.r5.entity.jq.queue.nonpreemptive.FCFS;
import org.javades.jqueues.r5.entity.jq.queue.nonpreemptive.IC;
import org.javades.jqueues.r5.entity.jq.queue.preemptive.P_LCFS;
import org.javades.jqueues.r5.entity.jq.queue.processorsharing.CUPS;
import org.javades.jqueues.r5.entity.jq.queue.serverless.ZERO;
import org.javades.jsimulation.r5.DefaultSimEventList_IOEL;
import org.javades.jsimulation.r5.DefaultSimEventList_ROEL;
import org.javades.jsimulation.r5.SimEventList;

/** An object capable of predicting the behavior of one or more {@link SimQueue}s under user-supplied workload and conditions.
 *
 * <p>
 * A {@link SimQueuePredictor} is a stateless object that is capable of predicting the behavior of a specific {@link SimQueue}
 * class or (if applicable) of (some of) its subclasses;
 * the (base) class of queues supported is present as generic type argument {@code Q}.
 * Objects like this are extensively used in the test sub-system of {@code jqueues}.
 * 
 * <p>
 * The most important feature of a {@link SimQueuePredictor} is the prediction of job visits to a given (stateless!)
 * {@link SimQueue} under a given external workload (e.g., arrivals, revocations, setting server-access credits,
 * or queue-specific external operations). The workload consists of a collection of {@link SimJQEvent}s, and this collection
 * may contain events scheduled at the same time. Depending on the method invoked on the predictor,
 * such simultaneous events are to be interpreted as occurring in "random order"
 * (as if processed by a ROEL {@link SimEventList} like {@link DefaultSimEventList_ROEL})
 * or as occurring in the strict and deterministic order (somehow) imposed by the collection 
 * (as if processed by a IOEL {@link SimEventList} like {@link DefaultSimEventList_IOEL}).
 * 
 * <p>
 * In the first case,
 * see {@link #predict_SQ_SV_ROEL_U},
 * the workload schedule itself can easily lead to ambiguities that prevent the delivery of a prediction,
 * for instance in the case of simultaneous arrivals (of jobs with non-zero required service time) at a {@link FCFS} queue.
 * On the other hand, queues like {@link ZERO} appear to be robust against simultaneous arrivals under ROEL,
 * and for queues like {@link IC}, simultaneous arrivals under ROEL are unambiguous <i>only</i> if sufficient
 * server-access credits are available, hence, the ambiguity of simultaneous arrivals for this queue is state-dependent.
 * However, for all {@link SimQueue}s, the simultaneous start of a queue-access vacation (again, state-dependent) and
 * an arrival <i>always</i> leads to ambiguities.
 * 
 * <p>
 * In the seconds case,
 * see {@link #predict_SQ_SV_IOEL_U},
 * workload events do not cause ambiguities among themselves, but they may still interfere with queue-internal
 * events like departures, for instance the simultaneous occurrence of an arrival and a scheduled departure in a {@link P_LCFS}
 * queue that is otherwise empty. Even worse, queues may exhibit internal ambiguities, for instance, the simultaneous
 * occurrence of a "catch-up" and a departure (both "internal events") in a {@link CUPS} queue.
 * Note that even with a ROEL {@link SimEventList}, certain {@link SimQueue} implementations
 * may process "simultaneous events" in a specific sequence, and heavily rely on their sequential execution.
 * For instance, the {@link Tandem} lets (delegate) jobs arrive at their first sub-queue if
 * server-access credits become available, yet it processes these arrivals in a specific order (the arrival order of the
 * corresponding "real" jobs) and it effectuates these arrivals immediately, without using the underlying event list.
 * 
 * <p>
 * In any case, implementations must provide a collection of visit logs, see {@link JobQueueVisitLog},
 * or throw an exception upon determining an ambiguity, see {@link SimQueuePredictionAmbiguityException}.
 * 
 * @param <Q> The type of {@link SimQueue}s supported.
 * 
 * @author Jan de Jongh, TNO
 * 
 * <p>
 * Copyright (C) 2005-2017 Jan de Jongh, TNO
 * 
 * <p>
 * This file is covered by the LICENSE file in the root of this project.
 * 
 */
public interface SimQueuePredictor<Q extends SimQueue>
extends SimQueueEventPredictor<Q>, SimQueueStatePredictor<Q>
{

  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // PREDICT (EVERYTHING)
  //
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
  /** Creates the unique prediction, if possible,
   *  resulting from subjecting a given queue to a given workload
   *  under a Random-Order Event List.
   * 
   * @param queue          The queue, non-{@code null}.
   * @param workloadEvents The workload events; events related to other queues are allowed and are to be ignored.
   * 
   * @return The prediction.
   * 
   * @throws IllegalArgumentException      If {@code queue == null} or the workload parameters are somehow illegal.
   * @throws UnsupportedOperationException If the queue type or the workload is (partially) unsupported.
   * @throws SimQueuePredictionException   If a prediction is (e.g.) too complex to generate
   *                                       ({@link SimQueuePredictionComplexityException}),
   *                                       if invalid input has been supplied to the predictor
   *                                       ({@link SimQueuePredictionInvalidInputException}),
   *                                       or if a <i>unique</i> prediction cannot be generated
   *                                       ({@link SimQueuePredictionAmbiguityException}).
   * 
   */
  SimQueuePrediction_SQ_SV<Q>
  predict_SQ_SV_ROEL_U
  (Q queue, Set<SimJQEvent> workloadEvents)
  throws SimQueuePredictionException;
    
  /** Creates the unique prediction, if possible,
   *  resulting from subjecting a given queue to a given workload
   *  under an Insertion-Order Event List.
   * 
   * <p>
   * Note that processed-events map parameter may be equal to the workload events map,
   * in which case processed (internal) events are inserted in situ.
   * If a different map is provided, it is cleared upon entry.
   * 
   * @param queue              The queue, non-{@code null}.
   * @param workloadEventsMap  The workload events as a map from event time onto the
   *                           (ordered!) set of events occurring at that time;
   *                           events related to other queues are allowed and are to be ignored.
   * @param processedEventsMap An optional map in which all events processed at the queue (including workload events)
   *                           are stored unambiguously; the events in a value set are in processing ordered
   *                           (you can use this to resolve ambiguities in the visit logs like equal departure times).
   * 
   * @return The prediction.
   * 
   * @throws IllegalArgumentException      If {@code queue == null} or the workload parameters are somehow illegal.
   * @throws UnsupportedOperationException If the queue type or the workload is (partially) unsupported.
   * @throws SimQueuePredictionException   If a prediction is (e.g.) too complex to generate
   *                                       ({@link SimQueuePredictionComplexityException}),
   *                                       if invalid input has been supplied to the predictor
   *                                       ({@link SimQueuePredictionInvalidInputException}),
   *                                       or if a <i>unique</i> prediction cannot be generated
   *                                       ({@link SimQueuePredictionAmbiguityException}).
   * 
   */
  SimQueuePrediction_SQ_SV<Q>
  predict_SQ_SV_IOEL_U
  (Q queue,
   NavigableMap<Double, Set<SimJQEvent>> workloadEventsMap,
   NavigableMap<Double, Set<SimJQEvent>> processedEventsMap)
  throws SimQueuePredictionException;
  
  /** Compares two maps of predicted and actual {@link JobQueueVisitLog}s for equality, within given accuracy.
   * 
   * <p>
   * The {@code actual} argument holds all (allowing multiple) job visits, and may contain visits to other {@link SimQueue}s;
   * the latter of which are (to be) ignored.
   * The map has the jobs as keys, and each value holds another map from arrival times (of the
   * particular job) to numbered {@link JobQueueVisitLog} of that job at that particular arrival time
   * (this allows multiple arrivals of the same job at the same time).
   * 
   * @param <Q> The (generic) type of the queue.
   * 
   * @param queue      The queue, non-{@code null}.
   * @param predicted  The predicted {@link JobQueueVisitLog}s, indexed by job-arrival time; arrival at other queues
   *                   are (to be) ignored.
   * @param actual     The actual {@link JobQueueVisitLog}s, see above.
   * @param accuracy   The accuracy (maximum  deviation of times in a {@link JobQueueVisitLog}), non-negative.
   * @param stream     An optional stream for mismatch reporting.
   * @param testString An optional String identifying the test in place.
   * 
   * @return Whether the predicted and actual maps map within the given accuracy.
   * 
   * @throws IllegalArgumentException If any of the arguments except the stream has {@code null} value,
   *                                  is illegally structured, or if the
   *                                  accuracy argument is negative.
   * 
   */
  public static <Q extends SimQueue> boolean matchVisitLogs_SQ_SV
    (final Q queue,
      final Map<SimJob, JobQueueVisitLog<SimJob, Q>> predicted,
      final Map<SimJob, TreeMap<Double, TreeMap<Integer, JobQueueVisitLog<SimJob, Q>>>> actual,
      final double accuracy,
      final PrintStream stream,
      final String testString)
  {
    if (queue == null || predicted == null || actual == null || accuracy < 0)
      throw new IllegalArgumentException ();
    final Map<SimJob, JobQueueVisitLog<SimJob, Q>> predictedAtQueue = new HashMap<> ();
    for (final Entry<SimJob, JobQueueVisitLog<SimJob, Q>> entry : predicted.entrySet ())
      if (entry.getValue ().queue == queue)
        predictedAtQueue.put (entry.getKey (), entry.getValue ());
    final Map<SimJob, JobQueueVisitLog<SimJob, Q>> actualAtQueue = new HashMap<> ();
    boolean success = true;
    for (final Entry<SimJob, TreeMap<Double, TreeMap<Integer, JobQueueVisitLog<SimJob, Q>>>> entry : actual.entrySet ())
    {
      if (entry == null)
        throw new IllegalArgumentException ();
      final SimJob job = entry.getKey ();
      for (final Entry<Double, TreeMap<Integer, JobQueueVisitLog<SimJob, Q>>> timeEntry : entry.getValue ().entrySet ())
      {
        if (timeEntry == null)
          throw new IllegalArgumentException ();
        for (final Entry<Integer, JobQueueVisitLog<SimJob, Q>> sequenceEntry : timeEntry.getValue ().entrySet ())
          if (sequenceEntry.getValue ().queue == queue)
          {
            if (actualAtQueue.containsKey (job))
            {
              success = false;
              if (stream != null)
              {
                stream.println ("[matchVisitLogs_SQ_SV:] Found multiple visits of job " + job + " to queue " + queue +".");
                if (testString != null)
                {
                  stream.println ("  Test: ");
                  stream.println (testString);
                }
              }
              else
                return false;
            }
            else
              actualAtQueue.put (job, sequenceEntry.getValue ());
          }
      }
    }
    for (final SimJob job : predictedAtQueue.keySet ())
    {
      final JobQueueVisitLog<SimJob, Q> predictedVisitLog = predictedAtQueue.get (job);
      if (! actualAtQueue.containsKey (job))
      {
        success = false;
        if (stream != null)
        {
          stream.println ("[matchVisitLogs_SQ_SV] Absent predicted visit of job " + job + " to queue " + queue +":");
          if (testString != null)
          {
            stream.println ("  Test: ");
            stream.println (testString);
          }
          stream.println ("Predicted visit log: ");
          predictedVisitLog.print (stream);
        }
        else
          return false;        
      }
      final JobQueueVisitLog<SimJob, Q> actualVisitLog = actualAtQueue.get (job);
      if (actualVisitLog != null && ! actualVisitLog.equals (predictedVisitLog, accuracy))
      {
        success = false;
        if (stream != null)
        {
          stream.println ("[matchVisitLogs_SQ_SV: Found mismatch for visit of job " + job + " to queue " + queue +":");
          if (testString != null)
          {
            stream.println ("  Test: ");
            stream.println (testString);
          }
          stream.println ("Accuracy = " + accuracy + ".");
          stream.println ("Predicted and actual visit logs: ");
          predictedVisitLog.print (stream);
          actualVisitLog.print (stream);
        }
        else
          return false;
      }
    }
    return success;
  }
    
}
