/* 
 * Copyright 2010-2018 Jan de Jongh <jfcmdejongh@gmail.com>, TNO.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package org.javades.jqueues.r5.util.stat;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.javades.jqueues.r5.entity.jq.queue.SimQueue;

/** A description of an entry for automatic statistics management.
 * 
 * @param <Q> The type of {@link SimQueue}s supported.
 * 
 * @author Jan de Jongh, TNO
 * 
 * <p>
 * Copyright (C) 2005-2017 Jan de Jongh, TNO
 * 
 * <p>
 * This file is covered by the LICENSE file in the root of this project.
 * 
 */
public class AutoSimQueueStatEntry<Q extends SimQueue>
{

  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // LOGGER
  //
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
  private static final Logger LOGGER = Logger.getLogger (AutoSimQueueStatEntry.class.getName ());
  
  /** Compile-time switch for logging.
   * 
   * <p>
   * Logging this class may lead to excessive object creation.
   * 
   */
  private static final boolean LOGGING = false;
  
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // CONSTRUCTOR(S)
  //
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
  public AutoSimQueueStatEntry (final String name, final SimQueueProbe<Q> probe)
  {
    if (name == null || probe == null)
      throw new IllegalArgumentException ();
    this.name = name;
    this.probe = probe;
    resetInt ();
  }
  
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // NAME
  //
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
  private final String name;
  
  public final String getName ()
  {
    return this.name;
  }
  
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // PROBE
  //
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
  private final SimQueueProbe<Q> probe;
  
  public final SimQueueProbe<Q> getSimQueueProbe ()
  {
    return this.probe;
  }

  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // NUMBER OF PROBES
  //
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
  private int numberOfProbes = 0;

  public final int getNumberOfProbes ()
  {
    return this.numberOfProbes;
  }
  
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // CUMULATIVE VALUE
  //
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
  private double cumValue = 0.0;

  public final double getCumValue ()
  {
    return this.cumValue;
  }
  
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // AVERAGE VALUE (IF CALCULATED)
  //
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
  private double avgValue = 0.0;
  
  public final double getAvgValue ()
  {
    return this.avgValue;
  }
  
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // MINIMUM VALUE
  //
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
  private double minValue = Double.NaN;
  
  public final double getMinValue ()
  {
    return this.minValue;
  }
  
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // MAXIMUM VALUE
  //
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
  private double maxValue = Double.NaN;
  
  public final double getMaxValue ()
  {
    return this.maxValue;
  }
  
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // RESET
  //
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
  private void resetInt ()
  {
    this.numberOfProbes = 0;
    this.cumValue = 0.0;
    this.avgValue = 0.0;
    this.minValue = Double.NaN;
    this.maxValue = Double.NaN;
  }
  
  public void reset ()
  {
    resetInt ();
  }
  
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // UPDATE
  //
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
  public void update (final Q queue, final double dt)
  {
    if (queue == null)
      LOGGER.log (Level.WARNING, "Updating {0} for queue=null, dt={1}.", new Object[]{this.name, dt});
    else
    {
      final double probeValue = this.probe.get (queue);
      if (LOGGING)
        LOGGER.log (Level.FINEST, "Updating {0} for queue={1}, value={2}, dt={3}.",
                    new Object[]{this.name, queue, probeValue, dt});
      this.cumValue += probeValue * dt;
      if (this.numberOfProbes == 0)
      {
        this.minValue = probeValue;
        this.maxValue = probeValue;
      }
      else
      {
        this.minValue = Math.min (this.minValue, probeValue);
        this.maxValue = Math.max (this.maxValue, probeValue);        
      }
      this.numberOfProbes++;
    }
  }
  
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // CALCULATE
  //
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
  public void calculate (final double startTime, final double endTime)
  {
    if (endTime < startTime)
      throw new IllegalArgumentException ();
    if (startTime == endTime)
      this.avgValue = 0.0;
    else
    {
      final double dT = endTime - startTime;
      this.avgValue = this.cumValue / dT;
    }
    if (LOGGING)
      LOGGER.log (Level.FINER,
                  "Calculating {0} with dT={1}, cumValue={2}: avgValue={3}, minValue={4}, maxValue={5}.",
                  new Object[]{this.name, endTime - startTime, this.cumValue, this.avgValue, this.minValue, this.maxValue});
  }

  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  // END OF FILE
  //
  //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
}
