/*
 * Decompiled with CFR 0.152.
 */
package org.jbake.app;

import com.orientechnologies.orient.core.Orient;
import com.orientechnologies.orient.core.record.impl.ODocument;
import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.configuration.CompositeConfiguration;
import org.jbake.app.Asset;
import org.jbake.app.ConfigUtil;
import org.jbake.app.ContentStore;
import org.jbake.app.Crawler;
import org.jbake.app.DBUtil;
import org.jbake.app.FileUtil;
import org.jbake.app.JBakeException;
import org.jbake.app.Renderer;
import org.jbake.model.DocumentTypes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.bridge.SLF4JBridgeHandler;

public class Oven {
    private static final Logger LOGGER = LoggerFactory.getLogger(Oven.class);
    private static final Pattern TEMPLATE_DOC_PATTERN = Pattern.compile("(?:template\\.)([a-zA-Z0-9]+)(?:\\.file)");
    private CompositeConfiguration config;
    private File source;
    private File destination;
    private File templatesPath;
    private File contentsPath;
    private File assetsPath;
    private boolean isClearCache;
    private List<String> errors = new LinkedList<String>();
    private int renderedCount = 0;

    public Oven(File source, File destination, boolean isClearCache) throws Exception {
        this(source, destination, ConfigUtil.load(source), isClearCache);
    }

    public Oven(File source, File destination, CompositeConfiguration config, boolean isClearCache) {
        this.source = source;
        this.destination = destination;
        this.config = config;
        this.isClearCache = isClearCache;
        SLF4JBridgeHandler.removeHandlersForRootLogger();
        SLF4JBridgeHandler.install();
    }

    public CompositeConfiguration getConfig() {
        return this.config;
    }

    public void setConfig(CompositeConfiguration config) {
        this.config = config;
    }

    private void ensureSource() {
        if (!FileUtil.isExistingFolder(this.source)) {
            throw new JBakeException("Error: Source folder must exist: " + this.source.getAbsolutePath());
        }
        if (!this.source.canRead()) {
            throw new JBakeException("Error: Source folder is not readable: " + this.source.getAbsolutePath());
        }
    }

    private void ensureDestination() {
        if (null == this.destination) {
            this.destination = new File(this.source, this.config.getString("destination.folder"));
        }
        if (!this.destination.exists()) {
            this.destination.mkdirs();
        }
        if (!this.destination.canWrite()) {
            throw new JBakeException("Error: Destination folder is not writable: " + this.destination.getAbsolutePath());
        }
    }

    public void setupPaths() {
        this.ensureSource();
        this.templatesPath = this.setupRequiredFolderFromConfig("template.folder");
        this.contentsPath = this.setupRequiredFolderFromConfig("content.folder");
        this.assetsPath = this.setupPathFromConfig("asset.folder");
        if (!this.assetsPath.exists()) {
            LOGGER.warn("No asset folder was found!");
        }
        this.ensureDestination();
    }

    private File setupPathFromConfig(String key) {
        return new File(this.source, this.config.getString(key));
    }

    private File setupRequiredFolderFromConfig(String key) {
        File path = this.setupPathFromConfig(key);
        if (!FileUtil.isExistingFolder(path)) {
            throw new JBakeException("Error: Required folder cannot be found! Expected to find [" + key + "] at: " + path.getAbsolutePath());
        }
        return path;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void bake() {
        ContentStore db = DBUtil.createDataStore(this.config.getString("db.store"), this.config.getString("db.path"));
        this.updateDocTypesFromConfiguration();
        DBUtil.updateSchema(db);
        try {
            long start = new Date().getTime();
            LOGGER.info("Baking has started...");
            this.clearCacheIfNeeded(db);
            Crawler crawler = new Crawler(db, this.source, this.config);
            crawler.crawl(this.contentsPath);
            LOGGER.info("Content detected:");
            for (String docType : DocumentTypes.getDocumentTypes()) {
                int count = crawler.getDocumentCount(docType);
                if (count <= 0) continue;
                LOGGER.info("Parsed {} files of type: {}", (Object)count, (Object)docType);
            }
            Renderer renderer = new Renderer(db, this.destination, this.templatesPath, this.config);
            for (String docType : DocumentTypes.getDocumentTypes()) {
                for (ODocument document : db.getUnrenderedContent(docType)) {
                    try {
                        renderer.render(DBUtil.documentToModel(document));
                        ++this.renderedCount;
                    }
                    catch (Exception e) {
                        this.errors.add(e.getMessage());
                    }
                }
            }
            if (this.config.getBoolean("render.index")) {
                try {
                    renderer.renderIndex(this.config.getString("index.file"));
                }
                catch (Exception e) {
                    this.errors.add(e.getMessage());
                }
            }
            if (this.config.getBoolean("render.feed")) {
                try {
                    renderer.renderFeed(this.config.getString("feed.file"));
                }
                catch (Exception e) {
                    this.errors.add(e.getMessage());
                }
            }
            if (this.config.getBoolean("render.sitemap")) {
                try {
                    renderer.renderSitemap(this.config.getString("sitemap.file"));
                }
                catch (Exception e) {
                    this.errors.add(e.getMessage());
                }
            }
            if (this.config.getBoolean("render.archive")) {
                try {
                    renderer.renderArchive(this.config.getString("archive.file"));
                }
                catch (Exception e) {
                    this.errors.add(e.getMessage());
                }
            }
            if (this.config.getBoolean("render.tags")) {
                try {
                    renderer.renderTags(crawler.getTags(), this.config.getString("tag.path"));
                }
                catch (Exception e) {
                    this.errors.add(e.getMessage());
                }
            }
            for (String docType : DocumentTypes.getDocumentTypes()) {
                db.markConentAsRendered(docType);
            }
            Asset asset = new Asset(this.source, this.destination, this.config);
            asset.copy(this.assetsPath);
            this.errors.addAll(asset.getErrors());
            LOGGER.info("Baking finished!");
            long end = new Date().getTime();
            LOGGER.info("Baked {} items in {}ms", (Object)this.renderedCount, (Object)(end - start));
            if (this.errors.size() > 0) {
                LOGGER.error("Failed to bake {} item(s)!", (Object)this.errors.size());
            }
        }
        finally {
            db.close();
            Orient.instance().shutdown();
        }
    }

    private void updateDocTypesFromConfiguration() {
        Iterator keyIterator = this.config.getKeys();
        while (keyIterator.hasNext()) {
            String key = (String)keyIterator.next();
            Matcher matcher = TEMPLATE_DOC_PATTERN.matcher(key);
            if (!matcher.find()) continue;
            DocumentTypes.addDocumentType(matcher.group(1));
        }
    }

    private void clearCacheIfNeeded(ContentStore db) {
        boolean needed = this.isClearCache;
        if (!needed) {
            String currentTemplatesSignature;
            List<ODocument> docs = db.getSignaturesForTemplates();
            try {
                currentTemplatesSignature = FileUtil.sha1(this.templatesPath);
            }
            catch (Exception e) {
                currentTemplatesSignature = "";
            }
            if (!docs.isEmpty()) {
                String sha1 = (String)docs.get(0).field("sha1");
                boolean bl = needed = !sha1.equals(currentTemplatesSignature);
                if (needed) {
                    db.updateSignatures(currentTemplatesSignature);
                }
            } else {
                db.insertSignature(currentTemplatesSignature);
                needed = true;
            }
        }
        if (needed) {
            for (String docType : DocumentTypes.getDocumentTypes()) {
                try {
                    db.deleteAllByDocType(docType);
                }
                catch (Exception e) {
                    // empty catch block
                }
            }
            DBUtil.updateSchema(db);
        }
    }

    public List<String> getErrors() {
        return new ArrayList<String>(this.errors);
    }
}

