/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.client;

// $Id: $

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.StartEvent;
import org.jboss.bpm.model.Process.ProcessStatus;
import org.jboss.bpm.runtime.Attachments;

/**
 * The ExecutionManager executes processes
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Jun-2008
 */
public abstract class ExecutionManager
{
  private ExecutorService procExecutor = Executors.newCachedThreadPool();
  
  // Hide public constructor
  protected ExecutionManager()
  {
  }

  /**
   * Locate the ExecutionManager
   */
  public static ExecutionManager locateExecutionManager()
  {
    ProcessEngine engine = ProcessEngine.locateProcessEngine();
    return engine.getExecutionManager();
  }
  
  /**
   * Get the process executor service
   */
  public ExecutorService getProcessExecutor()
  {
    return procExecutor;
  }

  /**
   * Start the Process
   * 
   * @param proc The Process to start
   * @param att The Attachments in the ExecutionContext
   */
  public abstract void startProcess(Process proc, Attachments att);

  /**
   * Start the Process from a given start event
   * 
   * @param start The StartEvent that triggers the process
   * @param att The Attachments in the ExecutionContext
   */
  public abstract void startProcess(StartEvent start, Attachments att);
  
  /**
   * All Tokens that are generated at the Start Event for that Process must eventually arrive at an End Event. 
   * The Process will be in a running state until all Tokens are consumed.
   * <p/>
   * This method until the process ends without timeout.
   */
  public abstract ProcessStatus waitForEnd(Process proc);
  
  /**
   * All Tokens that are generated at the Start Event for that Process must eventually arrive at an End Event. 
   * The Process will be in a running state until all Tokens are consumed.
   * <p/>
   * This method until the process ends with a given timeout.
   */
  public abstract ProcessStatus waitForEnd(Process proc, long timeout);
}
