/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.client;

import java.net.URL;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.BPMException;
import org.jboss.bpm.client.internal.EmbeddedBeansDeployer;
import org.jboss.kernel.Kernel;
import org.jboss.kernel.plugins.util.KernelLocator;
import org.jboss.kernel.spi.registry.KernelRegistryEntry;
import org.jboss.kernel.spi.registry.KernelRegistryEntryNotFoundException;


// $Id: ProcessEngine.java 1889 2008-08-14 13:29:28Z thomas.diesler@jboss.com $

/**
 * The process engine is an agregator of various manger objects used by the BPM engine
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Jun-2008
 */
public class ProcessEngine
{
  // provide logging
  private static final Log log = LogFactory.getLog(ProcessEngine.class);
  
  /** The process engine bean name - jBPMEngine */
  public static final String BEAN_NAME = "jBPMProcessEngine";
  /** The default bean config: jbpm-beans.xml */
  public static final String JBPM_ENGINE_CONFIG = "jbpm-beans.xml";

  // Injected through the MC
  protected ProcessManager processManager;
  // Injected through the MC
  protected ExecutionManager executionManager;
  // Injected through the MC
  protected SignalManager signalManager;
  // Injected through the MC
  protected MessageManager messageManager;
  // Flag to indicate that the Engine is shutting down
  private boolean prepareForShutdown;

  // Hide public constructor
  protected ProcessEngine()
  {
  }

  /**
   * Locate the ProcessEngine instance shared by all clients.
   * 
   * @return The configured instance of a process engine
   */
  @SuppressWarnings("deprecation")
  public static ProcessEngine locateProcessEngine()
  {
    KernelRegistryEntry entry = null;
    Kernel kernel = KernelLocator.getKernel();
    if (kernel == null)
    {
      deployEngineConfiguration();
      kernel = KernelLocator.getKernel();
      entry = kernel.getRegistry().getEntry(ProcessEngine.BEAN_NAME);
    }
    else
    {
      try
      {
        entry = kernel.getRegistry().getEntry(ProcessEngine.BEAN_NAME);
      }
      catch (KernelRegistryEntryNotFoundException ex)
      {
        deployEngineConfiguration();
        entry = kernel.getRegistry().getEntry(ProcessEngine.BEAN_NAME);
      }
    }

    ProcessEngine engine = (ProcessEngine)entry.getTarget();
    return engine;
  }

  private static void deployEngineConfiguration()
  {
    String config = ProcessEngine.JBPM_ENGINE_CONFIG;
    URL url = Thread.currentThread().getContextClassLoader().getResource(config);
    if (url == null)
      throw new BPMException("Cannot find resource: " + config);

    new EmbeddedBeansDeployer().deploy(url);
  }
  
  /**
   * Prepare the engine for shutdown. During shutdown the creation of new processes is disallowed.
   */
  public void prepareForShutdown()
  {
    log.debug("prepareForShutdown");
    prepareForShutdown = true;
  }

  /**
   * True, if engine is preparing for shutdown.
   */
  public boolean isPrepareForShutdown()
  {
    return prepareForShutdown;
  }

  /**
   * Cancel the prepare for shutdown request
   */
  public void cancelShutdown()
  {
    log.debug("cancelShutdown");
    prepareForShutdown = false;
  }

  /**
   * Get the configured instance of the ProcessManager
   * 
   * @return The ProcessManager
   */
  public ProcessManager getProcessManager()
  {
    if (processManager == null)
      throw new IllegalStateException("ProcessManager not available through kernel configuration");
    return processManager;
  }

  /**
   * Get the configured instance of the ExecutionManager
   * 
   * @return The ExecutionManager
   */
  public ExecutionManager getExecutionManager()
  {
    if (executionManager == null)
      throw new IllegalStateException("ExecutionManager not available through kernel configuration");
    return executionManager;
  }

  /**
   * Get the configured instance of the SignalManager
   * 
   * @return The SignalManager
   */
  public SignalManager getSignalManager()
  {
    if (signalManager == null)
      throw new IllegalStateException("SignalManager not available through kernel configuration");
    return signalManager;
  }

  /**
   * Get the configured instance of the MessageManager
   * 
   * @return The MessageManager
   */
  public MessageManager getMessageManager()
  {
    if (messageManager == null)
      throw new IllegalStateException("MessageManager not available through kernel configuration");
    return messageManager;
  }
}