/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.client;

//$Id: SignalManager.java 1987 2008-08-22 18:56:09Z thomas.diesler@jboss.com $

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.model.Signal;

/**
 * The ProcessEngine sends signals through the SignalManager. <p/> A {@link Signal} is like an undirected flare shot up
 * into the air. A component can register a {@link SignalListener} with the SignalManager.
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Jun-2008
 */
public class SignalManager
{
  // provide logging
  private static final Log log = LogFactory.getLog(SignalManager.class);

  // The map of registered signal listeners
  private Set<SignalListener> listeners = new HashSet<SignalListener>();

  /**
   * Locate the SignalManager
   */
  public static SignalManager locateSignalManager()
  {
    ProcessEngine engine = ProcessEngine.locateProcessEngine();
    return engine.getSignalManager();
  }

  /**
   * Add a SignalListener for a given source
   */
  public void addSignalListener(SignalListener listener)
  {
    synchronized (listeners)
    {
      log.debug("addSignalListener: " + listener);
      listeners.add(listener);
    }
  }

  /**
   * Get the set of registered SignalListeners 
   */
  public Set<SignalListener> getSignalListeners()
  {
    synchronized (listeners)
    {
      HashSet<SignalListener> set = new HashSet<SignalListener>(listeners);
      return Collections.unmodifiableSet(set);
    }
  }

  /**
   * Remove a SignalListener for a given source
   */
  public void removeSignalListener(SignalListener listener)
  {
    synchronized (listeners)
    {
      log.debug("removeSignalListener: " + listener);
      listeners.remove(listener);
    }
  }

  /**
   * Throw a signal to all registered listeners
   */
  public void throwSignal(Signal signal)
  {
    log.debug("throwSignal: " + signal);
    Set<SignalListener> currentSet = getSignalListeners();
    for (SignalListener listener : currentSet)
    {
      if (failsafeAccept(listener, signal))
        failsafeThrow(listener, signal);
    }
  }

  private boolean failsafeAccept(SignalListener listener, Signal signal)
  {
    try
    {
      boolean accept = listener.acceptSignal(signal);
      return accept;
    }
    catch (RuntimeException rte)
    {
      log.error("Signal processing error", rte);
      return false;
    }
  }

  private void failsafeThrow(SignalListener listener, Signal signal)
  {
    try
    {
      listener.catchSignal(signal);
    }
    catch (RuntimeException rte)
    {
      log.error("Signal processing error", rte);
    }
  }
}