/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.model;

import java.util.List;


//$Id$

/**
 * An activity is a generic term for work that a company or organization performs via business processes. An activity can be atomic or non-atomic (compound). The types
 * of activities that are a part of a Process Model are: Process, SubProcess, and Task.
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
public interface Activity extends FlowObject, PropertySupport, SingleInFlowSupport, SingleOutFlowSupport
{
  /**
   * The ActivityType MUST be of type Task or Sub-Process.
   */
  enum ActivityType
  {
    Task, SubProcess
  }
  
  /**
   * The Activity LoopType
   */
  enum LoopType
  {
    None, Standard, MultiInstance
  }
  
  /**
   * The ActivityType MUST be of type Task or Sub-Process.
   */
  ActivityType getActivityType();

  /**
   * One or more Performers MAY be entered. The Performer attribute defines the
   * resource that will perform or will be responsible for the activity. The Performer
   * entry could be in the form of a specific individual, a group, an organization role
   * or position, or an organization.
   */
  List<String> getPerformers();
  
  /**
   * The InputSets attribute defines the data requirements for input to the Activity.
   * Zero or more InputSets MAY be defined. Each Input set is sufficient to allow the
   * Activity to be performed (if it has first been instantiated by the appropriate signal
   * arriving from an incoming Sequence Flow)
   */
  List<InputSet> getInputSets();
  
  /**
   * The OutputSets attribute defines the data requirements for output from the
   * activity. Zero or more OutputSets MAY be defined. At the completion of the
   * activity, only one of the OutputSets may be produced--It is up to the
   * implementation of the activity to determine which set will be produced. However,
   * the IORules attribute MAY indicate a relationship between an OutputSet and an
   * InputSet that started the activity.
   */
  List<OutputSet> getOutputSets();
  
  /**
   * The IORules attribute is a collection of expressions, each of which specifies the
   * required relationship between one input and one output. That is, if the activity is
   * instantiated with a specified input, that activity shall complete with the specified
   * output.
   */
  List<Expression> getIORules();
  
  /**
   * The default value is 1. The value MUST NOT be less than 1. This attribute
   * defines the number of Tokens that must arrive before the activity can begin.
   */
  int getStartQuantity();
  
  /**
   * The default value is 1. The value MUST NOT be less than 1. This attribute
   * defines the number of Tokens that must be generated from the activity. This
   * number of Tokens will be sent down any outgoing Sequence Flow (assuming any
   * Sequence Flow Conditions are satisfied).
   */
  int getCompletionQuantity();
  
  /**
   * LoopType is an attribute and is by default None, but MAY be set to Standard or
   * MultiInstance. A Task of type Receive that has its Instantiate attribute set to True MUST NOT
   * have a Standard or MultiInstance LoopType.
   */
  LoopType getLoopType();
}