/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.model;

import java.util.List;



//$Id$

/**
 * Gateways are modeling elements that are used to control how Sequence Flow interact as they converge and diverge within
 * a Process. If the flow does not need to be controlled, then a Gateway is not needed. The term “Gateway” implies that
 * there is a gating mechanism that either allows or disallows passage through the Gateway--that is, as Tokens arrive at a
 * Gateway, they can be merged together on input and/or split apart on output as the Gateway mechanisms are invoked. To
 * be more descriptive, a Gateway is actually a collection of “Gates.” 
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
public interface Gateway extends FlowObject
{
  /**
   * The GatewayType
   * 
   */
  public enum GatewayType
  {
    Exclusive, Inclusive, Complex, Parallel
  }

  /**
   * GatewayType is by default Exclusive. The GatewayType MAY be set to Inclusive, Complex, or Parallel. The GatewayType will determine the behavior of the Gateway,
   * both for incoming and outgoing Sequence Flow
   */
  GatewayType getGatewayType();

  /**
   * There MAY be zero or more Gates (except where noted below). Zero Gates are allowed if the Gateway is last object in a Process flow and there are no Start or End
   * Events for the Process. If there are zero or only one incoming Sequence Flow, then there MUST be at least two Gates.
   * 
   * For Exclusive Data-Based Gateways. 
   * When two Gates are required, one of them MAY be the DefaultGate.
   * 
   * For Exclusive Event-Based Gateways.
   * There MUST be two or more Gates. (Note that this type of Gateway does not act only as a Merge--it is always a Decision, at
   * least.)
   * 
   * For Inclusive Gateways. 
   * When two Gates are required, one of them MAY be the DefaultGate.
   */
  List<Gate> getGates();
  
  /**
   * Get the optional default gate
   * @return null if there is none
   */
  Gate getDefaultGate();
  
  /**
   * Get the gate for the given target name
   * @return null if there is none
   */
  Gate getGateByTargetName(String targetName);
}