/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.runtime;

// $Id: $

import org.jboss.bpm.model.InputSet;
import org.jboss.bpm.model.OutputSet;
import org.jboss.bpm.model.SequenceFlow;

/**
 * A Token is a descriptive construct used to describe how the flow of a Process will proceed at runtime.
 * 
 * By tracking how the Token traverses the Flow Objects, gets diverted through alternative paths, and gets split into parallel paths, the normal Sequence Flow should be
 * completely definable.
 * 
 * A Token will have a unique identity that can be used to separate multiple Tokens that may exist because of concurrent process instances or the splitting of the Token
 * for parallel processing within a single process instance.
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 20-Apr-2007
 */
public interface Token
{
  public enum TokenStatus
  {
    Created, Started, Stoped, Destroyed, Suspended
  }

  /**
   * Get the token status
   */
  TokenStatus getTokenStatus();
  
  /**
   * Get the unique token identity
   */
  String getTokenID();

  /**
   * Get the associated {@link ExecutionContext}
   */
  ExecutionContext getExecutionContext();

  /**
   * Get the current {@link SequenceFlow} 
   */
  SequenceFlow getFlow();
  
  /**
   * Get the active {@link InputSet}  
   * @return null if the Activity does not define one
   */
  InputSet getInputSet();
  
  /**
   * Get the active {@link OutputSet}  
   * @return null if the Activity does not define one
   */
  OutputSet getOutputSet();
  
  /**
   * Create a schallow copy of this Token.
   * <p/>
   * The content in the ExecutionContext will be copied by reference. 
   */
  Token copyToken();

  /**
   * Merge this Token with another token.
   */
  void mergeToken(Token token);
}