/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.test;

// $Id: AbstractAPITestCase.java 1987 2008-08-22 18:56:09Z thomas.diesler@jboss.com $

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.management.ObjectName;

import junit.framework.TestCase;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.BPMException;
import org.jboss.bpm.client.DialectHandler;
import org.jboss.bpm.client.MessageListener;
import org.jboss.bpm.client.MessageManager;
import org.jboss.bpm.client.ProcessManager;
import org.jboss.bpm.client.SignalListener;
import org.jboss.bpm.client.SignalManager;
import org.jboss.bpm.client.internal.EmbeddedBeansDeployer;
import org.jboss.bpm.model.Constants;
import org.jboss.bpm.model.Message;
import org.jboss.bpm.model.ObjectNameFactory;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.Signal;

public abstract class AbstractAPITestCase extends TestCase
{
  // provide logging
  protected final Log log = LogFactory.getLog(getClass());

  // Every test case has a deployer
  private EmbeddedBeansDeployer deployer;
  // The embedded SignalListener
  private SignalListener signalListener;
  // The embedded MessageListener
  private MessageListener messageListener;
  // The signals caught by this test case
  private List<Signal> signals = new ArrayList<Signal>();
  // The messages caught by this test case
  private List<Message> messages = new ArrayList<Message>();

  /** Overwrite to provide beans config */
  protected abstract String getBeansConfig();

  @Override
  protected void setUp() throws Exception
  {
    super.setUp();
    log.debug("setUp: " + getClass().getName() + "." + getName());

    deployer = new EmbeddedBeansDeployer();

    // Setup the SignalListener
    clearAllSignalListeners();
    SignalManager sigManager = SignalManager.locateSignalManager();
    sigManager.addSignalListener(getSignalListener());
    synchronized (signals)
    {
      signals.clear();
    }

    // Setup the MessageListener
    clearAllMessageListeners();
    MessageManager msgManager = MessageManager.locateMessageManager();
    msgManager.addMessageListener(getMessageListener());
  }

  private void clearAllSignalListeners()
  {
    SignalManager sigManager = SignalManager.locateSignalManager();
    Set<SignalListener> sigListeners = sigManager.getSignalListeners();
    for (SignalListener sigListener : sigListeners)
    {
      sigManager.removeSignalListener(sigListener);
    }
  }

  private void clearAllMessageListeners()
  {
    MessageManager msgManager = MessageManager.locateMessageManager();
    Set<MessageListener> msgListeners = msgManager.getMessageListeners();
    for (MessageListener msgListener : msgListeners)
    {
      msgManager.removeMessageListener(msgListener.getID());
    }
  }

  @Override
  protected void tearDown() throws Exception
  {
    log.debug("tearDown: " + getClass().getName() + "." + getName());

    // Tear down the SignalListener
    SignalManager signalManager = SignalManager.locateSignalManager();
    signalManager.removeSignalListener(getSignalListener());

    // Tear down the MessageListener
    MessageManager messageManager = MessageManager.locateMessageManager();
    messageManager.removeMessageListener(getTestID());

    // Check that there are no registered processes left
    ProcessManager procManager = ProcessManager.locateProcessManager();
    Set<Process> procs = procManager.getProcesses();
    if (procs.size() > 0)
    {
      String logMsg = "Registered processes on tear down of " + getName() + ": " + procs;
      System.out.println(logMsg);
      log.warn(logMsg);
    }

    // Check that there are no registered signal listeners left
    Set<SignalListener> sigListeners = signalManager.getSignalListeners();
    if (sigListeners.size() > 0)
    {
      String logMsg = "Registered signal listeners on tear down of " + getName() + ": " + sigListeners;
      System.out.println(logMsg);
      log.warn(logMsg);
    }

    // Check that there are no registered signal listeners left
    Set<MessageListener> msgListeners = messageManager.getMessageListeners();
    if (msgListeners.size() > 0)
    {
      String logMsg = "Registered message listeners on tear down of " + getName() + ": " + msgListeners;
      System.out.println(logMsg);
      log.warn(logMsg);
    }

    super.tearDown();
  }

  // Deploy a beans config
  protected void deployBeans(String resourceName)
  {
    URL url = getBeansConfigURL(resourceName);
    deployer.deploy(url);
  }

  // Undeploy a beans config
  protected void undeployBeans(String resourceName)
  {
    URL url = getBeansConfigURL(resourceName);
    deployer.undeploy(url);
    deployer = null;
  }

  // Get the URL for an resource
  protected URL getResourceURL(String resource)
  {
    ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
    URL url = ctxLoader.getResource(resource);
    if (url == null)
      throw new IllegalStateException("Cannot find resource: " + resource);

    return url;
  }

  private URL getBeansConfigURL(String resourceName)
  {
    URL url = Thread.currentThread().getContextClassLoader().getResource(resourceName);
    if (url == null)
      throw new BPMException("Cannot find resource: " + resourceName);
    return url;
  }

  public SignalListener getSignalListener()
  {
    if (signalListener == null)
    {
      signalListener = new SignalListener()
      {
        public boolean acceptSignal(Signal signal)
        {
          return true;
        }
        
        public void catchSignal(Signal signal)
        {
          synchronized (signals)
          {
            signals.add(signal);
          }
        }

        public String toString()
        {
          return "SignalListener[" + getShortName() + "]";
        }
      };
    }
    return signalListener;
  }

  public List<Signal> getSignals()
  {
    synchronized (signals)
    {
      return Collections.unmodifiableList(signals);
    }
  }

  public List<Signal> getSignals(Signal.SignalType type)
  {
    synchronized (signals)
    {
      List<Signal> retSignals = new ArrayList<Signal>();
      for (Signal sig : signals)
      {
        if (sig.getSignalType() == type)
          retSignals.add(sig);
      }
      return Collections.unmodifiableList(retSignals);
    }
  }

  public MessageListener getMessageListener()
  {
    if (messageListener == null)
    {
      messageListener = new MessageListener()
      {
        public ObjectName getID()
        {
          return getTestID();
        }

        public void catchMessage(Message message)
        {
          synchronized (messages)
          {
            log.debug("catchMessage: " + message);
            messages.add(message);
          }
        }
      };
    }
    return messageListener;
  }

  public ObjectName getTestID()
  {
    String shortName = getShortName();
    shortName = shortName.replace("DescriptorTest", "Test"); 
    shortName = shortName.replace("MarshallerTest", "Test"); 
    return ObjectNameFactory.create(Constants.ID_DOMAIN, "test", shortName);
  }

  public List<Message> getMessages()
  {
    synchronized (messages)
    {
      return Collections.unmodifiableList(messages);
    }
  }

  /**
   * Marshall the given process
   * 
   * @param out if null, the proces is marshalled to a file
   */
  public String marshallProcess(Process proc, Writer out)
  {
    try
    {
      if (out == null)
      {
        File file = new File("target/" + getName() + "-" + getDialect() + ".xml");
        out = new FileWriter(file);
        System.out.println("Marshall process to: " + file.getCanonicalPath());
      }

      String procXML = marshallProcess(proc);
      out.write(procXML);
      out.close();

      return procXML;
    }
    catch (IOException ex)
    {
      throw new BPMException("Cannot marshall process", ex);
    }
  }

  /**
   * Marshall the given process
   */
  public String marshallProcess(Process proc) throws IOException
  {
    ProcessManager procManager = ProcessManager.locateProcessManager();
    DialectHandler dialectHandler = procManager.getDialectHandler(getDialectURI());

    StringWriter strwr = new StringWriter();
    dialectHandler.marshallProcess(proc, strwr);
    return strwr.toString();
  }

  /**
   * Get the test short name, which is the class name without the package
   */
  protected String getShortName()
  {
    String shortName = getClass().getName();
    shortName = shortName.substring(shortName.lastIndexOf(".") + 1);
    return shortName;
  }
  
  /**
   * Get the BPM descriptor dialect ID 
   */
  protected String getDialect()
  {
    String dialect = System.getProperty("jbpm.dialect", "api10");
    return dialect;
  }
  
  /**
   * Get the BPM descriptor dialect ID 
   */
  protected String getDialectURI()
  {
    String dialect = System.getProperty("jbpm.dialect.uri", DialectHandler.DEFAULT_NAMESPACE_URI);
    return dialect;
  }
}
