/*
 * JCusolver - Java bindings for CUSOLVER, the NVIDIA CUDA solver
 * library, to be used with JCuda
 *
 * Copyright (c) 2010-2015 Marco Hutter - http://www.jcuda.org
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

#include "JCusolverDn.hpp"
#include "JCusolver_common.hpp"
#include <iostream>
#include <string>

//=== Auto-generated part: ===================================================

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCreateNative(JNIEnv *env, jclass cls, jobject handle)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCreate");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCreate(handle=%p)\n",
        handle);

    // Native variable declarations
    cusolverDnHandle_t handle_native;

    // Obtain native variable values
    // handle is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCreate(&handle_native);

    // Write back native variable values
    setNativePointerValue(env, handle, (jlong)handle_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDestroyNative(JNIEnv *env, jclass cls, jobject handle)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDestroy");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDestroy(handle=%p)\n",
        handle);

    // Native variable declarations
    cusolverDnHandle_t handle_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDestroy(handle_native);

    // Write back native variable values
    // handle is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSetStreamNative(JNIEnv *env, jclass cls, jobject handle, jobject streamId)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSetStream");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // streamId is checked by the library

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSetStream(handle=%p, streamId=%p)\n",
        handle, streamId);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cudaStream_t streamId_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    streamId_native = (cudaStream_t)getNativePointerValue(env, streamId);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSetStream(handle_native, streamId_native);

    // Write back native variable values
    // handle is read-only
    // streamId is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnGetStreamNative(JNIEnv *env, jclass cls, jobject handle, jobject streamId)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnGetStream");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (streamId == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'streamId' is null for cusolverDnGetStream");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnGetStream(handle=%p, streamId=%p)\n",
        handle, streamId);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cudaStream_t streamId_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    // streamId is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnGetStream(handle_native, &streamId_native);

    // Write back native variable values
    // handle is read-only
    setNativePointerValue(env, streamId, (jlong)streamId_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

//============================================================
// IRS headers
//============================================================
// =============================================================================
// IRS helper function API
// =============================================================================
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSParamsCreateNative(JNIEnv *env, jclass cls, jobject params_ptr)
{
    // Null-checks for non-primitive arguments
    if (params_ptr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params_ptr' is null for cusolverDnIRSParamsCreate");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSParamsCreate(params_ptr=%p)\n",
        params_ptr);

    // Native variable declarations
    cusolverDnIRSParams_t params_ptr_native;

    // Obtain native variable values
    // params_ptr is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSParamsCreate(&params_ptr_native);

    // Write back native variable values
    setNativePointerValue(env, params_ptr, (jlong)params_ptr_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSParamsDestroyNative(JNIEnv *env, jclass cls, jobject params)
{
    // Null-checks for non-primitive arguments
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnIRSParamsDestroy");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSParamsDestroy(params=%p)\n",
        params);

    // Native variable declarations
    cusolverDnIRSParams_t params_native;

    // Obtain native variable values
    params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSParamsDestroy(params_native);

    // Write back native variable values
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSParamsSetRefinementSolverNative(JNIEnv *env, jclass cls, jobject params, jint refinement_solver)
{
    // Null-checks for non-primitive arguments
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnIRSParamsSetRefinementSolver");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // refinement_solver is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSParamsSetRefinementSolver(params=%p, refinement_solver=%d)\n",
        params, refinement_solver);

    // Native variable declarations
    cusolverDnIRSParams_t params_native;
    cusolverIRSRefinement_t refinement_solver_native;

    // Obtain native variable values
    params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, params);
    refinement_solver_native = (cusolverIRSRefinement_t)refinement_solver;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSParamsSetRefinementSolver(params_native, refinement_solver_native);

    // Write back native variable values
    // params is read-only
    // refinement_solver is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSParamsSetSolverMainPrecisionNative(JNIEnv *env, jclass cls, jobject params, jint solver_main_precision)
{
    // Null-checks for non-primitive arguments
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnIRSParamsSetSolverMainPrecision");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // solver_main_precision is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSParamsSetSolverMainPrecision(params=%p, solver_main_precision=%d)\n",
        params, solver_main_precision);

    // Native variable declarations
    cusolverDnIRSParams_t params_native;
    cusolverPrecType_t solver_main_precision_native;

    // Obtain native variable values
    params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, params);
    solver_main_precision_native = (cusolverPrecType_t)solver_main_precision;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSParamsSetSolverMainPrecision(params_native, solver_main_precision_native);

    // Write back native variable values
    // params is read-only
    // solver_main_precision is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSParamsSetSolverLowestPrecisionNative(JNIEnv *env, jclass cls, jobject params, jint solver_lowest_precision)
{
    // Null-checks for non-primitive arguments
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnIRSParamsSetSolverLowestPrecision");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // solver_lowest_precision is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSParamsSetSolverLowestPrecision(params=%p, solver_lowest_precision=%d)\n",
        params, solver_lowest_precision);

    // Native variable declarations
    cusolverDnIRSParams_t params_native;
    cusolverPrecType_t solver_lowest_precision_native;

    // Obtain native variable values
    params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, params);
    solver_lowest_precision_native = (cusolverPrecType_t)solver_lowest_precision;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSParamsSetSolverLowestPrecision(params_native, solver_lowest_precision_native);

    // Write back native variable values
    // params is read-only
    // solver_lowest_precision is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSParamsSetSolverPrecisionsNative(JNIEnv *env, jclass cls, jobject params, jint solver_main_precision, jint solver_lowest_precision)
{
    // Null-checks for non-primitive arguments
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnIRSParamsSetSolverPrecisions");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // solver_main_precision is primitive
    // solver_lowest_precision is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSParamsSetSolverPrecisions(params=%p, solver_main_precision=%d, solver_lowest_precision=%d)\n",
        params, solver_main_precision, solver_lowest_precision);

    // Native variable declarations
    cusolverDnIRSParams_t params_native;
    cusolverPrecType_t solver_main_precision_native;
    cusolverPrecType_t solver_lowest_precision_native;

    // Obtain native variable values
    params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, params);
    solver_main_precision_native = (cusolverPrecType_t)solver_main_precision;
    solver_lowest_precision_native = (cusolverPrecType_t)solver_lowest_precision;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSParamsSetSolverPrecisions(params_native, solver_main_precision_native, solver_lowest_precision_native);

    // Write back native variable values
    // params is read-only
    // solver_main_precision is primitive
    // solver_lowest_precision is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSParamsSetTolNative(JNIEnv *env, jclass cls, jobject params, jdouble val)
{
    // Null-checks for non-primitive arguments
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnIRSParamsSetTol");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // val is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSParamsSetTol(params=%p, val=%lf)\n",
        params, val);

    // Native variable declarations
    cusolverDnIRSParams_t params_native;
    double val_native = 0.0;

    // Obtain native variable values
    params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, params);
    val_native = (double)val;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSParamsSetTol(params_native, val_native);

    // Write back native variable values
    // params is read-only
    // val is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSParamsSetTolInnerNative(JNIEnv *env, jclass cls, jobject params, jdouble val)
{
    // Null-checks for non-primitive arguments
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnIRSParamsSetTolInner");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // val is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSParamsSetTolInner(params=%p, val=%lf)\n",
        params, val);

    // Native variable declarations
    cusolverDnIRSParams_t params_native;
    double val_native = 0.0;

    // Obtain native variable values
    params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, params);
    val_native = (double)val;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSParamsSetTolInner(params_native, val_native);

    // Write back native variable values
    // params is read-only
    // val is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSParamsSetMaxItersNative(JNIEnv *env, jclass cls, jobject params, jint maxiters)
{
    // Null-checks for non-primitive arguments
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnIRSParamsSetMaxIters");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // maxiters is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSParamsSetMaxIters(params=%p, maxiters=%d)\n",
        params, maxiters);

    // Native variable declarations
    cusolverDnIRSParams_t params_native;
    cusolver_int_t maxiters_native = 0;

    // Obtain native variable values
    params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, params);
    maxiters_native = (cusolver_int_t)maxiters;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSParamsSetMaxIters(params_native, maxiters_native);

    // Write back native variable values
    // params is read-only
    // maxiters is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSParamsSetMaxItersInnerNative(JNIEnv *env, jclass cls, jobject params, jint maxiters_inner)
{
    // Null-checks for non-primitive arguments
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnIRSParamsSetMaxItersInner");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // maxiters_inner is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSParamsSetMaxItersInner(params=%p, maxiters_inner=%d)\n",
        params, maxiters_inner);

    // Native variable declarations
    cusolverDnIRSParams_t params_native;
    cusolver_int_t maxiters_inner_native = 0;

    // Obtain native variable values
    params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, params);
    maxiters_inner_native = (cusolver_int_t)maxiters_inner;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSParamsSetMaxItersInner(params_native, maxiters_inner_native);

    // Write back native variable values
    // params is read-only
    // maxiters_inner is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSParamsGetMaxItersNative(JNIEnv *env, jclass cls, jobject params, jintArray maxiters)
{
    // Null-checks for non-primitive arguments
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnIRSParamsGetMaxIters");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (maxiters == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'maxiters' is null for cusolverDnIRSParamsGetMaxIters");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSParamsGetMaxIters(params=%p, maxiters=%p)\n",
        params, maxiters);

    // Native variable declarations
    cusolverDnIRSParams_t params_native;
    cusolver_int_t maxiters_native;

    // Obtain native variable values
    params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, params);
    // maxiters is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSParamsGetMaxIters(params_native, &maxiters_native);

    // Write back native variable values
    // params is read-only
    if (!set(env, maxiters, 0, (jint)maxiters_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSParamsEnableFallbackNative(JNIEnv *env, jclass cls, jobject params)
{
    // Null-checks for non-primitive arguments
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnIRSParamsEnableFallback");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSParamsEnableFallback(params=%p)\n",
        params);

    // Native variable declarations
    cusolverDnIRSParams_t params_native;

    // Obtain native variable values
    params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSParamsEnableFallback(params_native);

    // Write back native variable values
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSParamsDisableFallbackNative(JNIEnv *env, jclass cls, jobject params)
{
    // Null-checks for non-primitive arguments
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnIRSParamsDisableFallback");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSParamsDisableFallback(params=%p)\n",
        params);

    // Native variable declarations
    cusolverDnIRSParams_t params_native;

    // Obtain native variable values
    params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSParamsDisableFallback(params_native);

    // Write back native variable values
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

// =============================================================================
// cusolverDnIRSInfos prototypes
// =============================================================================
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSInfosDestroyNative(JNIEnv *env, jclass cls, jobject infos)
{
    // Null-checks for non-primitive arguments
    if (infos == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'infos' is null for cusolverDnIRSInfosDestroy");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSInfosDestroy(infos=%p)\n",
        infos);

    // Native variable declarations
    cusolverDnIRSInfos_t infos_native;

    // Obtain native variable values
    infos_native = (cusolverDnIRSInfos_t)getNativePointerValue(env, infos);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSInfosDestroy(infos_native);

    // Write back native variable values
    // infos is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSInfosCreateNative(JNIEnv *env, jclass cls, jobject infos_ptr)
{
    // Null-checks for non-primitive arguments
    if (infos_ptr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'infos_ptr' is null for cusolverDnIRSInfosCreate");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSInfosCreate(infos_ptr=%p)\n",
        infos_ptr);

    // Native variable declarations
    cusolverDnIRSInfos_t infos_ptr_native;

    // Obtain native variable values
    // infos_ptr is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSInfosCreate(&infos_ptr_native);

    // Write back native variable values
    setNativePointerValue(env, infos_ptr, (jlong)infos_ptr_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSInfosGetNitersNative(JNIEnv *env, jclass cls, jobject infos, jintArray niters)
{
    // Null-checks for non-primitive arguments
    if (infos == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'infos' is null for cusolverDnIRSInfosGetNiters");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (niters == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'niters' is null for cusolverDnIRSInfosGetNiters");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSInfosGetNiters(infos=%p, niters=%p)\n",
        infos, niters);

    // Native variable declarations
    cusolverDnIRSInfos_t infos_native;
    cusolver_int_t niters_native;

    // Obtain native variable values
    infos_native = (cusolverDnIRSInfos_t)getNativePointerValue(env, infos);
    // niters is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSInfosGetNiters(infos_native, &niters_native);

    // Write back native variable values
    // infos is read-only
    if (!set(env, niters, 0, (jint)niters_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSInfosGetOuterNitersNative(JNIEnv *env, jclass cls, jobject infos, jintArray outer_niters)
{
    // Null-checks for non-primitive arguments
    if (infos == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'infos' is null for cusolverDnIRSInfosGetOuterNiters");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (outer_niters == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'outer_niters' is null for cusolverDnIRSInfosGetOuterNiters");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSInfosGetOuterNiters(infos=%p, outer_niters=%p)\n",
        infos, outer_niters);

    // Native variable declarations
    cusolverDnIRSInfos_t infos_native;
    cusolver_int_t outer_niters_native;

    // Obtain native variable values
    infos_native = (cusolverDnIRSInfos_t)getNativePointerValue(env, infos);
    // outer_niters is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSInfosGetOuterNiters(infos_native, &outer_niters_native);

    // Write back native variable values
    // infos is read-only
    if (!set(env, outer_niters, 0, (jint)outer_niters_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSInfosRequestResidualNative(JNIEnv *env, jclass cls, jobject infos)
{
    // Null-checks for non-primitive arguments
    if (infos == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'infos' is null for cusolverDnIRSInfosRequestResidual");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSInfosRequestResidual(infos=%p)\n",
        infos);

    // Native variable declarations
    cusolverDnIRSInfos_t infos_native;

    // Obtain native variable values
    infos_native = (cusolverDnIRSInfos_t)getNativePointerValue(env, infos);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSInfosRequestResidual(infos_native);

    // Write back native variable values
    // infos is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSInfosGetResidualHistoryNative(JNIEnv *env, jclass cls, jobject infos, jobject residual_history)
{
    // Null-checks for non-primitive arguments
    if (infos == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'infos' is null for cusolverDnIRSInfosGetResidualHistory");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (residual_history == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'residual_history' is null for cusolverDnIRSInfosGetResidualHistory");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSInfosGetResidualHistory(infos=%p, residual_history=%p)\n",
        infos, residual_history);

    // Native variable declarations
    cusolverDnIRSInfos_t infos_native;
    void * residual_history_native;

    // Obtain native variable values
    infos_native = (cusolverDnIRSInfos_t)getNativePointerValue(env, infos);
    // residual_history is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSInfosGetResidualHistory(infos_native, &residual_history_native);

    // Write back native variable values
    // infos is read-only

    // The residual_history is returned via a pointer
    // TODO: It is not clear how long the history will be here
    // so the byte buffer cannot be created
    //jobject residual_history_ByteBuffer = env->NewDirectByteBuffer(residual_history_native, ?size?);
    //env->SetObjectField(residual_history, Pointer_buffer, residual_history_ByteBuffer);
    env->SetObjectField(residual_history, Pointer_pointers, NULL);
    env->SetLongField(residual_history, Pointer_byteOffset, 0);
    env->SetLongField(residual_history, NativePointerObject_nativePointer, (jlong)residual_history_native);

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSInfosGetMaxItersNative(JNIEnv *env, jclass cls, jobject infos, jintArray maxiters)
{
    // Null-checks for non-primitive arguments
    if (infos == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'infos' is null for cusolverDnIRSInfosGetMaxIters");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (maxiters == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'maxiters' is null for cusolverDnIRSInfosGetMaxIters");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSInfosGetMaxIters(infos=%p, maxiters=%p)\n",
        infos, maxiters);

    // Native variable declarations
    cusolverDnIRSInfos_t infos_native;
    cusolver_int_t maxiters_native;

    // Obtain native variable values
    infos_native = (cusolverDnIRSInfos_t)getNativePointerValue(env, infos);
    // maxiters is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSInfosGetMaxIters(infos_native, &maxiters_native);

    // Write back native variable values
    // infos is read-only
    if (!set(env, maxiters, 0, (jint)maxiters_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

//============================================================
//  IRS functions API
//============================================================
/*******************************************************************************/
/**
 * [ZZ, ZC, ZK, ZE, ZY, CC, CK, CE, CY, DD, DS, DH, DB, DX, SS, SH, SB, SX]gesv
 * users API Prototypes */
/*******************************************************************************/
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZZgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZZgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZZgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnZZgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZZgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZZgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZZgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnZZgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnZZgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZZgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZZgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZCgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnZCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnZCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnZCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZCgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZCgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZKgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnZKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnZKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnZKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZKgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZKgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZEgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnZEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnZEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnZEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZEgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZEgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZYgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnZYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnZYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnZYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZYgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZYgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCCgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnCCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnCCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnCCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnCCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnCCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnCCgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCCgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCCgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCEgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnCEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnCEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnCEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnCEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnCEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnCEgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCEgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCEgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCKgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnCKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnCKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnCKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnCKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnCKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnCKgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCKgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCKgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCYgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnCYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnCYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnCYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnCYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnCYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnCYgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCYgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCYgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDDgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDDgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDDgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnDDgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDDgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDDgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDDgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnDDgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnDDgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDDgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDDgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDSgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnDSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnDSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnDSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDSgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDSgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDHgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnDHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnDHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnDHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDHgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDHgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDBgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnDBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnDBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnDBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDBgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDBgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDXgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnDXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnDXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnDXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDXgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDXgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSSgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnSSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnSSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnSSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnSSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnSSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnSSgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSSgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSSgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSHgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnSHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnSHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnSHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnSHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnSHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnSHgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSHgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSHgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSBgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnSBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnSBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnSBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnSBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnSBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnSBgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSBgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSBgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSXgesvNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dipiv' is null for cusolverDnSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSXgesv(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSXgesv(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/*******************************************************************************/
/*******************************************************************************/
/**
 * [ZZ, ZC, ZK, ZE, ZY, CC, CK, CE, CY, DD, DS, DH, DB, DX, SS, SH, SB, SX]gesv_bufferSize
 * users API Prototypes */
/*******************************************************************************/
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZZgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZZgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZZgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnZZgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZZgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZZgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZCgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZCgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZCgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnZCgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZCgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZCgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZKgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZKgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZKgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnZKgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZKgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZKgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZEgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZEgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZEgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnZEgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZEgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZEgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZYgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZYgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZYgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnZYgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZYgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZYgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCCgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCCgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCCgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnCCgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCCgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCCgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCKgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCKgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCKgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnCKgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCKgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCKgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCEgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCEgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCEgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnCEgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCEgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCEgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCYgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCYgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCYgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnCYgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCYgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCYgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDDgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDDgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDDgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnDDgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDDgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDDgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDSgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDSgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDSgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnDSgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDSgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDSgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDHgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDHgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDHgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnDHgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDHgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDHgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDBgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDBgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDBgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnDBgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDBgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDBgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDXgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDXgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDXgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnDXgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDXgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDXgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSSgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSSgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSSgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnSSgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSSgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSSgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSHgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSHgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSHgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnSHgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSHgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSHgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSBgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSBgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSBgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnSBgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSBgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSBgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSXgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jint nrhs, jobject dA, jint ldda, jintArray dipiv, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSXgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    // dA is checked by the library
    // ldda is primitive
    // dipiv is checked by the library
    // dB is checked by the library
    // lddb is primitive
    // dX is checked by the library
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSXgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnSXgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSXgesv_bufferSize(handle=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dipiv=%p, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, n, nrhs, dA, ldda, dipiv, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cusolver_int_t * dipiv_native = NULL;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dipiv_native = (cusolver_int_t *)getPointer(env, dipiv);
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSXgesv_bufferSize(handle_native, n_native, nrhs_native, dA_native, ldda_native, dipiv_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dipiv is a native pointer
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/*******************************************************************************/
/*******************************************************************************/
/**
 * [ZZ, ZC, ZK, ZE, ZY, CC, CK, CE, CY, DD, DS, DH, DB, DX, SS, SH, SB, SX]gels
 * users API Prototypes */
/*******************************************************************************/
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZZgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZZgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZZgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZZgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZZgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZZgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnZZgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnZZgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZZgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZZgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZCgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZCgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZCgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZCgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZCgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZCgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnZCgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnZCgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZCgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZCgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZKgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZKgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZKgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZKgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZKgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZKgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnZKgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnZKgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZKgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZKgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZEgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZEgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZEgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZEgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZEgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZEgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnZEgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnZEgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZEgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZEgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZYgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZYgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZYgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZYgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZYgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZYgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnZYgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnZYgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZYgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZYgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCCgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCCgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnCCgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnCCgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnCCgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCCgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnCCgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnCCgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCCgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCCgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCKgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCKgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnCKgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnCKgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnCKgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCKgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnCKgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnCKgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCKgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCKgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCEgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCEgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnCEgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnCEgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnCEgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCEgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnCEgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnCEgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCEgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCEgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCYgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCYgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnCYgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnCYgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnCYgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCYgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnCYgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnCYgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCYgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCYgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDDgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDDgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDDgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDDgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDDgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDDgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnDDgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnDDgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDDgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDDgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDSgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDSgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDSgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDSgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDSgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDSgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnDSgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnDSgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDSgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDSgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDHgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDHgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDHgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDHgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDHgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDHgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnDHgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnDHgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDHgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDHgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDBgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDBgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDBgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDBgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDBgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDBgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnDBgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnDBgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDBgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDBgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDXgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnDXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnDXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDXgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDXgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSSgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSSgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnSSgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnSSgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnSSgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSSgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnSSgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnSSgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSSgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSSgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSHgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSHgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnSHgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnSHgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnSHgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSHgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnSHgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnSHgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSHgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSHgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSBgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSBgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnSBgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnSBgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnSBgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSBgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnSBgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnSBgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSBgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSBgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSXgelsNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray iter, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (iter == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'iter' is null for cusolverDnSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSXgels(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, iter=%p, d_info=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, iter, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t iter_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // iter is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSXgels(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &iter_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, iter, 0, (jint)iter_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/*******************************************************************************/
/*******************************************************************************/
/**
 * [ZZ, ZC, ZK, ZE, ZY, CC, CK, CE, CY, DD, DS, DH, DB, DX, SS, SH, SB, SX]gels_bufferSize
 * API prototypes */
/*******************************************************************************/
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZZgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZZgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZZgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZZgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZZgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZZgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnZZgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZZgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZZgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZCgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZCgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZCgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZCgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZCgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZCgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnZCgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZCgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZCgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZKgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZKgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZKgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZKgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZKgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZKgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnZKgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZKgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZKgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZEgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZEgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZEgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZEgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZEgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZEgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnZEgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZEgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZEgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZYgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZYgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnZYgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnZYgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnZYgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnZYgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnZYgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZYgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuDoubleComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuDoubleComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuDoubleComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuDoubleComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuDoubleComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuDoubleComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZYgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCCgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCCgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnCCgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnCCgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnCCgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCCgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnCCgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCCgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCCgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCKgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCKgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnCKgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnCKgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnCKgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCKgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnCKgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCKgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCKgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCEgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCEgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnCEgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnCEgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnCEgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCEgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnCEgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCEgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCEgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCYgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCYgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnCYgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnCYgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnCYgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnCYgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnCYgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCYgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    cuComplex * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    cuComplex * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    cuComplex * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (cuComplex *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (cuComplex *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (cuComplex *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCYgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDDgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDDgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDDgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDDgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDDgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDDgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnDDgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDDgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDDgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDSgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDSgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDSgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDSgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDSgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDSgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnDSgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDSgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDSgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDHgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDHgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDHgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDHgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDHgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDHgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnDHgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDHgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDHgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDBgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDBgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDBgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDBgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDBgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDBgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnDBgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDBgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDBgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDXgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnDXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnDXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnDXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnDXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnDXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDXgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    double * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    double * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    double * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (double *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (double *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (double *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDXgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSSgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSSgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnSSgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnSSgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnSSgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSSgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnSSgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSSgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSSgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSHgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSHgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnSHgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnSHgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnSHgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSHgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnSHgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSHgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSHgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSBgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSBgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnSBgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnSBgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnSBgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSBgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnSBgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSBgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSBgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSXgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnSXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnSXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnSXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnSXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnSXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSXgels_bufferSize(handle=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%p)\n",
        handle, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    float * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    float * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    float * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (float *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (float *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (float *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSXgels_bufferSize(handle_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/*******************************************************************************/
/*******************************************************************************/
/**
 * expert users API for IRS Prototypes
 * */
/*******************************************************************************/
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSXgesvNative(JNIEnv *env, jclass cls, jobject handle, jobject gesv_irs_params, jobject gesv_irs_infos, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray niters, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnIRSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (gesv_irs_params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'gesv_irs_params' is null for cusolverDnIRSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (gesv_irs_infos == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'gesv_irs_infos' is null for cusolverDnIRSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnIRSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    // dB is checked by the library
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnIRSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnIRSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (niters == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'niters' is null for cusolverDnIRSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnIRSXgesv");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSXgesv(handle=%p, gesv_irs_params=%p, gesv_irs_infos=%p, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, niters=%p, d_info=%p)\n",
        handle, gesv_irs_params, gesv_irs_infos, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, niters, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverDnIRSParams_t gesv_irs_params_native;
    cusolverDnIRSInfos_t gesv_irs_infos_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    void * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    void * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    void * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t niters_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    gesv_irs_params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, gesv_irs_params);
    gesv_irs_infos_native = (cusolverDnIRSInfos_t)getNativePointerValue(env, gesv_irs_infos);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (void *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (void *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (void *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // niters is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSXgesv(handle_native, gesv_irs_params_native, gesv_irs_infos_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &niters_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // gesv_irs_params is read-only
    // gesv_irs_infos is read-only
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, niters, 0, (jint)niters_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSXgesv_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jobject params, jint n, jint nrhs, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnIRSXgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnIRSXgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    // nrhs is primitive
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnIRSXgesv_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSXgesv_bufferSize(handle=%p, params=%p, n=%d, nrhs=%d, lwork_bytes=%p)\n",
        handle, params, n, nrhs, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverDnIRSParams_t params_native;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, params);
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSXgesv_bufferSize(handle_native, params_native, n_native, nrhs_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // params is read-only
    // n is primitive
    // nrhs is primitive
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSXgelsNative(JNIEnv *env, jclass cls, jobject handle, jobject gels_irs_params, jobject gels_irs_infos, jint m, jint n, jint nrhs, jobject dA, jint ldda, jobject dB, jint lddb, jobject dX, jint lddx, jobject dWorkspace, jlong lwork_bytes, jintArray niters, jobject d_info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnIRSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (gels_irs_params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'gels_irs_params' is null for cusolverDnIRSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (gels_irs_infos == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'gels_irs_infos' is null for cusolverDnIRSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (dA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dA' is null for cusolverDnIRSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldda is primitive
    if (dB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dB' is null for cusolverDnIRSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddb is primitive
    if (dX == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dX' is null for cusolverDnIRSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lddx is primitive
    if (dWorkspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dWorkspace' is null for cusolverDnIRSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork_bytes is primitive
    if (niters == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'niters' is null for cusolverDnIRSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnIRSXgels");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSXgels(handle=%p, gels_irs_params=%p, gels_irs_infos=%p, m=%d, n=%d, nrhs=%d, dA=%p, ldda=%d, dB=%p, lddb=%d, dX=%p, lddx=%d, dWorkspace=%p, lwork_bytes=%ld, niters=%p, d_info=%p)\n",
        handle, gels_irs_params, gels_irs_infos, m, n, nrhs, dA, ldda, dB, lddb, dX, lddx, dWorkspace, lwork_bytes, niters, d_info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverDnIRSParams_t gels_irs_params_native;
    cusolverDnIRSInfos_t gels_irs_infos_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    void * dA_native = NULL;
    cusolver_int_t ldda_native = 0;
    void * dB_native = NULL;
    cusolver_int_t lddb_native = 0;
    void * dX_native = NULL;
    cusolver_int_t lddx_native = 0;
    void * dWorkspace_native = NULL;
    size_t lwork_bytes_native = 0;
    cusolver_int_t niters_native;
    cusolver_int_t * d_info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    gels_irs_params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, gels_irs_params);
    gels_irs_infos_native = (cusolverDnIRSInfos_t)getNativePointerValue(env, gels_irs_infos);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    dA_native = (void *)getPointer(env, dA);
    ldda_native = (cusolver_int_t)ldda;
    dB_native = (void *)getPointer(env, dB);
    lddb_native = (cusolver_int_t)lddb;
    dX_native = (void *)getPointer(env, dX);
    lddx_native = (cusolver_int_t)lddx;
    dWorkspace_native = (void *)getPointer(env, dWorkspace);
    lwork_bytes_native = (size_t)lwork_bytes;
    // niters is write-only
    d_info_native = (cusolver_int_t *)getPointer(env, d_info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSXgels(handle_native, gels_irs_params_native, gels_irs_infos_native, m_native, n_native, nrhs_native, dA_native, ldda_native, dB_native, lddb_native, dX_native, lddx_native, dWorkspace_native, lwork_bytes_native, &niters_native, d_info_native);

    // Write back native variable values
    // handle is read-only
    // gels_irs_params is read-only
    // gels_irs_infos is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    // dA is a native pointer
    // ldda is primitive
    // dB is a native pointer
    // lddb is primitive
    // dX is a native pointer
    // lddx is primitive
    // dWorkspace is a native pointer
    // lwork_bytes is primitive
    if (!set(env, niters, 0, (jint)niters_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // d_info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnIRSXgels_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jobject params, jint m, jint n, jint nrhs, jlongArray lwork_bytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnIRSXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnIRSXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (lwork_bytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork_bytes' is null for cusolverDnIRSXgels_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnIRSXgels_bufferSize(handle=%p, params=%p, m=%d, n=%d, nrhs=%d, lwork_bytes=%p)\n",
        handle, params, m, n, nrhs, lwork_bytes);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverDnIRSParams_t params_native;
    cusolver_int_t m_native = 0;
    cusolver_int_t n_native = 0;
    cusolver_int_t nrhs_native = 0;
    size_t lwork_bytes_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    params_native = (cusolverDnIRSParams_t)getNativePointerValue(env, params);
    m_native = (cusolver_int_t)m;
    n_native = (cusolver_int_t)n;
    nrhs_native = (cusolver_int_t)nrhs;
    // lwork_bytes is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnIRSXgels_bufferSize(handle_native, params_native, m_native, n_native, nrhs_native, &lwork_bytes_native);

    // Write back native variable values
    // handle is read-only
    // params is read-only
    // m is primitive
    // n is primitive
    // nrhs is primitive
    if (!set(env, lwork_bytes, 0, (jlong)lwork_bytes_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/*******************************************************************************/
/** Cholesky factorization and its solver */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSpotrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jintArray Lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSpotrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSpotrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Lwork' is null for cusolverDnSpotrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSpotrf_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, Lwork=%p)\n",
        handle, uplo, n, A, lda, Lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    int Lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    // Lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSpotrf_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, &Lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, Lwork, 0, (jint)Lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDpotrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jintArray Lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDpotrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDpotrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Lwork' is null for cusolverDnDpotrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDpotrf_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, Lwork=%p)\n",
        handle, uplo, n, A, lda, Lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    int Lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    // Lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDpotrf_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, &Lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, Lwork, 0, (jint)Lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCpotrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jintArray Lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCpotrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCpotrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Lwork' is null for cusolverDnCpotrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCpotrf_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, Lwork=%p)\n",
        handle, uplo, n, A, lda, Lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    int Lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    // Lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCpotrf_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, &Lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, Lwork, 0, (jint)Lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZpotrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jintArray Lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZpotrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZpotrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Lwork' is null for cusolverDnZpotrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZpotrf_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, Lwork=%p)\n",
        handle, uplo, n, A, lda, Lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    int Lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    // Lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZpotrf_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, &Lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, Lwork, 0, (jint)Lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSpotrfNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject Workspace, jint Lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Workspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Workspace' is null for cusolverDnSpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // Lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnSpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSpotrf(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, Workspace=%p, Lwork=%d, devInfo=%p)\n",
        handle, uplo, n, A, lda, Workspace, Lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * Workspace_native = NULL;
    int Lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    Workspace_native = (float *)getPointer(env, Workspace);
    Lwork_native = (int)Lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSpotrf(handle_native, uplo_native, n_native, A_native, lda_native, Workspace_native, Lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // Workspace is a native pointer
    // Lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDpotrfNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject Workspace, jint Lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Workspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Workspace' is null for cusolverDnDpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // Lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnDpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDpotrf(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, Workspace=%p, Lwork=%d, devInfo=%p)\n",
        handle, uplo, n, A, lda, Workspace, Lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * Workspace_native = NULL;
    int Lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    Workspace_native = (double *)getPointer(env, Workspace);
    Lwork_native = (int)Lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDpotrf(handle_native, uplo_native, n_native, A_native, lda_native, Workspace_native, Lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // Workspace is a native pointer
    // Lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCpotrfNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject Workspace, jint Lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Workspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Workspace' is null for cusolverDnCpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // Lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnCpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCpotrf(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, Workspace=%p, Lwork=%d, devInfo=%p)\n",
        handle, uplo, n, A, lda, Workspace, Lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * Workspace_native = NULL;
    int Lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    Workspace_native = (cuComplex *)getPointer(env, Workspace);
    Lwork_native = (int)Lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCpotrf(handle_native, uplo_native, n_native, A_native, lda_native, Workspace_native, Lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // Workspace is a native pointer
    // Lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZpotrfNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject Workspace, jint Lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Workspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Workspace' is null for cusolverDnZpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // Lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnZpotrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZpotrf(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, Workspace=%p, Lwork=%d, devInfo=%p)\n",
        handle, uplo, n, A, lda, Workspace, Lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * Workspace_native = NULL;
    int Lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    Workspace_native = (cuDoubleComplex *)getPointer(env, Workspace);
    Lwork_native = (int)Lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZpotrf(handle_native, uplo_native, n_native, A_native, lda_native, Workspace_native, Lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // Workspace is a native pointer
    // Lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSpotrsNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jint nrhs, jobject A, jint lda, jobject B, jint ldb, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnSpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnSpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSpotrs(handle=%p, uplo=%d, n=%d, nrhs=%d, A=%p, lda=%d, B=%p, ldb=%d, devInfo=%p)\n",
        handle, uplo, n, nrhs, A, lda, B, ldb, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    int nrhs_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * B_native = NULL;
    int ldb_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    nrhs_native = (int)nrhs;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (float *)getPointer(env, B);
    ldb_native = (int)ldb;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSpotrs(handle_native, uplo_native, n_native, nrhs_native, A_native, lda_native, B_native, ldb_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDpotrsNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jint nrhs, jobject A, jint lda, jobject B, jint ldb, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnDpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnDpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDpotrs(handle=%p, uplo=%d, n=%d, nrhs=%d, A=%p, lda=%d, B=%p, ldb=%d, devInfo=%p)\n",
        handle, uplo, n, nrhs, A, lda, B, ldb, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    int nrhs_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * B_native = NULL;
    int ldb_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    nrhs_native = (int)nrhs;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (double *)getPointer(env, B);
    ldb_native = (int)ldb;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDpotrs(handle_native, uplo_native, n_native, nrhs_native, A_native, lda_native, B_native, ldb_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCpotrsNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jint nrhs, jobject A, jint lda, jobject B, jint ldb, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnCpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnCpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCpotrs(handle=%p, uplo=%d, n=%d, nrhs=%d, A=%p, lda=%d, B=%p, ldb=%d, devInfo=%p)\n",
        handle, uplo, n, nrhs, A, lda, B, ldb, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    int nrhs_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * B_native = NULL;
    int ldb_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    nrhs_native = (int)nrhs;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCpotrs(handle_native, uplo_native, n_native, nrhs_native, A_native, lda_native, B_native, ldb_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZpotrsNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jint nrhs, jobject A, jint lda, jobject B, jint ldb, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnZpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnZpotrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZpotrs(handle=%p, uplo=%d, n=%d, nrhs=%d, A=%p, lda=%d, B=%p, ldb=%d, devInfo=%p)\n",
        handle, uplo, n, nrhs, A, lda, B, ldb, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    int nrhs_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * B_native = NULL;
    int ldb_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    nrhs_native = (int)nrhs;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuDoubleComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZpotrs(handle_native, uplo_native, n_native, nrhs_native, A_native, lda_native, B_native, ldb_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** batched Cholesky factorization and its solver */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSpotrfBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject Aarray, jint lda, jobject infoArray, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSpotrfBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (Aarray == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Aarray' is null for cusolverDnSpotrfBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (infoArray == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'infoArray' is null for cusolverDnSpotrfBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSpotrfBatched(handle=%p, uplo=%d, n=%d, Aarray=%p, lda=%d, infoArray=%p, batchSize=%d)\n",
        handle, uplo, n, Aarray, lda, infoArray, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float** Aarray_native = NULL;
    int lda_native = 0;
    int * infoArray_native = NULL;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    Aarray_native = (float **)getPointer(env, Aarray);
    lda_native = (int)lda;
    infoArray_native = (int *)getPointer(env, infoArray);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSpotrfBatched(handle_native, uplo_native, n_native, Aarray_native, lda_native, infoArray_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // Aarray is a native pointer
    // lda is primitive
    // infoArray is a native pointer
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDpotrfBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject Aarray, jint lda, jobject infoArray, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDpotrfBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (Aarray == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Aarray' is null for cusolverDnDpotrfBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (infoArray == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'infoArray' is null for cusolverDnDpotrfBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDpotrfBatched(handle=%p, uplo=%d, n=%d, Aarray=%p, lda=%d, infoArray=%p, batchSize=%d)\n",
        handle, uplo, n, Aarray, lda, infoArray, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double** Aarray_native = NULL;
    int lda_native = 0;
    int * infoArray_native = NULL;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    Aarray_native = (double **)getPointer(env, Aarray);
    lda_native = (int)lda;
    infoArray_native = (int *)getPointer(env, infoArray);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDpotrfBatched(handle_native, uplo_native, n_native, Aarray_native, lda_native, infoArray_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // Aarray is a native pointer
    // lda is primitive
    // infoArray is a native pointer
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCpotrfBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject Aarray, jint lda, jobject infoArray, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCpotrfBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (Aarray == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Aarray' is null for cusolverDnCpotrfBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (infoArray == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'infoArray' is null for cusolverDnCpotrfBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCpotrfBatched(handle=%p, uplo=%d, n=%d, Aarray=%p, lda=%d, infoArray=%p, batchSize=%d)\n",
        handle, uplo, n, Aarray, lda, infoArray, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex** Aarray_native = NULL;
    int lda_native = 0;
    int * infoArray_native = NULL;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    Aarray_native = (cuComplex **)getPointer(env, Aarray);
    lda_native = (int)lda;
    infoArray_native = (int *)getPointer(env, infoArray);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCpotrfBatched(handle_native, uplo_native, n_native, Aarray_native, lda_native, infoArray_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // Aarray is a native pointer
    // lda is primitive
    // infoArray is a native pointer
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZpotrfBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject Aarray, jint lda, jobject infoArray, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZpotrfBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (Aarray == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Aarray' is null for cusolverDnZpotrfBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (infoArray == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'infoArray' is null for cusolverDnZpotrfBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZpotrfBatched(handle=%p, uplo=%d, n=%d, Aarray=%p, lda=%d, infoArray=%p, batchSize=%d)\n",
        handle, uplo, n, Aarray, lda, infoArray, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex** Aarray_native = NULL;
    int lda_native = 0;
    int * infoArray_native = NULL;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    Aarray_native = (cuDoubleComplex **)getPointer(env, Aarray);
    lda_native = (int)lda;
    infoArray_native = (int *)getPointer(env, infoArray);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZpotrfBatched(handle_native, uplo_native, n_native, Aarray_native, lda_native, infoArray_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // Aarray is a native pointer
    // lda is primitive
    // infoArray is a native pointer
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSpotrsBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jint nrhs, jobject A, jint lda, jobject B, jint ldb, jobject d_info, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnSpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnSpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSpotrsBatched(handle=%p, uplo=%d, n=%d, nrhs=%d, A=%p, lda=%d, B=%p, ldb=%d, d_info=%p, batchSize=%d)\n",
        handle, uplo, n, nrhs, A, lda, B, ldb, d_info, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    int nrhs_native = 0;
    float** A_native = NULL;
    int lda_native = 0;
    float** B_native = NULL;
    int ldb_native = 0;
    int * d_info_native = NULL;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    nrhs_native = (int)nrhs;
    A_native = (float **)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (float **)getPointer(env, B);
    ldb_native = (int)ldb;
    d_info_native = (int *)getPointer(env, d_info);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSpotrsBatched(handle_native, uplo_native, n_native, nrhs_native, A_native, lda_native, B_native, ldb_native, d_info_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // d_info is a native pointer
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDpotrsBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jint nrhs, jobject A, jint lda, jobject B, jint ldb, jobject d_info, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnDpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnDpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDpotrsBatched(handle=%p, uplo=%d, n=%d, nrhs=%d, A=%p, lda=%d, B=%p, ldb=%d, d_info=%p, batchSize=%d)\n",
        handle, uplo, n, nrhs, A, lda, B, ldb, d_info, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    int nrhs_native = 0;
    double** A_native = NULL;
    int lda_native = 0;
    double** B_native = NULL;
    int ldb_native = 0;
    int * d_info_native = NULL;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    nrhs_native = (int)nrhs;
    A_native = (double **)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (double **)getPointer(env, B);
    ldb_native = (int)ldb;
    d_info_native = (int *)getPointer(env, d_info);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDpotrsBatched(handle_native, uplo_native, n_native, nrhs_native, A_native, lda_native, B_native, ldb_native, d_info_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // d_info is a native pointer
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCpotrsBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jint nrhs, jobject A, jint lda, jobject B, jint ldb, jobject d_info, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnCpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnCpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCpotrsBatched(handle=%p, uplo=%d, n=%d, nrhs=%d, A=%p, lda=%d, B=%p, ldb=%d, d_info=%p, batchSize=%d)\n",
        handle, uplo, n, nrhs, A, lda, B, ldb, d_info, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    int nrhs_native = 0;
    cuComplex** A_native = NULL;
    int lda_native = 0;
    cuComplex** B_native = NULL;
    int ldb_native = 0;
    int * d_info_native = NULL;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    nrhs_native = (int)nrhs;
    A_native = (cuComplex **)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuComplex **)getPointer(env, B);
    ldb_native = (int)ldb;
    d_info_native = (int *)getPointer(env, d_info);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCpotrsBatched(handle_native, uplo_native, n_native, nrhs_native, A_native, lda_native, B_native, ldb_native, d_info_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // d_info is a native pointer
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZpotrsBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jint nrhs, jobject A, jint lda, jobject B, jint ldb, jobject d_info, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnZpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (d_info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd_info' is null for cusolverDnZpotrsBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZpotrsBatched(handle=%p, uplo=%d, n=%d, nrhs=%d, A=%p, lda=%d, B=%p, ldb=%d, d_info=%p, batchSize=%d)\n",
        handle, uplo, n, nrhs, A, lda, B, ldb, d_info, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    int nrhs_native = 0;
    cuDoubleComplex** A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex** B_native = NULL;
    int ldb_native = 0;
    int * d_info_native = NULL;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    nrhs_native = (int)nrhs;
    A_native = (cuDoubleComplex **)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuDoubleComplex **)getPointer(env, B);
    ldb_native = (int)ldb;
    d_info_native = (int *)getPointer(env, d_info);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZpotrsBatched(handle_native, uplo_native, n_native, nrhs_native, A_native, lda_native, B_native, ldb_native, d_info_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // d_info is a native pointer
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** s.p.d. matrix inversion (POTRI) and auxiliary routines (TRTRI and LAUUM) */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSpotri_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSpotri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSpotri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSpotri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSpotri_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, uplo, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSpotri_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDpotri_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDpotri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDpotri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDpotri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDpotri_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, uplo, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDpotri_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCpotri_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCpotri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCpotri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCpotri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCpotri_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, uplo, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCpotri_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZpotri_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZpotri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZpotri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZpotri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZpotri_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, uplo, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZpotri_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSpotriNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject work, jint lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnSpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSpotri(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, work=%p, lwork=%d, devInfo=%p)\n",
        handle, uplo, n, A, lda, work, lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * work_native = NULL;
    int lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSpotri(handle_native, uplo_native, n_native, A_native, lda_native, work_native, lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // work is a native pointer
    // lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDpotriNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject work, jint lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnDpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDpotri(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, work=%p, lwork=%d, devInfo=%p)\n",
        handle, uplo, n, A, lda, work, lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * work_native = NULL;
    int lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDpotri(handle_native, uplo_native, n_native, A_native, lda_native, work_native, lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // work is a native pointer
    // lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCpotriNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject work, jint lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnCpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnCpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCpotri(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, work=%p, lwork=%d, devInfo=%p)\n",
        handle, uplo, n, A, lda, work, lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCpotri(handle_native, uplo_native, n_native, A_native, lda_native, work_native, lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // work is a native pointer
    // lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZpotriNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject work, jint lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnZpotri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZpotri(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, work=%p, lwork=%d, devInfo=%p)\n",
        handle, uplo, n, A, lda, work, lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZpotri(handle_native, uplo_native, n_native, A_native, lda_native, work_native, lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // work is a native pointer
    // lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnXtrtri_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint diag, jlong n, jint dataTypeA, jobject A, jlong lda, jlongArray workspaceInBytesOnDevice, jlongArray workspaceInBytesOnHost)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnXtrtri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // diag is primitive
    // n is primitive
    // dataTypeA is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnXtrtri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (workspaceInBytesOnDevice == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'workspaceInBytesOnDevice' is null for cusolverDnXtrtri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (workspaceInBytesOnHost == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'workspaceInBytesOnHost' is null for cusolverDnXtrtri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnXtrtri_bufferSize(handle=%p, uplo=%d, diag=%d, n=%ld, dataTypeA=%d, A=%p, lda=%ld, workspaceInBytesOnDevice=%p, workspaceInBytesOnHost=%p)\n",
        handle, uplo, diag, n, dataTypeA, A, lda, workspaceInBytesOnDevice, workspaceInBytesOnHost);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    cublasDiagType_t diag_native;
    int64_t n_native = 0;
    cudaDataType dataTypeA_native;
    void * A_native = NULL;
    int64_t lda_native = 0;
    size_t * workspaceInBytesOnDevice_native = NULL;
    size_t * workspaceInBytesOnHost_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    diag_native = (cublasDiagType_t)diag;
    n_native = (int64_t)n;
    dataTypeA_native = (cudaDataType)dataTypeA;
    A_native = (void *)getPointer(env, A);
    lda_native = (int64_t)lda;
    workspaceInBytesOnDevice_native = (size_t *)getPointer(env, workspaceInBytesOnDevice);
    workspaceInBytesOnHost_native = (size_t *)getPointer(env, workspaceInBytesOnHost);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnXtrtri_bufferSize(handle_native, uplo_native, diag_native, n_native, dataTypeA_native, A_native, lda_native, workspaceInBytesOnDevice_native, workspaceInBytesOnHost_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // diag is primitive
    // n is primitive
    // dataTypeA is primitive
    // A is a native pointer
    // lda is primitive
    // workspaceInBytesOnDevice is a native pointer
    // workspaceInBytesOnHost is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnXtrtriNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint diag, jlong n, jint dataTypeA, jobject A, jlong lda, jobject bufferOnDevice, jlong workspaceInBytesOnDevice, jobject bufferOnHost, jlong workspaceInBytesOnHost, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnXtrtri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // diag is primitive
    // n is primitive
    // dataTypeA is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnXtrtri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (bufferOnDevice == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferOnDevice' is null for cusolverDnXtrtri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // workspaceInBytesOnDevice is primitive
    if (bufferOnHost == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferOnHost' is null for cusolverDnXtrtri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // workspaceInBytesOnHost is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnXtrtri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnXtrtri(handle=%p, uplo=%d, diag=%d, n=%ld, dataTypeA=%d, A=%p, lda=%ld, bufferOnDevice=%p, workspaceInBytesOnDevice=%ld, bufferOnHost=%p, workspaceInBytesOnHost=%ld, devInfo=%p)\n",
        handle, uplo, diag, n, dataTypeA, A, lda, bufferOnDevice, workspaceInBytesOnDevice, bufferOnHost, workspaceInBytesOnHost, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    cublasDiagType_t diag_native;
    int64_t n_native = 0;
    cudaDataType dataTypeA_native;
    void * A_native = NULL;
    int64_t lda_native = 0;
    void * bufferOnDevice_native = NULL;
    size_t workspaceInBytesOnDevice_native = 0;
    void * bufferOnHost_native = NULL;
    size_t workspaceInBytesOnHost_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    diag_native = (cublasDiagType_t)diag;
    n_native = (int64_t)n;
    dataTypeA_native = (cudaDataType)dataTypeA;
    A_native = (void *)getPointer(env, A);
    lda_native = (int64_t)lda;
    bufferOnDevice_native = (void *)getPointer(env, bufferOnDevice);
    workspaceInBytesOnDevice_native = (size_t)workspaceInBytesOnDevice;
    bufferOnHost_native = (void *)getPointer(env, bufferOnHost);
    workspaceInBytesOnHost_native = (size_t)workspaceInBytesOnHost;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnXtrtri(handle_native, uplo_native, diag_native, n_native, dataTypeA_native, A_native, lda_native, bufferOnDevice_native, workspaceInBytesOnDevice_native, bufferOnHost_native, workspaceInBytesOnHost_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // diag is primitive
    // n is primitive
    // dataTypeA is primitive
    // A is a native pointer
    // lda is primitive
    // bufferOnDevice is a native pointer
    // workspaceInBytesOnDevice is primitive
    // bufferOnHost is a native pointer
    // workspaceInBytesOnHost is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** lauum, auxiliar routine for s.p.d matrix inversion */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSlauum_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSlauum_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSlauum_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSlauum_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSlauum_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, uplo, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSlauum_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDlauum_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDlauum_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDlauum_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDlauum_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDlauum_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, uplo, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDlauum_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnClauum_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnClauum_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnClauum_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnClauum_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnClauum_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, uplo, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnClauum_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZlauum_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZlauum_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZlauum_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZlauum_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZlauum_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, uplo, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZlauum_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSlauumNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject work, jint lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSlauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSlauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSlauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnSlauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSlauum(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, work=%p, lwork=%d, devInfo=%p)\n",
        handle, uplo, n, A, lda, work, lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * work_native = NULL;
    int lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSlauum(handle_native, uplo_native, n_native, A_native, lda_native, work_native, lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // work is a native pointer
    // lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDlauumNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject work, jint lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDlauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDlauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDlauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnDlauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDlauum(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, work=%p, lwork=%d, devInfo=%p)\n",
        handle, uplo, n, A, lda, work, lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * work_native = NULL;
    int lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDlauum(handle_native, uplo_native, n_native, A_native, lda_native, work_native, lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // work is a native pointer
    // lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnClauumNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject work, jint lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnClauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnClauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnClauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnClauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnClauum(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, work=%p, lwork=%d, devInfo=%p)\n",
        handle, uplo, n, A, lda, work, lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnClauum(handle_native, uplo_native, n_native, A_native, lda_native, work_native, lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // work is a native pointer
    // lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZlauumNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject work, jint lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZlauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZlauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZlauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnZlauum");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZlauum(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, work=%p, lwork=%d, devInfo=%p)\n",
        handle, uplo, n, A, lda, work, lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZlauum(handle_native, uplo_native, n_native, A_native, lda_native, work_native, lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // work is a native pointer
    // lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** LU Factorization */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSgetrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jintArray Lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSgetrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSgetrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Lwork' is null for cusolverDnSgetrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSgetrf_bufferSize(handle=%p, m=%d, n=%d, A=%p, lda=%d, Lwork=%p)\n",
        handle, m, n, A, lda, Lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    int Lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    // Lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSgetrf_bufferSize(handle_native, m_native, n_native, A_native, lda_native, &Lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, Lwork, 0, (jint)Lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDgetrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jintArray Lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDgetrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDgetrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Lwork' is null for cusolverDnDgetrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDgetrf_bufferSize(handle=%p, m=%d, n=%d, A=%p, lda=%d, Lwork=%p)\n",
        handle, m, n, A, lda, Lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    int Lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    // Lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDgetrf_bufferSize(handle_native, m_native, n_native, A_native, lda_native, &Lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, Lwork, 0, (jint)Lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCgetrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jintArray Lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCgetrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCgetrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Lwork' is null for cusolverDnCgetrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCgetrf_bufferSize(handle=%p, m=%d, n=%d, A=%p, lda=%d, Lwork=%p)\n",
        handle, m, n, A, lda, Lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    int Lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    // Lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCgetrf_bufferSize(handle_native, m_native, n_native, A_native, lda_native, &Lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, Lwork, 0, (jint)Lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZgetrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jintArray Lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZgetrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZgetrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Lwork' is null for cusolverDnZgetrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZgetrf_bufferSize(handle=%p, m=%d, n=%d, A=%p, lda=%d, Lwork=%p)\n",
        handle, m, n, A, lda, Lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    int Lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    // Lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZgetrf_bufferSize(handle_native, m_native, n_native, A_native, lda_native, &Lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, Lwork, 0, (jint)Lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSgetrfNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jobject Workspace, jobject devIpiv, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Workspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Workspace' is null for cusolverDnSgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (devIpiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devIpiv' is null for cusolverDnSgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnSgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSgetrf(handle=%p, m=%d, n=%d, A=%p, lda=%d, Workspace=%p, devIpiv=%p, devInfo=%p)\n",
        handle, m, n, A, lda, Workspace, devIpiv, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * Workspace_native = NULL;
    int * devIpiv_native = NULL;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    Workspace_native = (float *)getPointer(env, Workspace);
    devIpiv_native = (int *)getPointer(env, devIpiv);
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSgetrf(handle_native, m_native, n_native, A_native, lda_native, Workspace_native, devIpiv_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // Workspace is a native pointer
    // devIpiv is a native pointer
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDgetrfNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jobject Workspace, jobject devIpiv, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Workspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Workspace' is null for cusolverDnDgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (devIpiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devIpiv' is null for cusolverDnDgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnDgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDgetrf(handle=%p, m=%d, n=%d, A=%p, lda=%d, Workspace=%p, devIpiv=%p, devInfo=%p)\n",
        handle, m, n, A, lda, Workspace, devIpiv, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * Workspace_native = NULL;
    int * devIpiv_native = NULL;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    Workspace_native = (double *)getPointer(env, Workspace);
    devIpiv_native = (int *)getPointer(env, devIpiv);
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDgetrf(handle_native, m_native, n_native, A_native, lda_native, Workspace_native, devIpiv_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // Workspace is a native pointer
    // devIpiv is a native pointer
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCgetrfNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jobject Workspace, jobject devIpiv, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Workspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Workspace' is null for cusolverDnCgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (devIpiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devIpiv' is null for cusolverDnCgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnCgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCgetrf(handle=%p, m=%d, n=%d, A=%p, lda=%d, Workspace=%p, devIpiv=%p, devInfo=%p)\n",
        handle, m, n, A, lda, Workspace, devIpiv, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * Workspace_native = NULL;
    int * devIpiv_native = NULL;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    Workspace_native = (cuComplex *)getPointer(env, Workspace);
    devIpiv_native = (int *)getPointer(env, devIpiv);
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCgetrf(handle_native, m_native, n_native, A_native, lda_native, Workspace_native, devIpiv_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // Workspace is a native pointer
    // devIpiv is a native pointer
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZgetrfNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jobject Workspace, jobject devIpiv, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (Workspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Workspace' is null for cusolverDnZgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (devIpiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devIpiv' is null for cusolverDnZgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnZgetrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZgetrf(handle=%p, m=%d, n=%d, A=%p, lda=%d, Workspace=%p, devIpiv=%p, devInfo=%p)\n",
        handle, m, n, A, lda, Workspace, devIpiv, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * Workspace_native = NULL;
    int * devIpiv_native = NULL;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    Workspace_native = (cuDoubleComplex *)getPointer(env, Workspace);
    devIpiv_native = (int *)getPointer(env, devIpiv);
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZgetrf(handle_native, m_native, n_native, A_native, lda_native, Workspace_native, devIpiv_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // Workspace is a native pointer
    // devIpiv is a native pointer
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Row pivoting */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSlaswpNative(JNIEnv *env, jclass cls, jobject handle, jint n, jobject A, jint lda, jint k1, jint k2, jobject devIpiv, jint incx)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSlaswp");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSlaswp");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // k1 is primitive
    // k2 is primitive
    if (devIpiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devIpiv' is null for cusolverDnSlaswp");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // incx is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSlaswp(handle=%p, n=%d, A=%p, lda=%d, k1=%d, k2=%d, devIpiv=%p, incx=%d)\n",
        handle, n, A, lda, k1, k2, devIpiv, incx);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    int k1_native = 0;
    int k2_native = 0;
    int * devIpiv_native = NULL;
    int incx_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    k1_native = (int)k1;
    k2_native = (int)k2;
    devIpiv_native = (int *)getPointer(env, devIpiv);
    incx_native = (int)incx;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSlaswp(handle_native, n_native, A_native, lda_native, k1_native, k2_native, devIpiv_native, incx_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // k1 is primitive
    // k2 is primitive
    // devIpiv is a native pointer
    // incx is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDlaswpNative(JNIEnv *env, jclass cls, jobject handle, jint n, jobject A, jint lda, jint k1, jint k2, jobject devIpiv, jint incx)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDlaswp");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDlaswp");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // k1 is primitive
    // k2 is primitive
    if (devIpiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devIpiv' is null for cusolverDnDlaswp");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // incx is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDlaswp(handle=%p, n=%d, A=%p, lda=%d, k1=%d, k2=%d, devIpiv=%p, incx=%d)\n",
        handle, n, A, lda, k1, k2, devIpiv, incx);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    int k1_native = 0;
    int k2_native = 0;
    int * devIpiv_native = NULL;
    int incx_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    k1_native = (int)k1;
    k2_native = (int)k2;
    devIpiv_native = (int *)getPointer(env, devIpiv);
    incx_native = (int)incx;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDlaswp(handle_native, n_native, A_native, lda_native, k1_native, k2_native, devIpiv_native, incx_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // k1 is primitive
    // k2 is primitive
    // devIpiv is a native pointer
    // incx is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnClaswpNative(JNIEnv *env, jclass cls, jobject handle, jint n, jobject A, jint lda, jint k1, jint k2, jobject devIpiv, jint incx)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnClaswp");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnClaswp");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // k1 is primitive
    // k2 is primitive
    if (devIpiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devIpiv' is null for cusolverDnClaswp");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // incx is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnClaswp(handle=%p, n=%d, A=%p, lda=%d, k1=%d, k2=%d, devIpiv=%p, incx=%d)\n",
        handle, n, A, lda, k1, k2, devIpiv, incx);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    int k1_native = 0;
    int k2_native = 0;
    int * devIpiv_native = NULL;
    int incx_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    k1_native = (int)k1;
    k2_native = (int)k2;
    devIpiv_native = (int *)getPointer(env, devIpiv);
    incx_native = (int)incx;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnClaswp(handle_native, n_native, A_native, lda_native, k1_native, k2_native, devIpiv_native, incx_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // k1 is primitive
    // k2 is primitive
    // devIpiv is a native pointer
    // incx is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZlaswpNative(JNIEnv *env, jclass cls, jobject handle, jint n, jobject A, jint lda, jint k1, jint k2, jobject devIpiv, jint incx)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZlaswp");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZlaswp");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // k1 is primitive
    // k2 is primitive
    if (devIpiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devIpiv' is null for cusolverDnZlaswp");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // incx is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZlaswp(handle=%p, n=%d, A=%p, lda=%d, k1=%d, k2=%d, devIpiv=%p, incx=%d)\n",
        handle, n, A, lda, k1, k2, devIpiv, incx);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    int k1_native = 0;
    int k2_native = 0;
    int * devIpiv_native = NULL;
    int incx_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    k1_native = (int)k1;
    k2_native = (int)k2;
    devIpiv_native = (int *)getPointer(env, devIpiv);
    incx_native = (int)incx;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZlaswp(handle_native, n_native, A_native, lda_native, k1_native, k2_native, devIpiv_native, incx_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // k1 is primitive
    // k2 is primitive
    // devIpiv is a native pointer
    // incx is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** LU solve */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSgetrsNative(JNIEnv *env, jclass cls, jobject handle, jint trans, jint n, jint nrhs, jobject A, jint lda, jobject devIpiv, jobject B, jint ldb, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    // n is primitive
    // nrhs is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (devIpiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devIpiv' is null for cusolverDnSgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnSgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnSgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSgetrs(handle=%p, trans=%d, n=%d, nrhs=%d, A=%p, lda=%d, devIpiv=%p, B=%p, ldb=%d, devInfo=%p)\n",
        handle, trans, n, nrhs, A, lda, devIpiv, B, ldb, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasOperation_t trans_native;
    int n_native = 0;
    int nrhs_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    int * devIpiv_native = NULL;
    float * B_native = NULL;
    int ldb_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    trans_native = (cublasOperation_t)trans;
    n_native = (int)n;
    nrhs_native = (int)nrhs;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    devIpiv_native = (int *)getPointer(env, devIpiv);
    B_native = (float *)getPointer(env, B);
    ldb_native = (int)ldb;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSgetrs(handle_native, trans_native, n_native, nrhs_native, A_native, lda_native, devIpiv_native, B_native, ldb_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // trans is primitive
    // n is primitive
    // nrhs is primitive
    // A is a native pointer
    // lda is primitive
    // devIpiv is a native pointer
    // B is a native pointer
    // ldb is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDgetrsNative(JNIEnv *env, jclass cls, jobject handle, jint trans, jint n, jint nrhs, jobject A, jint lda, jobject devIpiv, jobject B, jint ldb, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    // n is primitive
    // nrhs is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (devIpiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devIpiv' is null for cusolverDnDgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnDgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnDgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDgetrs(handle=%p, trans=%d, n=%d, nrhs=%d, A=%p, lda=%d, devIpiv=%p, B=%p, ldb=%d, devInfo=%p)\n",
        handle, trans, n, nrhs, A, lda, devIpiv, B, ldb, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasOperation_t trans_native;
    int n_native = 0;
    int nrhs_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    int * devIpiv_native = NULL;
    double * B_native = NULL;
    int ldb_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    trans_native = (cublasOperation_t)trans;
    n_native = (int)n;
    nrhs_native = (int)nrhs;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    devIpiv_native = (int *)getPointer(env, devIpiv);
    B_native = (double *)getPointer(env, B);
    ldb_native = (int)ldb;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDgetrs(handle_native, trans_native, n_native, nrhs_native, A_native, lda_native, devIpiv_native, B_native, ldb_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // trans is primitive
    // n is primitive
    // nrhs is primitive
    // A is a native pointer
    // lda is primitive
    // devIpiv is a native pointer
    // B is a native pointer
    // ldb is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCgetrsNative(JNIEnv *env, jclass cls, jobject handle, jint trans, jint n, jint nrhs, jobject A, jint lda, jobject devIpiv, jobject B, jint ldb, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    // n is primitive
    // nrhs is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (devIpiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devIpiv' is null for cusolverDnCgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnCgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnCgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCgetrs(handle=%p, trans=%d, n=%d, nrhs=%d, A=%p, lda=%d, devIpiv=%p, B=%p, ldb=%d, devInfo=%p)\n",
        handle, trans, n, nrhs, A, lda, devIpiv, B, ldb, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasOperation_t trans_native;
    int n_native = 0;
    int nrhs_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    int * devIpiv_native = NULL;
    cuComplex * B_native = NULL;
    int ldb_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    trans_native = (cublasOperation_t)trans;
    n_native = (int)n;
    nrhs_native = (int)nrhs;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    devIpiv_native = (int *)getPointer(env, devIpiv);
    B_native = (cuComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCgetrs(handle_native, trans_native, n_native, nrhs_native, A_native, lda_native, devIpiv_native, B_native, ldb_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // trans is primitive
    // n is primitive
    // nrhs is primitive
    // A is a native pointer
    // lda is primitive
    // devIpiv is a native pointer
    // B is a native pointer
    // ldb is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZgetrsNative(JNIEnv *env, jclass cls, jobject handle, jint trans, jint n, jint nrhs, jobject A, jint lda, jobject devIpiv, jobject B, jint ldb, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    // n is primitive
    // nrhs is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (devIpiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devIpiv' is null for cusolverDnZgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnZgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnZgetrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZgetrs(handle=%p, trans=%d, n=%d, nrhs=%d, A=%p, lda=%d, devIpiv=%p, B=%p, ldb=%d, devInfo=%p)\n",
        handle, trans, n, nrhs, A, lda, devIpiv, B, ldb, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasOperation_t trans_native;
    int n_native = 0;
    int nrhs_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    int * devIpiv_native = NULL;
    cuDoubleComplex * B_native = NULL;
    int ldb_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    trans_native = (cublasOperation_t)trans;
    n_native = (int)n;
    nrhs_native = (int)nrhs;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    devIpiv_native = (int *)getPointer(env, devIpiv);
    B_native = (cuDoubleComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZgetrs(handle_native, trans_native, n_native, nrhs_native, A_native, lda_native, devIpiv_native, B_native, ldb_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // trans is primitive
    // n is primitive
    // nrhs is primitive
    // A is a native pointer
    // lda is primitive
    // devIpiv is a native pointer
    // B is a native pointer
    // ldb is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** QR factorization */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSgeqrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSgeqrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSgeqrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSgeqrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSgeqrf_bufferSize(handle=%p, m=%d, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, m, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSgeqrf_bufferSize(handle_native, m_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDgeqrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDgeqrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDgeqrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDgeqrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDgeqrf_bufferSize(handle=%p, m=%d, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, m, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDgeqrf_bufferSize(handle_native, m_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCgeqrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCgeqrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCgeqrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCgeqrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCgeqrf_bufferSize(handle=%p, m=%d, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, m, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCgeqrf_bufferSize(handle_native, m_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZgeqrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZgeqrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZgeqrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZgeqrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZgeqrf_bufferSize(handle=%p, m=%d, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, m, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZgeqrf_bufferSize(handle_native, m_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSgeqrfNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jobject TAU, jobject Workspace, jint Lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (TAU == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'TAU' is null for cusolverDnSgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (Workspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Workspace' is null for cusolverDnSgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // Lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnSgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSgeqrf(handle=%p, m=%d, n=%d, A=%p, lda=%d, TAU=%p, Workspace=%p, Lwork=%d, devInfo=%p)\n",
        handle, m, n, A, lda, TAU, Workspace, Lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * TAU_native = NULL;
    float * Workspace_native = NULL;
    int Lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    TAU_native = (float *)getPointer(env, TAU);
    Workspace_native = (float *)getPointer(env, Workspace);
    Lwork_native = (int)Lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSgeqrf(handle_native, m_native, n_native, A_native, lda_native, TAU_native, Workspace_native, Lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // TAU is a native pointer
    // Workspace is a native pointer
    // Lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDgeqrfNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jobject TAU, jobject Workspace, jint Lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (TAU == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'TAU' is null for cusolverDnDgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (Workspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Workspace' is null for cusolverDnDgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // Lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnDgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDgeqrf(handle=%p, m=%d, n=%d, A=%p, lda=%d, TAU=%p, Workspace=%p, Lwork=%d, devInfo=%p)\n",
        handle, m, n, A, lda, TAU, Workspace, Lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * TAU_native = NULL;
    double * Workspace_native = NULL;
    int Lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    TAU_native = (double *)getPointer(env, TAU);
    Workspace_native = (double *)getPointer(env, Workspace);
    Lwork_native = (int)Lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDgeqrf(handle_native, m_native, n_native, A_native, lda_native, TAU_native, Workspace_native, Lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // TAU is a native pointer
    // Workspace is a native pointer
    // Lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCgeqrfNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jobject TAU, jobject Workspace, jint Lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (TAU == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'TAU' is null for cusolverDnCgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (Workspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Workspace' is null for cusolverDnCgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // Lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnCgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCgeqrf(handle=%p, m=%d, n=%d, A=%p, lda=%d, TAU=%p, Workspace=%p, Lwork=%d, devInfo=%p)\n",
        handle, m, n, A, lda, TAU, Workspace, Lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * TAU_native = NULL;
    cuComplex * Workspace_native = NULL;
    int Lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    TAU_native = (cuComplex *)getPointer(env, TAU);
    Workspace_native = (cuComplex *)getPointer(env, Workspace);
    Lwork_native = (int)Lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCgeqrf(handle_native, m_native, n_native, A_native, lda_native, TAU_native, Workspace_native, Lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // TAU is a native pointer
    // Workspace is a native pointer
    // Lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZgeqrfNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jobject TAU, jobject Workspace, jint Lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (TAU == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'TAU' is null for cusolverDnZgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (Workspace == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Workspace' is null for cusolverDnZgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // Lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnZgeqrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZgeqrf(handle=%p, m=%d, n=%d, A=%p, lda=%d, TAU=%p, Workspace=%p, Lwork=%d, devInfo=%p)\n",
        handle, m, n, A, lda, TAU, Workspace, Lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * TAU_native = NULL;
    cuDoubleComplex * Workspace_native = NULL;
    int Lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    TAU_native = (cuDoubleComplex *)getPointer(env, TAU);
    Workspace_native = (cuDoubleComplex *)getPointer(env, Workspace);
    Lwork_native = (int)Lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZgeqrf(handle_native, m_native, n_native, A_native, lda_native, TAU_native, Workspace_native, Lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // TAU is a native pointer
    // Workspace is a native pointer
    // Lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** generate unitary matrix Q from QR factorization */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSorgqr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSorgqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSorgqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnSorgqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSorgqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSorgqr_bufferSize(handle=%p, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, lwork=%p)\n",
        handle, m, n, k, A, lda, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (float *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSorgqr_bufferSize(handle_native, m_native, n_native, k_native, A_native, lda_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDorgqr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDorgqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDorgqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnDorgqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDorgqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDorgqr_bufferSize(handle=%p, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, lwork=%p)\n",
        handle, m, n, k, A, lda, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (double *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDorgqr_bufferSize(handle_native, m_native, n_native, k_native, A_native, lda_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCungqr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCungqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCungqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnCungqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCungqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCungqr_bufferSize(handle=%p, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, lwork=%p)\n",
        handle, m, n, k, A, lda, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuComplex *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCungqr_bufferSize(handle_native, m_native, n_native, k_native, A_native, lda_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZungqr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZungqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZungqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnZungqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZungqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZungqr_bufferSize(handle=%p, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, lwork=%p)\n",
        handle, m, n, k, A, lda, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuDoubleComplex *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZungqr_bufferSize(handle_native, m_native, n_native, k_native, A_native, lda_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSorgqrNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSorgqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSorgqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnSorgqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSorgqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSorgqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSorgqr(handle=%p, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, m, n, k, A, lda, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * tau_native = NULL;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (float *)getPointer(env, tau);
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSorgqr(handle_native, m_native, n_native, k_native, A_native, lda_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDorgqrNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDorgqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDorgqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnDorgqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDorgqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDorgqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDorgqr(handle=%p, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, m, n, k, A, lda, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * tau_native = NULL;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (double *)getPointer(env, tau);
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDorgqr(handle_native, m_native, n_native, k_native, A_native, lda_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCungqrNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCungqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCungqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnCungqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnCungqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnCungqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCungqr(handle=%p, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, m, n, k, A, lda, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * tau_native = NULL;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuComplex *)getPointer(env, tau);
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCungqr(handle_native, m_native, n_native, k_native, A_native, lda_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZungqrNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZungqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZungqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnZungqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZungqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZungqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZungqr(handle=%p, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, m, n, k, A, lda, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * tau_native = NULL;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuDoubleComplex *)getPointer(env, tau);
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZungqr(handle_native, m_native, n_native, k_native, A_native, lda_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** compute Q**T*b in solve min||A*x = b|| */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSormqr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint trans, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject C, jint ldc, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSormqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSormqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnSormqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnSormqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSormqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSormqr_bufferSize(handle=%p, side=%d, trans=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, lwork=%p)\n",
        handle, side, trans, m, n, k, A, lda, tau, C, ldc, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * tau_native = NULL;
    float * C_native = NULL;
    int ldc_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (float *)getPointer(env, tau);
    C_native = (float *)getPointer(env, C);
    ldc_native = (int)ldc;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSormqr_bufferSize(handle_native, side_native, trans_native, m_native, n_native, k_native, A_native, lda_native, tau_native, C_native, ldc_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDormqr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint trans, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject C, jint ldc, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDormqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDormqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnDormqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnDormqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDormqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDormqr_bufferSize(handle=%p, side=%d, trans=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, lwork=%p)\n",
        handle, side, trans, m, n, k, A, lda, tau, C, ldc, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * tau_native = NULL;
    double * C_native = NULL;
    int ldc_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (double *)getPointer(env, tau);
    C_native = (double *)getPointer(env, C);
    ldc_native = (int)ldc;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDormqr_bufferSize(handle_native, side_native, trans_native, m_native, n_native, k_native, A_native, lda_native, tau_native, C_native, ldc_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCunmqr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint trans, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject C, jint ldc, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCunmqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCunmqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnCunmqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnCunmqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCunmqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCunmqr_bufferSize(handle=%p, side=%d, trans=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, lwork=%p)\n",
        handle, side, trans, m, n, k, A, lda, tau, C, ldc, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * tau_native = NULL;
    cuComplex * C_native = NULL;
    int ldc_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuComplex *)getPointer(env, tau);
    C_native = (cuComplex *)getPointer(env, C);
    ldc_native = (int)ldc;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCunmqr_bufferSize(handle_native, side_native, trans_native, m_native, n_native, k_native, A_native, lda_native, tau_native, C_native, ldc_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZunmqr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint trans, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject C, jint ldc, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZunmqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZunmqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnZunmqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnZunmqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZunmqr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZunmqr_bufferSize(handle=%p, side=%d, trans=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, lwork=%p)\n",
        handle, side, trans, m, n, k, A, lda, tau, C, ldc, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * tau_native = NULL;
    cuDoubleComplex * C_native = NULL;
    int ldc_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuDoubleComplex *)getPointer(env, tau);
    C_native = (cuDoubleComplex *)getPointer(env, C);
    ldc_native = (int)ldc;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZunmqr_bufferSize(handle_native, side_native, trans_native, m_native, n_native, k_native, A_native, lda_native, tau_native, C_native, ldc_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSormqrNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint trans, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject C, jint ldc, jobject work, jint lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSormqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSormqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnSormqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnSormqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSormqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnSormqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSormqr(handle=%p, side=%d, trans=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, work=%p, lwork=%d, devInfo=%p)\n",
        handle, side, trans, m, n, k, A, lda, tau, C, ldc, work, lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * tau_native = NULL;
    float * C_native = NULL;
    int ldc_native = 0;
    float * work_native = NULL;
    int lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (float *)getPointer(env, tau);
    C_native = (float *)getPointer(env, C);
    ldc_native = (int)ldc;
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSormqr(handle_native, side_native, trans_native, m_native, n_native, k_native, A_native, lda_native, tau_native, C_native, ldc_native, work_native, lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    // work is a native pointer
    // lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDormqrNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint trans, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject C, jint ldc, jobject work, jint lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDormqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDormqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnDormqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnDormqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDormqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnDormqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDormqr(handle=%p, side=%d, trans=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, work=%p, lwork=%d, devInfo=%p)\n",
        handle, side, trans, m, n, k, A, lda, tau, C, ldc, work, lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * tau_native = NULL;
    double * C_native = NULL;
    int ldc_native = 0;
    double * work_native = NULL;
    int lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (double *)getPointer(env, tau);
    C_native = (double *)getPointer(env, C);
    ldc_native = (int)ldc;
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDormqr(handle_native, side_native, trans_native, m_native, n_native, k_native, A_native, lda_native, tau_native, C_native, ldc_native, work_native, lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    // work is a native pointer
    // lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCunmqrNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint trans, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject C, jint ldc, jobject work, jint lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCunmqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCunmqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnCunmqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnCunmqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnCunmqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnCunmqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCunmqr(handle=%p, side=%d, trans=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, work=%p, lwork=%d, devInfo=%p)\n",
        handle, side, trans, m, n, k, A, lda, tau, C, ldc, work, lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * tau_native = NULL;
    cuComplex * C_native = NULL;
    int ldc_native = 0;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuComplex *)getPointer(env, tau);
    C_native = (cuComplex *)getPointer(env, C);
    ldc_native = (int)ldc;
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCunmqr(handle_native, side_native, trans_native, m_native, n_native, k_native, A_native, lda_native, tau_native, C_native, ldc_native, work_native, lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    // work is a native pointer
    // lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZunmqrNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint trans, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject C, jint ldc, jobject work, jint lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZunmqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZunmqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnZunmqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnZunmqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZunmqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnZunmqr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZunmqr(handle=%p, side=%d, trans=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, work=%p, lwork=%d, devInfo=%p)\n",
        handle, side, trans, m, n, k, A, lda, tau, C, ldc, work, lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * tau_native = NULL;
    cuDoubleComplex * C_native = NULL;
    int ldc_native = 0;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuDoubleComplex *)getPointer(env, tau);
    C_native = (cuDoubleComplex *)getPointer(env, C);
    ldc_native = (int)ldc;
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZunmqr(handle_native, side_native, trans_native, m_native, n_native, k_native, A_native, lda_native, tau_native, C_native, ldc_native, work_native, lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    // work is a native pointer
    // lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** L*D*L**T,U*D*U**T factorization */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsytrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsytrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsytrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSsytrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsytrf_bufferSize(handle=%p, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsytrf_bufferSize(handle_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsytrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsytrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsytrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDsytrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsytrf_bufferSize(handle=%p, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsytrf_bufferSize(handle_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCsytrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCsytrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCsytrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCsytrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCsytrf_bufferSize(handle=%p, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCsytrf_bufferSize(handle_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZsytrf_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint n, jobject A, jint lda, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZsytrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZsytrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZsytrf_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZsytrf_bufferSize(handle=%p, n=%d, A=%p, lda=%d, lwork=%p)\n",
        handle, n, A, lda, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZsytrf_bufferSize(handle_native, n_native, A_native, lda_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // n is primitive
    // A is a native pointer
    // lda is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsytrfNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject ipiv, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (ipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ipiv' is null for cusolverDnSsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsytrf(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, ipiv=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, ipiv, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    int * ipiv_native = NULL;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    ipiv_native = (int *)getPointer(env, ipiv);
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsytrf(handle_native, uplo_native, n_native, A_native, lda_native, ipiv_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // ipiv is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsytrfNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject ipiv, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (ipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ipiv' is null for cusolverDnDsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsytrf(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, ipiv=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, ipiv, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    int * ipiv_native = NULL;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    ipiv_native = (int *)getPointer(env, ipiv);
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsytrf(handle_native, uplo_native, n_native, A_native, lda_native, ipiv_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // ipiv is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCsytrfNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject ipiv, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (ipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ipiv' is null for cusolverDnCsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnCsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnCsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCsytrf(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, ipiv=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, ipiv, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    int * ipiv_native = NULL;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    ipiv_native = (int *)getPointer(env, ipiv);
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCsytrf(handle_native, uplo_native, n_native, A_native, lda_native, ipiv_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // ipiv is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZsytrfNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject ipiv, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (ipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ipiv' is null for cusolverDnZsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZsytrf");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZsytrf(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, ipiv=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, ipiv, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    int * ipiv_native = NULL;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    ipiv_native = (int *)getPointer(env, ipiv);
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZsytrf(handle_native, uplo_native, n_native, A_native, lda_native, ipiv_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // ipiv is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Symmetric indefinite solve (SYTRS) */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnXsytrs_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jlong n, jlong nrhs, jint dataTypeA, jobject A, jlong lda, jlongArray ipiv, jint dataTypeB, jobject B, jlong ldb, jlongArray workspaceInBytesOnDevice, jlongArray workspaceInBytesOnHost)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnXsytrs_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    // dataTypeA is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnXsytrs_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (ipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ipiv' is null for cusolverDnXsytrs_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // dataTypeB is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnXsytrs_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (workspaceInBytesOnDevice == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'workspaceInBytesOnDevice' is null for cusolverDnXsytrs_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (workspaceInBytesOnHost == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'workspaceInBytesOnHost' is null for cusolverDnXsytrs_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnXsytrs_bufferSize(handle=%p, uplo=%d, n=%ld, nrhs=%ld, dataTypeA=%d, A=%p, lda=%ld, ipiv=%p, dataTypeB=%d, B=%p, ldb=%ld, workspaceInBytesOnDevice=%p, workspaceInBytesOnHost=%p)\n",
        handle, uplo, n, nrhs, dataTypeA, A, lda, ipiv, dataTypeB, B, ldb, workspaceInBytesOnDevice, workspaceInBytesOnHost);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int64_t n_native = 0;
    int64_t nrhs_native = 0;
    cudaDataType dataTypeA_native;
    void * A_native = NULL;
    int64_t lda_native = 0;
    int64_t * ipiv_native = NULL;
    cudaDataType dataTypeB_native;
    void * B_native = NULL;
    int64_t ldb_native = 0;
    size_t * workspaceInBytesOnDevice_native = NULL;
    size_t * workspaceInBytesOnHost_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int64_t)n;
    nrhs_native = (int64_t)nrhs;
    dataTypeA_native = (cudaDataType)dataTypeA;
    A_native = (void *)getPointer(env, A);
    lda_native = (int64_t)lda;
    ipiv_native = (int64_t *)getPointer(env, ipiv);
    dataTypeB_native = (cudaDataType)dataTypeB;
    B_native = (void *)getPointer(env, B);
    ldb_native = (int64_t)ldb;
    workspaceInBytesOnDevice_native = (size_t *)getPointer(env, workspaceInBytesOnDevice);
    workspaceInBytesOnHost_native = (size_t *)getPointer(env, workspaceInBytesOnHost);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnXsytrs_bufferSize(handle_native, uplo_native, n_native, nrhs_native, dataTypeA_native, A_native, lda_native, ipiv_native, dataTypeB_native, B_native, ldb_native, workspaceInBytesOnDevice_native, workspaceInBytesOnHost_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    // dataTypeA is primitive
    // A is a native pointer
    // lda is primitive
    // ipiv is a native pointer
    // dataTypeB is primitive
    // B is a native pointer
    // ldb is primitive
    // workspaceInBytesOnDevice is a native pointer
    // workspaceInBytesOnHost is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnXsytrsNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jlong n, jlong nrhs, jint dataTypeA, jobject A, jlong lda, jlongArray ipiv, jint dataTypeB, jobject B, jlong ldb, jobject bufferOnDevice, jlong workspaceInBytesOnDevice, jobject bufferOnHost, jlong workspaceInBytesOnHost, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnXsytrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    // dataTypeA is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnXsytrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (ipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ipiv' is null for cusolverDnXsytrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // dataTypeB is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnXsytrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (bufferOnDevice == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferOnDevice' is null for cusolverDnXsytrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // workspaceInBytesOnDevice is primitive
    if (bufferOnHost == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bufferOnHost' is null for cusolverDnXsytrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // workspaceInBytesOnHost is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnXsytrs");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnXsytrs(handle=%p, uplo=%d, n=%ld, nrhs=%ld, dataTypeA=%d, A=%p, lda=%ld, ipiv=%p, dataTypeB=%d, B=%p, ldb=%ld, bufferOnDevice=%p, workspaceInBytesOnDevice=%ld, bufferOnHost=%p, workspaceInBytesOnHost=%ld, info=%p)\n",
        handle, uplo, n, nrhs, dataTypeA, A, lda, ipiv, dataTypeB, B, ldb, bufferOnDevice, workspaceInBytesOnDevice, bufferOnHost, workspaceInBytesOnHost, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int64_t n_native = 0;
    int64_t nrhs_native = 0;
    cudaDataType dataTypeA_native;
    void * A_native = NULL;
    int64_t lda_native = 0;
    int64_t * ipiv_native = NULL;
    cudaDataType dataTypeB_native;
    void * B_native = NULL;
    int64_t ldb_native = 0;
    void * bufferOnDevice_native = NULL;
    size_t workspaceInBytesOnDevice_native = 0;
    void * bufferOnHost_native = NULL;
    size_t workspaceInBytesOnHost_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int64_t)n;
    nrhs_native = (int64_t)nrhs;
    dataTypeA_native = (cudaDataType)dataTypeA;
    A_native = (void *)getPointer(env, A);
    lda_native = (int64_t)lda;
    ipiv_native = (int64_t *)getPointer(env, ipiv);
    dataTypeB_native = (cudaDataType)dataTypeB;
    B_native = (void *)getPointer(env, B);
    ldb_native = (int64_t)ldb;
    bufferOnDevice_native = (void *)getPointer(env, bufferOnDevice);
    workspaceInBytesOnDevice_native = (size_t)workspaceInBytesOnDevice;
    bufferOnHost_native = (void *)getPointer(env, bufferOnHost);
    workspaceInBytesOnHost_native = (size_t)workspaceInBytesOnHost;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnXsytrs(handle_native, uplo_native, n_native, nrhs_native, dataTypeA_native, A_native, lda_native, ipiv_native, dataTypeB_native, B_native, ldb_native, bufferOnDevice_native, workspaceInBytesOnDevice_native, bufferOnHost_native, workspaceInBytesOnHost_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // nrhs is primitive
    // dataTypeA is primitive
    // A is a native pointer
    // lda is primitive
    // ipiv is a native pointer
    // dataTypeB is primitive
    // B is a native pointer
    // ldb is primitive
    // bufferOnDevice is a native pointer
    // workspaceInBytesOnDevice is primitive
    // bufferOnHost is a native pointer
    // workspaceInBytesOnHost is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Symmetric indefinite inversion (sytri) */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsytri_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject ipiv, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (ipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ipiv' is null for cusolverDnSsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsytri_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, ipiv=%p, lwork=%p)\n",
        handle, uplo, n, A, lda, ipiv, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    int * ipiv_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    ipiv_native = (int *)getPointer(env, ipiv);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsytri_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, ipiv_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // ipiv is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsytri_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject ipiv, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (ipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ipiv' is null for cusolverDnDsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsytri_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, ipiv=%p, lwork=%p)\n",
        handle, uplo, n, A, lda, ipiv, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    int * ipiv_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    ipiv_native = (int *)getPointer(env, ipiv);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsytri_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, ipiv_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // ipiv is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCsytri_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject ipiv, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (ipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ipiv' is null for cusolverDnCsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCsytri_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, ipiv=%p, lwork=%p)\n",
        handle, uplo, n, A, lda, ipiv, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    int * ipiv_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    ipiv_native = (int *)getPointer(env, ipiv);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCsytri_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, ipiv_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // ipiv is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZsytri_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject ipiv, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (ipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ipiv' is null for cusolverDnZsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZsytri_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZsytri_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, ipiv=%p, lwork=%p)\n",
        handle, uplo, n, A, lda, ipiv, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    int * ipiv_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    ipiv_native = (int *)getPointer(env, ipiv);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZsytri_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, ipiv_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // ipiv is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsytriNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject ipiv, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (ipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ipiv' is null for cusolverDnSsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsytri(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, ipiv=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, ipiv, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    int * ipiv_native = NULL;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    ipiv_native = (int *)getPointer(env, ipiv);
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsytri(handle_native, uplo_native, n_native, A_native, lda_native, ipiv_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // ipiv is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsytriNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject ipiv, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (ipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ipiv' is null for cusolverDnDsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsytri(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, ipiv=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, ipiv, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    int * ipiv_native = NULL;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    ipiv_native = (int *)getPointer(env, ipiv);
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsytri(handle_native, uplo_native, n_native, A_native, lda_native, ipiv_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // ipiv is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCsytriNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject ipiv, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (ipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ipiv' is null for cusolverDnCsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnCsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnCsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCsytri(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, ipiv=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, ipiv, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    int * ipiv_native = NULL;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    ipiv_native = (int *)getPointer(env, ipiv);
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCsytri(handle_native, uplo_native, n_native, A_native, lda_native, ipiv_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // ipiv is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZsytriNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject ipiv, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (ipiv == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'ipiv' is null for cusolverDnZsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZsytri");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZsytri(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, ipiv=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, ipiv, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    int * ipiv_native = NULL;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    ipiv_native = (int *)getPointer(env, ipiv);
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZsytri(handle_native, uplo_native, n_native, A_native, lda_native, ipiv_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // ipiv is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** bidiagonal factorization */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSgebrd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jintArray Lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSgebrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (Lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Lwork' is null for cusolverDnSgebrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSgebrd_bufferSize(handle=%p, m=%d, n=%d, Lwork=%p)\n",
        handle, m, n, Lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int Lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    // Lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSgebrd_bufferSize(handle_native, m_native, n_native, &Lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!set(env, Lwork, 0, (jint)Lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDgebrd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jintArray Lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDgebrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (Lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Lwork' is null for cusolverDnDgebrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDgebrd_bufferSize(handle=%p, m=%d, n=%d, Lwork=%p)\n",
        handle, m, n, Lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int Lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    // Lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDgebrd_bufferSize(handle_native, m_native, n_native, &Lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!set(env, Lwork, 0, (jint)Lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCgebrd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jintArray Lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCgebrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (Lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Lwork' is null for cusolverDnCgebrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCgebrd_bufferSize(handle=%p, m=%d, n=%d, Lwork=%p)\n",
        handle, m, n, Lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int Lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    // Lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCgebrd_bufferSize(handle_native, m_native, n_native, &Lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!set(env, Lwork, 0, (jint)Lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZgebrd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jintArray Lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZgebrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (Lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Lwork' is null for cusolverDnZgebrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZgebrd_bufferSize(handle=%p, m=%d, n=%d, Lwork=%p)\n",
        handle, m, n, Lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int Lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    // Lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZgebrd_bufferSize(handle_native, m_native, n_native, &Lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!set(env, Lwork, 0, (jint)Lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSgebrdNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jobject D, jobject E, jobject TAUQ, jobject TAUP, jobject Work, jint Lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (D == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'D' is null for cusolverDnSgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (E == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'E' is null for cusolverDnSgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (TAUQ == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'TAUQ' is null for cusolverDnSgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (TAUP == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'TAUP' is null for cusolverDnSgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (Work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Work' is null for cusolverDnSgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // Lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnSgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSgebrd(handle=%p, m=%d, n=%d, A=%p, lda=%d, D=%p, E=%p, TAUQ=%p, TAUP=%p, Work=%p, Lwork=%d, devInfo=%p)\n",
        handle, m, n, A, lda, D, E, TAUQ, TAUP, Work, Lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * D_native = NULL;
    float * E_native = NULL;
    float * TAUQ_native = NULL;
    float * TAUP_native = NULL;
    float * Work_native = NULL;
    int Lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    D_native = (float *)getPointer(env, D);
    E_native = (float *)getPointer(env, E);
    TAUQ_native = (float *)getPointer(env, TAUQ);
    TAUP_native = (float *)getPointer(env, TAUP);
    Work_native = (float *)getPointer(env, Work);
    Lwork_native = (int)Lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSgebrd(handle_native, m_native, n_native, A_native, lda_native, D_native, E_native, TAUQ_native, TAUP_native, Work_native, Lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // D is a native pointer
    // E is a native pointer
    // TAUQ is a native pointer
    // TAUP is a native pointer
    // Work is a native pointer
    // Lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDgebrdNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jobject D, jobject E, jobject TAUQ, jobject TAUP, jobject Work, jint Lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (D == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'D' is null for cusolverDnDgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (E == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'E' is null for cusolverDnDgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (TAUQ == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'TAUQ' is null for cusolverDnDgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (TAUP == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'TAUP' is null for cusolverDnDgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (Work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Work' is null for cusolverDnDgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // Lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnDgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDgebrd(handle=%p, m=%d, n=%d, A=%p, lda=%d, D=%p, E=%p, TAUQ=%p, TAUP=%p, Work=%p, Lwork=%d, devInfo=%p)\n",
        handle, m, n, A, lda, D, E, TAUQ, TAUP, Work, Lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * D_native = NULL;
    double * E_native = NULL;
    double * TAUQ_native = NULL;
    double * TAUP_native = NULL;
    double * Work_native = NULL;
    int Lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    D_native = (double *)getPointer(env, D);
    E_native = (double *)getPointer(env, E);
    TAUQ_native = (double *)getPointer(env, TAUQ);
    TAUP_native = (double *)getPointer(env, TAUP);
    Work_native = (double *)getPointer(env, Work);
    Lwork_native = (int)Lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDgebrd(handle_native, m_native, n_native, A_native, lda_native, D_native, E_native, TAUQ_native, TAUP_native, Work_native, Lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // D is a native pointer
    // E is a native pointer
    // TAUQ is a native pointer
    // TAUP is a native pointer
    // Work is a native pointer
    // Lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCgebrdNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jobject D, jobject E, jobject TAUQ, jobject TAUP, jobject Work, jint Lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (D == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'D' is null for cusolverDnCgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (E == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'E' is null for cusolverDnCgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (TAUQ == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'TAUQ' is null for cusolverDnCgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (TAUP == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'TAUP' is null for cusolverDnCgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (Work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Work' is null for cusolverDnCgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // Lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnCgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCgebrd(handle=%p, m=%d, n=%d, A=%p, lda=%d, D=%p, E=%p, TAUQ=%p, TAUP=%p, Work=%p, Lwork=%d, devInfo=%p)\n",
        handle, m, n, A, lda, D, E, TAUQ, TAUP, Work, Lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    float * D_native = NULL;
    float * E_native = NULL;
    cuComplex * TAUQ_native = NULL;
    cuComplex * TAUP_native = NULL;
    cuComplex * Work_native = NULL;
    int Lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    D_native = (float *)getPointer(env, D);
    E_native = (float *)getPointer(env, E);
    TAUQ_native = (cuComplex *)getPointer(env, TAUQ);
    TAUP_native = (cuComplex *)getPointer(env, TAUP);
    Work_native = (cuComplex *)getPointer(env, Work);
    Lwork_native = (int)Lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCgebrd(handle_native, m_native, n_native, A_native, lda_native, D_native, E_native, TAUQ_native, TAUP_native, Work_native, Lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // D is a native pointer
    // E is a native pointer
    // TAUQ is a native pointer
    // TAUP is a native pointer
    // Work is a native pointer
    // Lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZgebrdNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject A, jint lda, jobject D, jobject E, jobject TAUQ, jobject TAUP, jobject Work, jint Lwork, jobject devInfo)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (D == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'D' is null for cusolverDnZgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (E == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'E' is null for cusolverDnZgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (TAUQ == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'TAUQ' is null for cusolverDnZgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (TAUP == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'TAUP' is null for cusolverDnZgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (Work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'Work' is null for cusolverDnZgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // Lwork is primitive
    if (devInfo == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'devInfo' is null for cusolverDnZgebrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZgebrd(handle=%p, m=%d, n=%d, A=%p, lda=%d, D=%p, E=%p, TAUQ=%p, TAUP=%p, Work=%p, Lwork=%d, devInfo=%p)\n",
        handle, m, n, A, lda, D, E, TAUQ, TAUP, Work, Lwork, devInfo);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    double * D_native = NULL;
    double * E_native = NULL;
    cuDoubleComplex * TAUQ_native = NULL;
    cuDoubleComplex * TAUP_native = NULL;
    cuDoubleComplex * Work_native = NULL;
    int Lwork_native = 0;
    int * devInfo_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    D_native = (double *)getPointer(env, D);
    E_native = (double *)getPointer(env, E);
    TAUQ_native = (cuDoubleComplex *)getPointer(env, TAUQ);
    TAUP_native = (cuDoubleComplex *)getPointer(env, TAUP);
    Work_native = (cuDoubleComplex *)getPointer(env, Work);
    Lwork_native = (int)Lwork;
    devInfo_native = (int *)getPointer(env, devInfo);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZgebrd(handle_native, m_native, n_native, A_native, lda_native, D_native, E_native, TAUQ_native, TAUP_native, Work_native, Lwork_native, devInfo_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // D is a native pointer
    // E is a native pointer
    // TAUQ is a native pointer
    // TAUP is a native pointer
    // Work is a native pointer
    // Lwork is primitive
    // devInfo is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** generates one of the unitary matrices Q or P**T determined by GEBRD*/
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSorgbr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSorgbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSorgbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnSorgbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSorgbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSorgbr_bufferSize(handle=%p, side=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, lwork=%p)\n",
        handle, side, m, n, k, A, lda, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (float *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSorgbr_bufferSize(handle_native, side_native, m_native, n_native, k_native, A_native, lda_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDorgbr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDorgbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDorgbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnDorgbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDorgbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDorgbr_bufferSize(handle=%p, side=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, lwork=%p)\n",
        handle, side, m, n, k, A, lda, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (double *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDorgbr_bufferSize(handle_native, side_native, m_native, n_native, k_native, A_native, lda_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCungbr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCungbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCungbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnCungbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCungbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCungbr_bufferSize(handle=%p, side=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, lwork=%p)\n",
        handle, side, m, n, k, A, lda, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuComplex *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCungbr_bufferSize(handle_native, side_native, m_native, n_native, k_native, A_native, lda_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZungbr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZungbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZungbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnZungbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZungbr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZungbr_bufferSize(handle=%p, side=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, lwork=%p)\n",
        handle, side, m, n, k, A, lda, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuDoubleComplex *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZungbr_bufferSize(handle_native, side_native, m_native, n_native, k_native, A_native, lda_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSorgbrNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSorgbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSorgbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnSorgbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSorgbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSorgbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSorgbr(handle=%p, side=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, side, m, n, k, A, lda, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * tau_native = NULL;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (float *)getPointer(env, tau);
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSorgbr(handle_native, side_native, m_native, n_native, k_native, A_native, lda_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDorgbrNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDorgbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDorgbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnDorgbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDorgbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDorgbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDorgbr(handle=%p, side=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, side, m, n, k, A, lda, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * tau_native = NULL;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (double *)getPointer(env, tau);
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDorgbr(handle_native, side_native, m_native, n_native, k_native, A_native, lda_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCungbrNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCungbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCungbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnCungbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnCungbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnCungbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCungbr(handle=%p, side=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, side, m, n, k, A, lda, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * tau_native = NULL;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuComplex *)getPointer(env, tau);
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCungbr(handle_native, side_native, m_native, n_native, k_native, A_native, lda_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZungbrNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint m, jint n, jint k, jobject A, jint lda, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZungbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZungbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnZungbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZungbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZungbr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZungbr(handle=%p, side=%d, m=%d, n=%d, k=%d, A=%p, lda=%d, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, side, m, n, k, A, lda, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * tau_native = NULL;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuDoubleComplex *)getPointer(env, tau);
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZungbr(handle_native, side_native, m_native, n_native, k_native, A_native, lda_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** tridiagonal factorization */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsytrd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject d, jobject e, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsytrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsytrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusolverDnSsytrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (e == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'e' is null for cusolverDnSsytrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnSsytrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSsytrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsytrd_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, d=%p, e=%p, tau=%p, lwork=%p)\n",
        handle, uplo, n, A, lda, d, e, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * d_native = NULL;
    float * e_native = NULL;
    float * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    d_native = (float *)getPointer(env, d);
    e_native = (float *)getPointer(env, e);
    tau_native = (float *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsytrd_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, d_native, e_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // d is a native pointer
    // e is a native pointer
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsytrd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject d, jobject e, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsytrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsytrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusolverDnDsytrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (e == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'e' is null for cusolverDnDsytrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnDsytrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDsytrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsytrd_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, d=%p, e=%p, tau=%p, lwork=%p)\n",
        handle, uplo, n, A, lda, d, e, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * d_native = NULL;
    double * e_native = NULL;
    double * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    d_native = (double *)getPointer(env, d);
    e_native = (double *)getPointer(env, e);
    tau_native = (double *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsytrd_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, d_native, e_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // d is a native pointer
    // e is a native pointer
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnChetrd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject d, jobject e, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnChetrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnChetrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusolverDnChetrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (e == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'e' is null for cusolverDnChetrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnChetrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnChetrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnChetrd_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, d=%p, e=%p, tau=%p, lwork=%p)\n",
        handle, uplo, n, A, lda, d, e, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    float * d_native = NULL;
    float * e_native = NULL;
    cuComplex * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    d_native = (float *)getPointer(env, d);
    e_native = (float *)getPointer(env, e);
    tau_native = (cuComplex *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnChetrd_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, d_native, e_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // d is a native pointer
    // e is a native pointer
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZhetrd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject d, jobject e, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZhetrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZhetrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusolverDnZhetrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (e == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'e' is null for cusolverDnZhetrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnZhetrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZhetrd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZhetrd_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, d=%p, e=%p, tau=%p, lwork=%p)\n",
        handle, uplo, n, A, lda, d, e, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    double * d_native = NULL;
    double * e_native = NULL;
    cuDoubleComplex * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    d_native = (double *)getPointer(env, d);
    e_native = (double *)getPointer(env, e);
    tau_native = (cuDoubleComplex *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZhetrd_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, d_native, e_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // d is a native pointer
    // e is a native pointer
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsytrdNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject d, jobject e, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsytrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsytrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusolverDnSsytrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (e == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'e' is null for cusolverDnSsytrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnSsytrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSsytrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSsytrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsytrd(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, d=%p, e=%p, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, d, e, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * d_native = NULL;
    float * e_native = NULL;
    float * tau_native = NULL;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    d_native = (float *)getPointer(env, d);
    e_native = (float *)getPointer(env, e);
    tau_native = (float *)getPointer(env, tau);
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsytrd(handle_native, uplo_native, n_native, A_native, lda_native, d_native, e_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // d is a native pointer
    // e is a native pointer
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsytrdNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject d, jobject e, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsytrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsytrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusolverDnDsytrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (e == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'e' is null for cusolverDnDsytrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnDsytrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDsytrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDsytrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsytrd(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, d=%p, e=%p, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, d, e, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * d_native = NULL;
    double * e_native = NULL;
    double * tau_native = NULL;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    d_native = (double *)getPointer(env, d);
    e_native = (double *)getPointer(env, e);
    tau_native = (double *)getPointer(env, tau);
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsytrd(handle_native, uplo_native, n_native, A_native, lda_native, d_native, e_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // d is a native pointer
    // e is a native pointer
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnChetrdNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject d, jobject e, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnChetrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnChetrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusolverDnChetrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (e == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'e' is null for cusolverDnChetrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnChetrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnChetrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnChetrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnChetrd(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, d=%p, e=%p, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, d, e, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    float * d_native = NULL;
    float * e_native = NULL;
    cuComplex * tau_native = NULL;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    d_native = (float *)getPointer(env, d);
    e_native = (float *)getPointer(env, e);
    tau_native = (cuComplex *)getPointer(env, tau);
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnChetrd(handle_native, uplo_native, n_native, A_native, lda_native, d_native, e_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // d is a native pointer
    // e is a native pointer
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZhetrdNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject d, jobject e, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZhetrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZhetrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusolverDnZhetrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (e == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'e' is null for cusolverDnZhetrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnZhetrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZhetrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZhetrd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZhetrd(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, d=%p, e=%p, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, d, e, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    double * d_native = NULL;
    double * e_native = NULL;
    cuDoubleComplex * tau_native = NULL;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    d_native = (double *)getPointer(env, d);
    e_native = (double *)getPointer(env, e);
    tau_native = (cuDoubleComplex *)getPointer(env, tau);
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZhetrd(handle_native, uplo_native, n_native, A_native, lda_native, d_native, e_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // d is a native pointer
    // e is a native pointer
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** generate unitary Q comes from sytrd */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSorgtr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSorgtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSorgtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnSorgtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSorgtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSorgtr_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, tau=%p, lwork=%p)\n",
        handle, uplo, n, A, lda, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (float *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSorgtr_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDorgtr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDorgtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDorgtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnDorgtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDorgtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDorgtr_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, tau=%p, lwork=%p)\n",
        handle, uplo, n, A, lda, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (double *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDorgtr_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCungtr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCungtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCungtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnCungtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCungtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCungtr_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, tau=%p, lwork=%p)\n",
        handle, uplo, n, A, lda, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuComplex *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCungtr_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZungtr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject tau, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZungtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZungtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnZungtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZungtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZungtr_bufferSize(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, tau=%p, lwork=%p)\n",
        handle, uplo, n, A, lda, tau, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * tau_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuDoubleComplex *)getPointer(env, tau);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZungtr_bufferSize(handle_native, uplo_native, n_native, A_native, lda_native, tau_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSorgtrNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSorgtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSorgtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnSorgtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSorgtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSorgtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSorgtr(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * tau_native = NULL;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (float *)getPointer(env, tau);
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSorgtr(handle_native, uplo_native, n_native, A_native, lda_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDorgtrNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDorgtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDorgtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnDorgtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDorgtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDorgtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDorgtr(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * tau_native = NULL;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (double *)getPointer(env, tau);
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDorgtr(handle_native, uplo_native, n_native, A_native, lda_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCungtrNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCungtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCungtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnCungtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnCungtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnCungtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCungtr(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * tau_native = NULL;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuComplex *)getPointer(env, tau);
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCungtr(handle_native, uplo_native, n_native, A_native, lda_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZungtrNative(JNIEnv *env, jclass cls, jobject handle, jint uplo, jint n, jobject A, jint lda, jobject tau, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZungtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZungtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnZungtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZungtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZungtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZungtr(handle=%p, uplo=%d, n=%d, A=%p, lda=%d, tau=%p, work=%p, lwork=%d, info=%p)\n",
        handle, uplo, n, A, lda, tau, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * tau_native = NULL;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuDoubleComplex *)getPointer(env, tau);
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZungtr(handle_native, uplo_native, n_native, A_native, lda_native, tau_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** compute op(Q)*C or C*op(Q) where Q comes from sytrd */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSormtr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint uplo, jint trans, jint m, jint n, jobject A, jint lda, jobject tau, jobject C, jint ldc, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSormtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSormtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnSormtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnSormtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSormtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSormtr_bufferSize(handle=%p, side=%d, uplo=%d, trans=%d, m=%d, n=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, lwork=%p)\n",
        handle, side, uplo, trans, m, n, A, lda, tau, C, ldc, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasFillMode_t uplo_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * tau_native = NULL;
    float * C_native = NULL;
    int ldc_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    uplo_native = (cublasFillMode_t)uplo;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (float *)getPointer(env, tau);
    C_native = (float *)getPointer(env, C);
    ldc_native = (int)ldc;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSormtr_bufferSize(handle_native, side_native, uplo_native, trans_native, m_native, n_native, A_native, lda_native, tau_native, C_native, ldc_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDormtr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint uplo, jint trans, jint m, jint n, jobject A, jint lda, jobject tau, jobject C, jint ldc, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDormtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDormtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnDormtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnDormtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDormtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDormtr_bufferSize(handle=%p, side=%d, uplo=%d, trans=%d, m=%d, n=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, lwork=%p)\n",
        handle, side, uplo, trans, m, n, A, lda, tau, C, ldc, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasFillMode_t uplo_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * tau_native = NULL;
    double * C_native = NULL;
    int ldc_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    uplo_native = (cublasFillMode_t)uplo;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (double *)getPointer(env, tau);
    C_native = (double *)getPointer(env, C);
    ldc_native = (int)ldc;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDormtr_bufferSize(handle_native, side_native, uplo_native, trans_native, m_native, n_native, A_native, lda_native, tau_native, C_native, ldc_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCunmtr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint uplo, jint trans, jint m, jint n, jobject A, jint lda, jobject tau, jobject C, jint ldc, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCunmtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCunmtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnCunmtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnCunmtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCunmtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCunmtr_bufferSize(handle=%p, side=%d, uplo=%d, trans=%d, m=%d, n=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, lwork=%p)\n",
        handle, side, uplo, trans, m, n, A, lda, tau, C, ldc, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasFillMode_t uplo_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * tau_native = NULL;
    cuComplex * C_native = NULL;
    int ldc_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    uplo_native = (cublasFillMode_t)uplo;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuComplex *)getPointer(env, tau);
    C_native = (cuComplex *)getPointer(env, C);
    ldc_native = (int)ldc;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCunmtr_bufferSize(handle_native, side_native, uplo_native, trans_native, m_native, n_native, A_native, lda_native, tau_native, C_native, ldc_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZunmtr_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint uplo, jint trans, jint m, jint n, jobject A, jint lda, jobject tau, jobject C, jint ldc, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZunmtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZunmtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnZunmtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnZunmtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZunmtr_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZunmtr_bufferSize(handle=%p, side=%d, uplo=%d, trans=%d, m=%d, n=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, lwork=%p)\n",
        handle, side, uplo, trans, m, n, A, lda, tau, C, ldc, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasFillMode_t uplo_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * tau_native = NULL;
    cuDoubleComplex * C_native = NULL;
    int ldc_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    uplo_native = (cublasFillMode_t)uplo;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuDoubleComplex *)getPointer(env, tau);
    C_native = (cuDoubleComplex *)getPointer(env, C);
    ldc_native = (int)ldc;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZunmtr_bufferSize(handle_native, side_native, uplo_native, trans_native, m_native, n_native, A_native, lda_native, tau_native, C_native, ldc_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSormtrNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint uplo, jint trans, jint m, jint n, jobject A, jint lda, jobject tau, jobject C, jint ldc, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSormtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSormtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnSormtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnSormtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSormtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSormtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSormtr(handle=%p, side=%d, uplo=%d, trans=%d, m=%d, n=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, work=%p, lwork=%d, info=%p)\n",
        handle, side, uplo, trans, m, n, A, lda, tau, C, ldc, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasFillMode_t uplo_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * tau_native = NULL;
    float * C_native = NULL;
    int ldc_native = 0;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    uplo_native = (cublasFillMode_t)uplo;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (float *)getPointer(env, tau);
    C_native = (float *)getPointer(env, C);
    ldc_native = (int)ldc;
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSormtr(handle_native, side_native, uplo_native, trans_native, m_native, n_native, A_native, lda_native, tau_native, C_native, ldc_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDormtrNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint uplo, jint trans, jint m, jint n, jobject A, jint lda, jobject tau, jobject C, jint ldc, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDormtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDormtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnDormtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnDormtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDormtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDormtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDormtr(handle=%p, side=%d, uplo=%d, trans=%d, m=%d, n=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, work=%p, lwork=%d, info=%p)\n",
        handle, side, uplo, trans, m, n, A, lda, tau, C, ldc, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasFillMode_t uplo_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * tau_native = NULL;
    double * C_native = NULL;
    int ldc_native = 0;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    uplo_native = (cublasFillMode_t)uplo;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (double *)getPointer(env, tau);
    C_native = (double *)getPointer(env, C);
    ldc_native = (int)ldc;
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDormtr(handle_native, side_native, uplo_native, trans_native, m_native, n_native, A_native, lda_native, tau_native, C_native, ldc_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCunmtrNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint uplo, jint trans, jint m, jint n, jobject A, jint lda, jobject tau, jobject C, jint ldc, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCunmtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCunmtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnCunmtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnCunmtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnCunmtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnCunmtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCunmtr(handle=%p, side=%d, uplo=%d, trans=%d, m=%d, n=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, work=%p, lwork=%d, info=%p)\n",
        handle, side, uplo, trans, m, n, A, lda, tau, C, ldc, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasFillMode_t uplo_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * tau_native = NULL;
    cuComplex * C_native = NULL;
    int ldc_native = 0;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    uplo_native = (cublasFillMode_t)uplo;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuComplex *)getPointer(env, tau);
    C_native = (cuComplex *)getPointer(env, C);
    ldc_native = (int)ldc;
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCunmtr(handle_native, side_native, uplo_native, trans_native, m_native, n_native, A_native, lda_native, tau_native, C_native, ldc_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZunmtrNative(JNIEnv *env, jclass cls, jobject handle, jint side, jint uplo, jint trans, jint m, jint n, jobject A, jint lda, jobject tau, jobject C, jint ldc, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZunmtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZunmtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (tau == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tau' is null for cusolverDnZunmtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusolverDnZunmtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZunmtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZunmtr");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZunmtr(handle=%p, side=%d, uplo=%d, trans=%d, m=%d, n=%d, A=%p, lda=%d, tau=%p, C=%p, ldc=%d, work=%p, lwork=%d, info=%p)\n",
        handle, side, uplo, trans, m, n, A, lda, tau, C, ldc, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cublasSideMode_t side_native;
    cublasFillMode_t uplo_native;
    cublasOperation_t trans_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * tau_native = NULL;
    cuDoubleComplex * C_native = NULL;
    int ldc_native = 0;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    side_native = (cublasSideMode_t)side;
    uplo_native = (cublasFillMode_t)uplo;
    trans_native = (cublasOperation_t)trans;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    tau_native = (cuDoubleComplex *)getPointer(env, tau);
    C_native = (cuDoubleComplex *)getPointer(env, C);
    ldc_native = (int)ldc;
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZunmtr(handle_native, side_native, uplo_native, trans_native, m_native, n_native, A_native, lda_native, tau_native, C_native, ldc_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // side is primitive
    // uplo is primitive
    // trans is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // tau is a native pointer
    // C is a native pointer
    // ldc is primitive
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** singular value decomposition, A = U * Sigma * V^H */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSgesvd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSgesvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSgesvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSgesvd_bufferSize(handle=%p, m=%d, n=%d, lwork=%p)\n",
        handle, m, n, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSgesvd_bufferSize(handle_native, m_native, n_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDgesvd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDgesvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDgesvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDgesvd_bufferSize(handle=%p, m=%d, n=%d, lwork=%p)\n",
        handle, m, n, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDgesvd_bufferSize(handle_native, m_native, n_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCgesvd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCgesvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCgesvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCgesvd_bufferSize(handle=%p, m=%d, n=%d, lwork=%p)\n",
        handle, m, n, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCgesvd_bufferSize(handle_native, m_native, n_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZgesvd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZgesvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZgesvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZgesvd_bufferSize(handle=%p, m=%d, n=%d, lwork=%p)\n",
        handle, m, n, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZgesvd_bufferSize(handle_native, m_native, n_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // m is primitive
    // n is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSgesvdNative(JNIEnv *env, jclass cls, jobject handle, jchar jobu, jchar jobvt, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject VT, jint ldvt, jobject work, jint lwork, jobject rwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobu is primitive
    // jobvt is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnSgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnSgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (VT == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'VT' is null for cusolverDnSgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldvt is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (rwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'rwork' is null for cusolverDnSgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSgesvd(handle=%p, jobu=%c, jobvt=%c, m=%d, n=%d, A=%p, lda=%d, S=%p, U=%p, ldu=%d, VT=%p, ldvt=%d, work=%p, lwork=%d, rwork=%p, info=%p)\n",
        handle, jobu, jobvt, m, n, A, lda, S, U, ldu, VT, ldvt, work, lwork, rwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    char jobu_native = 0;
    char jobvt_native = 0;
    int m_native = 0;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * S_native = NULL;
    float * U_native = NULL;
    int ldu_native = 0;
    float * VT_native = NULL;
    int ldvt_native = 0;
    float * work_native = NULL;
    int lwork_native = 0;
    float * rwork_native = NULL;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobu_native = (char)jobu;
    jobvt_native = (char)jobvt;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    S_native = (float *)getPointer(env, S);
    U_native = (float *)getPointer(env, U);
    ldu_native = (int)ldu;
    VT_native = (float *)getPointer(env, VT);
    ldvt_native = (int)ldvt;
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    PointerData *rwork_pointerData = initPointerData(env, rwork);
    if (rwork_pointerData == NULL)
    {
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    rwork_native = (float *)rwork_pointerData->getPointer(env);
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSgesvd(handle_native, jobu_native, jobvt_native, m_native, n_native, A_native, lda_native, S_native, U_native, ldu_native, VT_native, ldvt_native, work_native, lwork_native, rwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // jobu is primitive
    // jobvt is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // S is a native pointer
    // U is a native pointer
    // ldu is primitive
    // VT is a native pointer
    // ldvt is primitive
    // work is a native pointer
    // lwork is primitive
    if (!releasePointerData(env, rwork_pointerData, JNI_ABORT)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDgesvdNative(JNIEnv *env, jclass cls, jobject handle, jchar jobu, jchar jobvt, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject VT, jint ldvt, jobject work, jint lwork, jobject rwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobu is primitive
    // jobvt is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnDgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnDgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (VT == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'VT' is null for cusolverDnDgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldvt is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (rwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'rwork' is null for cusolverDnDgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDgesvd(handle=%p, jobu=%c, jobvt=%c, m=%d, n=%d, A=%p, lda=%d, S=%p, U=%p, ldu=%d, VT=%p, ldvt=%d, work=%p, lwork=%d, rwork=%p, info=%p)\n",
        handle, jobu, jobvt, m, n, A, lda, S, U, ldu, VT, ldvt, work, lwork, rwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    char jobu_native = 0;
    char jobvt_native = 0;
    int m_native = 0;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * S_native = NULL;
    double * U_native = NULL;
    int ldu_native = 0;
    double * VT_native = NULL;
    int ldvt_native = 0;
    double * work_native = NULL;
    int lwork_native = 0;
    double * rwork_native = NULL;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobu_native = (char)jobu;
    jobvt_native = (char)jobvt;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    S_native = (double *)getPointer(env, S);
    U_native = (double *)getPointer(env, U);
    ldu_native = (int)ldu;
    VT_native = (double *)getPointer(env, VT);
    ldvt_native = (int)ldvt;
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    PointerData *rwork_pointerData = initPointerData(env, rwork);
    if (rwork_pointerData == NULL)
    {
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    rwork_native = (double *)rwork_pointerData->getPointer(env);
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDgesvd(handle_native, jobu_native, jobvt_native, m_native, n_native, A_native, lda_native, S_native, U_native, ldu_native, VT_native, ldvt_native, work_native, lwork_native, rwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // jobu is primitive
    // jobvt is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // S is a native pointer
    // U is a native pointer
    // ldu is primitive
    // VT is a native pointer
    // ldvt is primitive
    // work is a native pointer
    // lwork is primitive
    if (!releasePointerData(env, rwork_pointerData, JNI_ABORT)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCgesvdNative(JNIEnv *env, jclass cls, jobject handle, jchar jobu, jchar jobvt, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject VT, jint ldvt, jobject work, jint lwork, jobject rwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobu is primitive
    // jobvt is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnCgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnCgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (VT == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'VT' is null for cusolverDnCgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldvt is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnCgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (rwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'rwork' is null for cusolverDnCgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnCgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCgesvd(handle=%p, jobu=%c, jobvt=%c, m=%d, n=%d, A=%p, lda=%d, S=%p, U=%p, ldu=%d, VT=%p, ldvt=%d, work=%p, lwork=%d, rwork=%p, info=%p)\n",
        handle, jobu, jobvt, m, n, A, lda, S, U, ldu, VT, ldvt, work, lwork, rwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    char jobu_native = 0;
    char jobvt_native = 0;
    int m_native = 0;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    float * S_native = NULL;
    cuComplex * U_native = NULL;
    int ldu_native = 0;
    cuComplex * VT_native = NULL;
    int ldvt_native = 0;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    float * rwork_native = NULL;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobu_native = (char)jobu;
    jobvt_native = (char)jobvt;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    S_native = (float *)getPointer(env, S);
    U_native = (cuComplex *)getPointer(env, U);
    ldu_native = (int)ldu;
    VT_native = (cuComplex *)getPointer(env, VT);
    ldvt_native = (int)ldvt;
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    PointerData *rwork_pointerData = initPointerData(env, rwork);
    if (rwork_pointerData == NULL)
    {
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    rwork_native = (float *)rwork_pointerData->getPointer(env);
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCgesvd(handle_native, jobu_native, jobvt_native, m_native, n_native, A_native, lda_native, S_native, U_native, ldu_native, VT_native, ldvt_native, work_native, lwork_native, rwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // jobu is primitive
    // jobvt is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // S is a native pointer
    // U is a native pointer
    // ldu is primitive
    // VT is a native pointer
    // ldvt is primitive
    // work is a native pointer
    // lwork is primitive
    if (!releasePointerData(env, rwork_pointerData, JNI_ABORT)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZgesvdNative(JNIEnv *env, jclass cls, jobject handle, jchar jobu, jchar jobvt, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject VT, jint ldvt, jobject work, jint lwork, jobject rwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobu is primitive
    // jobvt is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnZgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnZgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (VT == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'VT' is null for cusolverDnZgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldvt is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (rwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'rwork' is null for cusolverDnZgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZgesvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZgesvd(handle=%p, jobu=%c, jobvt=%c, m=%d, n=%d, A=%p, lda=%d, S=%p, U=%p, ldu=%d, VT=%p, ldvt=%d, work=%p, lwork=%d, rwork=%p, info=%p)\n",
        handle, jobu, jobvt, m, n, A, lda, S, U, ldu, VT, ldvt, work, lwork, rwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    char jobu_native = 0;
    char jobvt_native = 0;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    double * S_native = NULL;
    cuDoubleComplex * U_native = NULL;
    int ldu_native = 0;
    cuDoubleComplex * VT_native = NULL;
    int ldvt_native = 0;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    double * rwork_native = NULL;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobu_native = (char)jobu;
    jobvt_native = (char)jobvt;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    S_native = (double *)getPointer(env, S);
    U_native = (cuDoubleComplex *)getPointer(env, U);
    ldu_native = (int)ldu;
    VT_native = (cuDoubleComplex *)getPointer(env, VT);
    ldvt_native = (int)ldvt;
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    PointerData *rwork_pointerData = initPointerData(env, rwork);
    if (rwork_pointerData == NULL)
    {
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    rwork_native = (double *)rwork_pointerData->getPointer(env);
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZgesvd(handle_native, jobu_native, jobvt_native, m_native, n_native, A_native, lda_native, S_native, U_native, ldu_native, VT_native, ldvt_native, work_native, lwork_native, rwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // jobu is primitive
    // jobvt is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // S is a native pointer
    // U is a native pointer
    // ldu is primitive
    // VT is a native pointer
    // ldvt is primitive
    // work is a native pointer
    // lwork is primitive
    if (!releasePointerData(env, rwork_pointerData, JNI_ABORT)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** standard symmetric eigenvalue solver, A*x = lambda*x, by divide-and-conquer
 */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsyevd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsyevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsyevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnSsyevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSsyevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsyevd_bufferSize(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, lwork=%p)\n",
        handle, jobz, uplo, n, A, lda, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (float *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsyevd_bufferSize(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsyevd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsyevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsyevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnDsyevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDsyevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsyevd_bufferSize(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, lwork=%p)\n",
        handle, jobz, uplo, n, A, lda, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (double *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsyevd_bufferSize(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCheevd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCheevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCheevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnCheevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCheevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCheevd_bufferSize(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, lwork=%p)\n",
        handle, jobz, uplo, n, A, lda, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    float * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (float *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCheevd_bufferSize(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZheevd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZheevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZheevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnZheevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZheevd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZheevd_bufferSize(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, lwork=%p)\n",
        handle, jobz, uplo, n, A, lda, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    double * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (double *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZheevd_bufferSize(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsyevdNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsyevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsyevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnSsyevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSsyevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSsyevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsyevd(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, jobz, uplo, n, A, lda, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * W_native = NULL;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (float *)getPointer(env, W);
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsyevd(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsyevdNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsyevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsyevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnDsyevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDsyevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDsyevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsyevd(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, jobz, uplo, n, A, lda, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * W_native = NULL;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (double *)getPointer(env, W);
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsyevd(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCheevdNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCheevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCheevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnCheevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnCheevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnCheevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCheevd(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, jobz, uplo, n, A, lda, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    float * W_native = NULL;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (float *)getPointer(env, W);
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCheevd(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZheevdNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZheevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZheevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnZheevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZheevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZheevd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZheevd(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, jobz, uplo, n, A, lda, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    double * W_native = NULL;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (double *)getPointer(env, W);
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZheevd(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** standard selective symmetric eigenvalue solver, A*x = lambda*x, by
 * divide-and-conquer  */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsyevdx_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jfloat vl, jfloat vu, jint il, jint iu, jobject meig, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsyevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsyevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnSsyevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnSsyevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSsyevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsyevdx_bufferSize(handle=%p, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, vl=%f, vu=%f, il=%d, iu=%d, meig=%p, W=%p, lwork=%p)\n",
        handle, jobz, range, uplo, n, A, lda, vl, vu, il, iu, meig, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float vl_native = 0.0f;
    float vu_native = 0.0f;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    float * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    vl_native = (float)vl;
    vu_native = (float)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (float *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsyevdx_bufferSize(handle_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsyevdx_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jdouble vl, jdouble vu, jint il, jint iu, jobject meig, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsyevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsyevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnDsyevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnDsyevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDsyevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsyevdx_bufferSize(handle=%p, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, vl=%lf, vu=%lf, il=%d, iu=%d, meig=%p, W=%p, lwork=%p)\n",
        handle, jobz, range, uplo, n, A, lda, vl, vu, il, iu, meig, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double vl_native = 0.0;
    double vu_native = 0.0;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    double * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    vl_native = (double)vl;
    vu_native = (double)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (double *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsyevdx_bufferSize(handle_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCheevdx_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jfloat vl, jfloat vu, jint il, jint iu, jobject meig, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCheevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCheevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnCheevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnCheevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCheevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCheevdx_bufferSize(handle=%p, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, vl=%f, vu=%f, il=%d, iu=%d, meig=%p, W=%p, lwork=%p)\n",
        handle, jobz, range, uplo, n, A, lda, vl, vu, il, iu, meig, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    float vl_native = 0.0f;
    float vu_native = 0.0f;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    float * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    vl_native = (float)vl;
    vu_native = (float)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (float *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCheevdx_bufferSize(handle_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZheevdx_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jdouble vl, jdouble vu, jint il, jint iu, jobject meig, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZheevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZheevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnZheevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnZheevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZheevdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZheevdx_bufferSize(handle=%p, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, vl=%lf, vu=%lf, il=%d, iu=%d, meig=%p, W=%p, lwork=%p)\n",
        handle, jobz, range, uplo, n, A, lda, vl, vu, il, iu, meig, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    double vl_native = 0.0;
    double vu_native = 0.0;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    double * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    vl_native = (double)vl;
    vu_native = (double)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (double *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZheevdx_bufferSize(handle_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsyevdxNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jfloat vl, jfloat vu, jint il, jint iu, jobject meig, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsyevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsyevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnSsyevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnSsyevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSsyevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSsyevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsyevdx(handle=%p, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, vl=%f, vu=%f, il=%d, iu=%d, meig=%p, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, jobz, range, uplo, n, A, lda, vl, vu, il, iu, meig, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float vl_native = 0.0f;
    float vu_native = 0.0f;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    float * W_native = NULL;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    vl_native = (float)vl;
    vu_native = (float)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (float *)getPointer(env, W);
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsyevdx(handle_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsyevdxNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jdouble vl, jdouble vu, jint il, jint iu, jobject meig, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsyevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsyevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnDsyevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnDsyevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDsyevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDsyevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsyevdx(handle=%p, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, vl=%lf, vu=%lf, il=%d, iu=%d, meig=%p, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, jobz, range, uplo, n, A, lda, vl, vu, il, iu, meig, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double vl_native = 0.0;
    double vu_native = 0.0;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    double * W_native = NULL;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    vl_native = (double)vl;
    vu_native = (double)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (double *)getPointer(env, W);
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsyevdx(handle_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCheevdxNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jfloat vl, jfloat vu, jint il, jint iu, jobject meig, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCheevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCheevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnCheevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnCheevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnCheevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnCheevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCheevdx(handle=%p, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, vl=%f, vu=%f, il=%d, iu=%d, meig=%p, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, jobz, range, uplo, n, A, lda, vl, vu, il, iu, meig, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    float vl_native = 0.0f;
    float vu_native = 0.0f;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    float * W_native = NULL;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    vl_native = (float)vl;
    vu_native = (float)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (float *)getPointer(env, W);
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCheevdx(handle_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZheevdxNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jdouble vl, jdouble vu, jint il, jint iu, jobject meig, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZheevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZheevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnZheevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnZheevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZheevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZheevdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZheevdx(handle=%p, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, vl=%lf, vu=%lf, il=%d, iu=%d, meig=%p, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, jobz, range, uplo, n, A, lda, vl, vu, il, iu, meig, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    double vl_native = 0.0;
    double vu_native = 0.0;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    double * W_native = NULL;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    vl_native = (double)vl;
    vu_native = (double)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (double *)getPointer(env, W);
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZheevdx(handle_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** selective generalized symmetric eigenvalue solver, A*x = lambda*B*x, by
 * divide-and-conquer  */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsygvdx_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jfloat vl, jfloat vu, jint il, jint iu, jobject meig, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsygvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsygvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnSsygvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnSsygvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnSsygvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSsygvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsygvdx_bufferSize(handle=%p, itype=%d, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, vl=%f, vu=%f, il=%d, iu=%d, meig=%p, W=%p, lwork=%p)\n",
        handle, itype, jobz, range, uplo, n, A, lda, B, ldb, vl, vu, il, iu, meig, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * B_native = NULL;
    int ldb_native = 0;
    float vl_native = 0.0f;
    float vu_native = 0.0f;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    float * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (float *)getPointer(env, B);
    ldb_native = (int)ldb;
    vl_native = (float)vl;
    vu_native = (float)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (float *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsygvdx_bufferSize(handle_native, itype_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsygvdx_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jdouble vl, jdouble vu, jint il, jint iu, jobject meig, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsygvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsygvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnDsygvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnDsygvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnDsygvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDsygvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsygvdx_bufferSize(handle=%p, itype=%d, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, vl=%lf, vu=%lf, il=%d, iu=%d, meig=%p, W=%p, lwork=%p)\n",
        handle, itype, jobz, range, uplo, n, A, lda, B, ldb, vl, vu, il, iu, meig, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * B_native = NULL;
    int ldb_native = 0;
    double vl_native = 0.0;
    double vu_native = 0.0;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    double * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (double *)getPointer(env, B);
    ldb_native = (int)ldb;
    vl_native = (double)vl;
    vu_native = (double)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (double *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsygvdx_bufferSize(handle_native, itype_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnChegvdx_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jfloat vl, jfloat vu, jint il, jint iu, jobject meig, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnChegvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnChegvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnChegvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnChegvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnChegvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnChegvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnChegvdx_bufferSize(handle=%p, itype=%d, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, vl=%f, vu=%f, il=%d, iu=%d, meig=%p, W=%p, lwork=%p)\n",
        handle, itype, jobz, range, uplo, n, A, lda, B, ldb, vl, vu, il, iu, meig, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * B_native = NULL;
    int ldb_native = 0;
    float vl_native = 0.0f;
    float vu_native = 0.0f;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    float * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    vl_native = (float)vl;
    vu_native = (float)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (float *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnChegvdx_bufferSize(handle_native, itype_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZhegvdx_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jdouble vl, jdouble vu, jint il, jint iu, jobject meig, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZhegvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZhegvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnZhegvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnZhegvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnZhegvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZhegvdx_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZhegvdx_bufferSize(handle=%p, itype=%d, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, vl=%lf, vu=%lf, il=%d, iu=%d, meig=%p, W=%p, lwork=%p)\n",
        handle, itype, jobz, range, uplo, n, A, lda, B, ldb, vl, vu, il, iu, meig, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * B_native = NULL;
    int ldb_native = 0;
    double vl_native = 0.0;
    double vu_native = 0.0;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    double * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuDoubleComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    vl_native = (double)vl;
    vu_native = (double)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (double *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZhegvdx_bufferSize(handle_native, itype_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsygvdxNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jfloat vl, jfloat vu, jint il, jint iu, jobject meig, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsygvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsygvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnSsygvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnSsygvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnSsygvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSsygvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSsygvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsygvdx(handle=%p, itype=%d, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, vl=%f, vu=%f, il=%d, iu=%d, meig=%p, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, itype, jobz, range, uplo, n, A, lda, B, ldb, vl, vu, il, iu, meig, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * B_native = NULL;
    int ldb_native = 0;
    float vl_native = 0.0f;
    float vu_native = 0.0f;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    float * W_native = NULL;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (float *)getPointer(env, B);
    ldb_native = (int)ldb;
    vl_native = (float)vl;
    vu_native = (float)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (float *)getPointer(env, W);
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsygvdx(handle_native, itype_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsygvdxNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jdouble vl, jdouble vu, jint il, jint iu, jobject meig, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsygvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsygvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnDsygvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnDsygvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnDsygvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDsygvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDsygvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsygvdx(handle=%p, itype=%d, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, vl=%lf, vu=%lf, il=%d, iu=%d, meig=%p, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, itype, jobz, range, uplo, n, A, lda, B, ldb, vl, vu, il, iu, meig, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * B_native = NULL;
    int ldb_native = 0;
    double vl_native = 0.0;
    double vu_native = 0.0;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    double * W_native = NULL;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (double *)getPointer(env, B);
    ldb_native = (int)ldb;
    vl_native = (double)vl;
    vu_native = (double)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (double *)getPointer(env, W);
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsygvdx(handle_native, itype_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnChegvdxNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jfloat vl, jfloat vu, jint il, jint iu, jobject meig, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnChegvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnChegvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnChegvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnChegvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnChegvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnChegvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnChegvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnChegvdx(handle=%p, itype=%d, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, vl=%f, vu=%f, il=%d, iu=%d, meig=%p, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, itype, jobz, range, uplo, n, A, lda, B, ldb, vl, vu, il, iu, meig, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * B_native = NULL;
    int ldb_native = 0;
    float vl_native = 0.0f;
    float vu_native = 0.0f;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    float * W_native = NULL;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    vl_native = (float)vl;
    vu_native = (float)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (float *)getPointer(env, W);
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnChegvdx(handle_native, itype_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZhegvdxNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint range, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jdouble vl, jdouble vu, jint il, jint iu, jobject meig, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZhegvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZhegvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnZhegvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    if (meig == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'meig' is null for cusolverDnZhegvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnZhegvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZhegvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZhegvdx");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZhegvdx(handle=%p, itype=%d, jobz=%d, range=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, vl=%lf, vu=%lf, il=%d, iu=%d, meig=%p, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, itype, jobz, range, uplo, n, A, lda, B, ldb, vl, vu, il, iu, meig, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cusolverEigRange_t range_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * B_native = NULL;
    int ldb_native = 0;
    double vl_native = 0.0;
    double vu_native = 0.0;
    int il_native = 0;
    int iu_native = 0;
    int * meig_native = NULL;
    double * W_native = NULL;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    range_native = (cusolverEigRange_t)range;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuDoubleComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    vl_native = (double)vl;
    vu_native = (double)vu;
    il_native = (int)il;
    iu_native = (int)iu;
    meig_native = (int *)getPointer(env, meig);
    W_native = (double *)getPointer(env, W);
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZhegvdx(handle_native, itype_native, jobz_native, range_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, vl_native, vu_native, il_native, iu_native, meig_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // range is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // vl is primitive
    // vu is primitive
    // il is primitive
    // iu is primitive
    // meig is a native pointer
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

/** generalized symmetric eigenvalue solver, A*x = lambda*B*x, by
 * divide-and-conquer  */
JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsygvd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsygvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsygvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnSsygvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnSsygvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSsygvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsygvd_bufferSize(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, lwork=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * B_native = NULL;
    int ldb_native = 0;
    float * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (float *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (float *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsygvd_bufferSize(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsygvd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsygvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsygvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnDsygvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnDsygvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDsygvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsygvd_bufferSize(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, lwork=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * B_native = NULL;
    int ldb_native = 0;
    double * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (double *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (double *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsygvd_bufferSize(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnChegvd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnChegvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnChegvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnChegvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnChegvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnChegvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnChegvd_bufferSize(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, lwork=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * B_native = NULL;
    int ldb_native = 0;
    float * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (float *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnChegvd_bufferSize(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZhegvd_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jintArray lwork)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZhegvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZhegvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnZhegvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnZhegvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZhegvd_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZhegvd_bufferSize(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, lwork=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, lwork);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * B_native = NULL;
    int ldb_native = 0;
    double * W_native = NULL;
    int lwork_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuDoubleComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (double *)getPointer(env, W);
    // lwork is write-only

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZhegvd_bufferSize(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, &lwork_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsygvdNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsygvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsygvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnSsygvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnSsygvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSsygvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSsygvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsygvd(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * B_native = NULL;
    int ldb_native = 0;
    float * W_native = NULL;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (float *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (float *)getPointer(env, W);
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsygvd(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsygvdNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsygvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsygvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnDsygvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnDsygvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDsygvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDsygvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsygvd(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * B_native = NULL;
    int ldb_native = 0;
    double * W_native = NULL;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (double *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (double *)getPointer(env, W);
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsygvd(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnChegvdNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnChegvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnChegvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnChegvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnChegvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnChegvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnChegvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnChegvd(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * B_native = NULL;
    int ldb_native = 0;
    float * W_native = NULL;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (float *)getPointer(env, W);
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnChegvd(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZhegvdNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jobject work, jint lwork, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZhegvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZhegvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnZhegvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnZhegvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZhegvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZhegvd");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZhegvd(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, work=%p, lwork=%d, info=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, work, lwork, info);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * B_native = NULL;
    int ldb_native = 0;
    double * W_native = NULL;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuDoubleComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (double *)getPointer(env, W);
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZhegvd(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, work_native, lwork_native, info_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCreateSyevjInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnCreateSyevjInfo");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCreateSyevjInfo(info=%p)\n",
        info);

    // Native variable declarations
    syevjInfo_t * info_native;

    // Obtain native variable values
    info_native = (syevjInfo_t *)getNativePointerValue(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCreateSyevjInfo(info_native);

    // Write back native variable values
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDestroySyevjInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDestroySyevjInfo");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDestroySyevjInfo(info=%p)\n",
        info);

    // Native variable declarations
    syevjInfo_t info_native;

    // Obtain native variable values
    info_native = (syevjInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDestroySyevjInfo(info_native);

    // Write back native variable values
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnXsyevjSetToleranceNative(JNIEnv *env, jclass cls, jobject info, jdouble tolerance)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnXsyevjSetTolerance");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // tolerance is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnXsyevjSetTolerance(info=%p, tolerance=%lf)\n",
        info, tolerance);

    // Native variable declarations
    syevjInfo_t info_native;
    double tolerance_native = 0.0;

    // Obtain native variable values
    info_native = (syevjInfo_t)getNativePointerValue(env, info);
    tolerance_native = (double)tolerance;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnXsyevjSetTolerance(info_native, tolerance_native);

    // Write back native variable values
    // info is read-only
    // tolerance is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnXsyevjSetMaxSweepsNative(JNIEnv *env, jclass cls, jobject info, jint max_sweeps)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnXsyevjSetMaxSweeps");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // max_sweeps is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnXsyevjSetMaxSweeps(info=%p, max_sweeps=%d)\n",
        info, max_sweeps);

    // Native variable declarations
    syevjInfo_t info_native;
    int max_sweeps_native = 0;

    // Obtain native variable values
    info_native = (syevjInfo_t)getNativePointerValue(env, info);
    max_sweeps_native = (int)max_sweeps;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnXsyevjSetMaxSweeps(info_native, max_sweeps_native);

    // Write back native variable values
    // info is read-only
    // max_sweeps is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnXsyevjSetSortEigNative(JNIEnv *env, jclass cls, jobject info, jint sort_eig)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnXsyevjSetSortEig");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // sort_eig is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnXsyevjSetSortEig(info=%p, sort_eig=%d)\n",
        info, sort_eig);

    // Native variable declarations
    syevjInfo_t info_native;
    int sort_eig_native = 0;

    // Obtain native variable values
    info_native = (syevjInfo_t)getNativePointerValue(env, info);
    sort_eig_native = (int)sort_eig;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnXsyevjSetSortEig(info_native, sort_eig_native);

    // Write back native variable values
    // info is read-only
    // sort_eig is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnXsyevjGetResidualNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject residual)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnXsyevjGetResidual");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnXsyevjGetResidual");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (residual == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'residual' is null for cusolverDnXsyevjGetResidual");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnXsyevjGetResidual(handle=%p, info=%p, residual=%p)\n",
        handle, info, residual);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    syevjInfo_t info_native;
    double * residual_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    info_native = (syevjInfo_t)getNativePointerValue(env, info);
    residual_native = (double *)getPointer(env, residual);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnXsyevjGetResidual(handle_native, info_native, residual_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // residual is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnXsyevjGetSweepsNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject executed_sweeps)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnXsyevjGetSweeps");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnXsyevjGetSweeps");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (executed_sweeps == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'executed_sweeps' is null for cusolverDnXsyevjGetSweeps");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnXsyevjGetSweeps(handle=%p, info=%p, executed_sweeps=%p)\n",
        handle, info, executed_sweeps);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    syevjInfo_t info_native;
    int * executed_sweeps_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    info_native = (syevjInfo_t)getNativePointerValue(env, info);
    executed_sweeps_native = (int *)getPointer(env, executed_sweeps);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnXsyevjGetSweeps(handle_native, info_native, executed_sweeps_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // executed_sweeps is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsyevjBatched_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jintArray lwork, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsyevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsyevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnSsyevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSsyevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnSsyevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsyevjBatched_bufferSize(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, lwork=%p, params=%p, batchSize=%d)\n",
        handle, jobz, uplo, n, A, lda, W, lwork, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * W_native = NULL;
    int lwork_native;
    syevjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (float *)getPointer(env, W);
    // lwork is write-only
    params_native = (syevjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsyevjBatched_bufferSize(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, &lwork_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsyevjBatched_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jintArray lwork, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsyevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsyevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnDsyevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDsyevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnDsyevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsyevjBatched_bufferSize(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, lwork=%p, params=%p, batchSize=%d)\n",
        handle, jobz, uplo, n, A, lda, W, lwork, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * W_native = NULL;
    int lwork_native;
    syevjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (double *)getPointer(env, W);
    // lwork is write-only
    params_native = (syevjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsyevjBatched_bufferSize(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, &lwork_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCheevjBatched_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jintArray lwork, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCheevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCheevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnCheevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCheevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnCheevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCheevjBatched_bufferSize(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, lwork=%p, params=%p, batchSize=%d)\n",
        handle, jobz, uplo, n, A, lda, W, lwork, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    float * W_native = NULL;
    int lwork_native;
    syevjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (float *)getPointer(env, W);
    // lwork is write-only
    params_native = (syevjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCheevjBatched_bufferSize(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, &lwork_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZheevjBatched_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jintArray lwork, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZheevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZheevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnZheevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZheevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnZheevjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZheevjBatched_bufferSize(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, lwork=%p, params=%p, batchSize=%d)\n",
        handle, jobz, uplo, n, A, lda, W, lwork, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    double * W_native = NULL;
    int lwork_native;
    syevjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (double *)getPointer(env, W);
    // lwork is write-only
    params_native = (syevjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZheevjBatched_bufferSize(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, &lwork_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsyevjBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jobject work, jint lwork, jobject info, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsyevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsyevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnSsyevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSsyevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSsyevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnSsyevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsyevjBatched(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, work=%p, lwork=%d, info=%p, params=%p, batchSize=%d)\n",
        handle, jobz, uplo, n, A, lda, W, work, lwork, info, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * W_native = NULL;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    syevjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (float *)getPointer(env, W);
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (syevjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsyevjBatched(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, work_native, lwork_native, info_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsyevjBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jobject work, jint lwork, jobject info, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsyevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsyevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnDsyevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDsyevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDsyevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnDsyevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsyevjBatched(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, work=%p, lwork=%d, info=%p, params=%p, batchSize=%d)\n",
        handle, jobz, uplo, n, A, lda, W, work, lwork, info, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * W_native = NULL;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    syevjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (double *)getPointer(env, W);
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (syevjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsyevjBatched(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, work_native, lwork_native, info_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCheevjBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jobject work, jint lwork, jobject info, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCheevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCheevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnCheevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnCheevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnCheevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnCheevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCheevjBatched(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, work=%p, lwork=%d, info=%p, params=%p, batchSize=%d)\n",
        handle, jobz, uplo, n, A, lda, W, work, lwork, info, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    float * W_native = NULL;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    syevjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (float *)getPointer(env, W);
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (syevjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCheevjBatched(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, work_native, lwork_native, info_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZheevjBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jobject work, jint lwork, jobject info, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZheevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZheevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnZheevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZheevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZheevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnZheevjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZheevjBatched(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, work=%p, lwork=%d, info=%p, params=%p, batchSize=%d)\n",
        handle, jobz, uplo, n, A, lda, W, work, lwork, info, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    double * W_native = NULL;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    syevjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (double *)getPointer(env, W);
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (syevjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZheevjBatched(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, work_native, lwork_native, info_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsyevj_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jintArray lwork, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsyevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsyevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnSsyevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSsyevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnSsyevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsyevj_bufferSize(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, lwork=%p, params=%p)\n",
        handle, jobz, uplo, n, A, lda, W, lwork, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * W_native = NULL;
    int lwork_native;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (float *)getPointer(env, W);
    // lwork is write-only
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsyevj_bufferSize(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, &lwork_native, params_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsyevj_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jintArray lwork, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsyevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsyevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnDsyevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDsyevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnDsyevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsyevj_bufferSize(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, lwork=%p, params=%p)\n",
        handle, jobz, uplo, n, A, lda, W, lwork, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * W_native = NULL;
    int lwork_native;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (double *)getPointer(env, W);
    // lwork is write-only
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsyevj_bufferSize(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, &lwork_native, params_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCheevj_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jintArray lwork, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCheevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCheevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnCheevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCheevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnCheevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCheevj_bufferSize(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, lwork=%p, params=%p)\n",
        handle, jobz, uplo, n, A, lda, W, lwork, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    float * W_native = NULL;
    int lwork_native;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (float *)getPointer(env, W);
    // lwork is write-only
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCheevj_bufferSize(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, &lwork_native, params_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZheevj_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jintArray lwork, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZheevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZheevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnZheevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZheevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnZheevj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZheevj_bufferSize(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, lwork=%p, params=%p)\n",
        handle, jobz, uplo, n, A, lda, W, lwork, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    double * W_native = NULL;
    int lwork_native;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (double *)getPointer(env, W);
    // lwork is write-only
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZheevj_bufferSize(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, &lwork_native, params_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsyevjNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jobject work, jint lwork, jobject info, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsyevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsyevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnSsyevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSsyevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSsyevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnSsyevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsyevj(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, work=%p, lwork=%d, info=%p, params=%p)\n",
        handle, jobz, uplo, n, A, lda, W, work, lwork, info, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * W_native = NULL;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (float *)getPointer(env, W);
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsyevj(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, work_native, lwork_native, info_native, params_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsyevjNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jobject work, jint lwork, jobject info, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsyevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsyevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnDsyevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDsyevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDsyevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnDsyevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsyevj(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, work=%p, lwork=%d, info=%p, params=%p)\n",
        handle, jobz, uplo, n, A, lda, W, work, lwork, info, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * W_native = NULL;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (double *)getPointer(env, W);
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsyevj(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, work_native, lwork_native, info_native, params_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCheevjNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jobject work, jint lwork, jobject info, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCheevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCheevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnCheevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnCheevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnCheevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnCheevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCheevj(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, work=%p, lwork=%d, info=%p, params=%p)\n",
        handle, jobz, uplo, n, A, lda, W, work, lwork, info, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    float * W_native = NULL;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (float *)getPointer(env, W);
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCheevj(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, work_native, lwork_native, info_native, params_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZheevjNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject W, jobject work, jint lwork, jobject info, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZheevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZheevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnZheevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZheevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZheevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnZheevj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZheevj(handle=%p, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, W=%p, work=%p, lwork=%d, info=%p, params=%p)\n",
        handle, jobz, uplo, n, A, lda, W, work, lwork, info, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    double * W_native = NULL;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    W_native = (double *)getPointer(env, W);
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZheevj(handle_native, jobz_native, uplo_native, n_native, A_native, lda_native, W_native, work_native, lwork_native, info_native, params_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsygvj_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jintArray lwork, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsygvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsygvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnSsygvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnSsygvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSsygvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnSsygvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsygvj_bufferSize(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, lwork=%p, params=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, lwork, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * B_native = NULL;
    int ldb_native = 0;
    float * W_native = NULL;
    int lwork_native;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (float *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (float *)getPointer(env, W);
    // lwork is write-only
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsygvj_bufferSize(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, &lwork_native, params_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsygvj_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jintArray lwork, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsygvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsygvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnDsygvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnDsygvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDsygvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnDsygvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsygvj_bufferSize(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, lwork=%p, params=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, lwork, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * B_native = NULL;
    int ldb_native = 0;
    double * W_native = NULL;
    int lwork_native;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (double *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (double *)getPointer(env, W);
    // lwork is write-only
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsygvj_bufferSize(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, &lwork_native, params_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnChegvj_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jintArray lwork, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnChegvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnChegvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnChegvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnChegvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnChegvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnChegvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnChegvj_bufferSize(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, lwork=%p, params=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, lwork, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * B_native = NULL;
    int ldb_native = 0;
    float * W_native = NULL;
    int lwork_native;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (float *)getPointer(env, W);
    // lwork is write-only
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnChegvj_bufferSize(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, &lwork_native, params_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZhegvj_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jintArray lwork, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZhegvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZhegvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnZhegvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnZhegvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZhegvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnZhegvj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZhegvj_bufferSize(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, lwork=%p, params=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, lwork, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * B_native = NULL;
    int ldb_native = 0;
    double * W_native = NULL;
    int lwork_native;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuDoubleComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (double *)getPointer(env, W);
    // lwork is write-only
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZhegvj_bufferSize(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, &lwork_native, params_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSsygvjNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jobject work, jint lwork, jobject info, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSsygvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSsygvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnSsygvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnSsygvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSsygvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSsygvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnSsygvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSsygvj(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, work=%p, lwork=%d, info=%p, params=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, work, lwork, info, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * B_native = NULL;
    int ldb_native = 0;
    float * W_native = NULL;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (float *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (float *)getPointer(env, W);
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSsygvj(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, work_native, lwork_native, info_native, params_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDsygvjNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jobject work, jint lwork, jobject info, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDsygvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDsygvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnDsygvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnDsygvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDsygvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDsygvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnDsygvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDsygvj(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, work=%p, lwork=%d, info=%p, params=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, work, lwork, info, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * B_native = NULL;
    int ldb_native = 0;
    double * W_native = NULL;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (double *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (double *)getPointer(env, W);
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDsygvj(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, work_native, lwork_native, info_native, params_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnChegvjNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jobject work, jint lwork, jobject info, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnChegvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnChegvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnChegvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnChegvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnChegvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnChegvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnChegvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnChegvj(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, work=%p, lwork=%d, info=%p, params=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, work, lwork, info, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    cuComplex * B_native = NULL;
    int ldb_native = 0;
    float * W_native = NULL;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (float *)getPointer(env, W);
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnChegvj(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, work_native, lwork_native, info_native, params_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZhegvjNative(JNIEnv *env, jclass cls, jobject handle, jint itype, jint jobz, jint uplo, jint n, jobject A, jint lda, jobject B, jint ldb, jobject W, jobject work, jint lwork, jobject info, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZhegvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZhegvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusolverDnZhegvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (W == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'W' is null for cusolverDnZhegvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZhegvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZhegvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnZhegvj");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZhegvj(handle=%p, itype=%d, jobz=%d, uplo=%d, n=%d, A=%p, lda=%d, B=%p, ldb=%d, W=%p, work=%p, lwork=%d, info=%p, params=%p)\n",
        handle, itype, jobz, uplo, n, A, lda, B, ldb, W, work, lwork, info, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigType_t itype_native;
    cusolverEigMode_t jobz_native;
    cublasFillMode_t uplo_native;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    cuDoubleComplex * B_native = NULL;
    int ldb_native = 0;
    double * W_native = NULL;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    syevjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    itype_native = (cusolverEigType_t)itype;
    jobz_native = (cusolverEigMode_t)jobz;
    uplo_native = (cublasFillMode_t)uplo;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    B_native = (cuDoubleComplex *)getPointer(env, B);
    ldb_native = (int)ldb;
    W_native = (double *)getPointer(env, W);
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (syevjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZhegvj(handle_native, itype_native, jobz_native, uplo_native, n_native, A_native, lda_native, B_native, ldb_native, W_native, work_native, lwork_native, info_native, params_native);

    // Write back native variable values
    // handle is read-only
    // itype is primitive
    // jobz is primitive
    // uplo is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // B is a native pointer
    // ldb is primitive
    // W is a native pointer
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCreateGesvdjInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnCreateGesvdjInfo");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCreateGesvdjInfo(info=%p)\n",
        info);

    // Native variable declarations
    gesvdjInfo_t * info_native;

    // Obtain native variable values
    info_native = (gesvdjInfo_t *)getNativePointerValue(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCreateGesvdjInfo(info_native);

    // Write back native variable values
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDestroyGesvdjInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDestroyGesvdjInfo");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDestroyGesvdjInfo(info=%p)\n",
        info);

    // Native variable declarations
    gesvdjInfo_t info_native;

    // Obtain native variable values
    info_native = (gesvdjInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDestroyGesvdjInfo(info_native);

    // Write back native variable values
    // info is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnXgesvdjSetToleranceNative(JNIEnv *env, jclass cls, jobject info, jdouble tolerance)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnXgesvdjSetTolerance");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // tolerance is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnXgesvdjSetTolerance(info=%p, tolerance=%lf)\n",
        info, tolerance);

    // Native variable declarations
    gesvdjInfo_t info_native;
    double tolerance_native = 0.0;

    // Obtain native variable values
    info_native = (gesvdjInfo_t)getNativePointerValue(env, info);
    tolerance_native = (double)tolerance;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnXgesvdjSetTolerance(info_native, tolerance_native);

    // Write back native variable values
    // info is read-only
    // tolerance is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnXgesvdjSetMaxSweepsNative(JNIEnv *env, jclass cls, jobject info, jint max_sweeps)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnXgesvdjSetMaxSweeps");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // max_sweeps is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnXgesvdjSetMaxSweeps(info=%p, max_sweeps=%d)\n",
        info, max_sweeps);

    // Native variable declarations
    gesvdjInfo_t info_native;
    int max_sweeps_native = 0;

    // Obtain native variable values
    info_native = (gesvdjInfo_t)getNativePointerValue(env, info);
    max_sweeps_native = (int)max_sweeps;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnXgesvdjSetMaxSweeps(info_native, max_sweeps_native);

    // Write back native variable values
    // info is read-only
    // max_sweeps is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnXgesvdjSetSortEigNative(JNIEnv *env, jclass cls, jobject info, jint sort_svd)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnXgesvdjSetSortEig");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // sort_svd is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnXgesvdjSetSortEig(info=%p, sort_svd=%d)\n",
        info, sort_svd);

    // Native variable declarations
    gesvdjInfo_t info_native;
    int sort_svd_native = 0;

    // Obtain native variable values
    info_native = (gesvdjInfo_t)getNativePointerValue(env, info);
    sort_svd_native = (int)sort_svd;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnXgesvdjSetSortEig(info_native, sort_svd_native);

    // Write back native variable values
    // info is read-only
    // sort_svd is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnXgesvdjGetResidualNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject residual)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnXgesvdjGetResidual");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnXgesvdjGetResidual");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (residual == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'residual' is null for cusolverDnXgesvdjGetResidual");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnXgesvdjGetResidual(handle=%p, info=%p, residual=%p)\n",
        handle, info, residual);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    gesvdjInfo_t info_native;
    double * residual_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    info_native = (gesvdjInfo_t)getNativePointerValue(env, info);
    residual_native = (double *)getPointer(env, residual);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnXgesvdjGetResidual(handle_native, info_native, residual_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // residual is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnXgesvdjGetSweepsNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject executed_sweeps)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnXgesvdjGetSweeps");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnXgesvdjGetSweeps");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (executed_sweeps == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'executed_sweeps' is null for cusolverDnXgesvdjGetSweeps");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnXgesvdjGetSweeps(handle=%p, info=%p, executed_sweeps=%p)\n",
        handle, info, executed_sweeps);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    gesvdjInfo_t info_native;
    int * executed_sweeps_native = NULL;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    info_native = (gesvdjInfo_t)getNativePointerValue(env, info);
    executed_sweeps_native = (int *)getPointer(env, executed_sweeps);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnXgesvdjGetSweeps(handle_native, info_native, executed_sweeps_native);

    // Write back native variable values
    // handle is read-only
    // info is read-only
    // executed_sweeps is a native pointer

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSgesvdjBatched_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject V, jint ldv, jintArray lwork, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnSgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnSgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (V == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'V' is null for cusolverDnSgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldv is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnSgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSgesvdjBatched_bufferSize(handle=%p, jobz=%d, m=%d, n=%d, A=%p, lda=%d, S=%p, U=%p, ldu=%d, V=%p, ldv=%d, lwork=%p, params=%p, batchSize=%d)\n",
        handle, jobz, m, n, A, lda, S, U, ldu, V, ldv, lwork, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    int m_native = 0;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * S_native = NULL;
    float * U_native = NULL;
    int ldu_native = 0;
    float * V_native = NULL;
    int ldv_native = 0;
    int lwork_native;
    gesvdjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    S_native = (float *)getPointer(env, S);
    U_native = (float *)getPointer(env, U);
    ldu_native = (int)ldu;
    V_native = (float *)getPointer(env, V);
    ldv_native = (int)ldv;
    // lwork is write-only
    params_native = (gesvdjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSgesvdjBatched_bufferSize(handle_native, jobz_native, m_native, n_native, A_native, lda_native, S_native, U_native, ldu_native, V_native, ldv_native, &lwork_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // S is a native pointer
    // U is a native pointer
    // ldu is primitive
    // V is a native pointer
    // ldv is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDgesvdjBatched_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject V, jint ldv, jintArray lwork, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnDgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnDgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (V == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'V' is null for cusolverDnDgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldv is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnDgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDgesvdjBatched_bufferSize(handle=%p, jobz=%d, m=%d, n=%d, A=%p, lda=%d, S=%p, U=%p, ldu=%d, V=%p, ldv=%d, lwork=%p, params=%p, batchSize=%d)\n",
        handle, jobz, m, n, A, lda, S, U, ldu, V, ldv, lwork, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    int m_native = 0;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * S_native = NULL;
    double * U_native = NULL;
    int ldu_native = 0;
    double * V_native = NULL;
    int ldv_native = 0;
    int lwork_native;
    gesvdjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    S_native = (double *)getPointer(env, S);
    U_native = (double *)getPointer(env, U);
    ldu_native = (int)ldu;
    V_native = (double *)getPointer(env, V);
    ldv_native = (int)ldv;
    // lwork is write-only
    params_native = (gesvdjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDgesvdjBatched_bufferSize(handle_native, jobz_native, m_native, n_native, A_native, lda_native, S_native, U_native, ldu_native, V_native, ldv_native, &lwork_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // S is a native pointer
    // U is a native pointer
    // ldu is primitive
    // V is a native pointer
    // ldv is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCgesvdjBatched_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject V, jint ldv, jintArray lwork, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnCgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnCgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (V == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'V' is null for cusolverDnCgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldv is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnCgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnCgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCgesvdjBatched_bufferSize(handle=%p, jobz=%d, m=%d, n=%d, A=%p, lda=%d, S=%p, U=%p, ldu=%d, V=%p, ldv=%d, lwork=%p, params=%p, batchSize=%d)\n",
        handle, jobz, m, n, A, lda, S, U, ldu, V, ldv, lwork, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    float * S_native = NULL;
    cuComplex * U_native = NULL;
    int ldu_native = 0;
    cuComplex * V_native = NULL;
    int ldv_native = 0;
    int lwork_native;
    gesvdjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    S_native = (float *)getPointer(env, S);
    U_native = (cuComplex *)getPointer(env, U);
    ldu_native = (int)ldu;
    V_native = (cuComplex *)getPointer(env, V);
    ldv_native = (int)ldv;
    // lwork is write-only
    params_native = (gesvdjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCgesvdjBatched_bufferSize(handle_native, jobz_native, m_native, n_native, A_native, lda_native, S_native, U_native, ldu_native, V_native, ldv_native, &lwork_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // S is a native pointer
    // U is a native pointer
    // ldu is primitive
    // V is a native pointer
    // ldv is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZgesvdjBatched_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject V, jint ldv, jintArray lwork, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnZgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnZgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (V == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'V' is null for cusolverDnZgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldv is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnZgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnZgesvdjBatched_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZgesvdjBatched_bufferSize(handle=%p, jobz=%d, m=%d, n=%d, A=%p, lda=%d, S=%p, U=%p, ldu=%d, V=%p, ldv=%d, lwork=%p, params=%p, batchSize=%d)\n",
        handle, jobz, m, n, A, lda, S, U, ldu, V, ldv, lwork, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    double * S_native = NULL;
    cuDoubleComplex * U_native = NULL;
    int ldu_native = 0;
    cuDoubleComplex * V_native = NULL;
    int ldv_native = 0;
    int lwork_native;
    gesvdjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    S_native = (double *)getPointer(env, S);
    U_native = (cuDoubleComplex *)getPointer(env, U);
    ldu_native = (int)ldu;
    V_native = (cuDoubleComplex *)getPointer(env, V);
    ldv_native = (int)ldv;
    // lwork is write-only
    params_native = (gesvdjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZgesvdjBatched_bufferSize(handle_native, jobz_native, m_native, n_native, A_native, lda_native, S_native, U_native, ldu_native, V_native, ldv_native, &lwork_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // S is a native pointer
    // U is a native pointer
    // ldu is primitive
    // V is a native pointer
    // ldv is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSgesvdjBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject V, jint ldv, jobject work, jint lwork, jobject info, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnSgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnSgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (V == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'V' is null for cusolverDnSgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldv is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnSgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnSgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnSgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSgesvdjBatched(handle=%p, jobz=%d, m=%d, n=%d, A=%p, lda=%d, S=%p, U=%p, ldu=%d, V=%p, ldv=%d, work=%p, lwork=%d, info=%p, params=%p, batchSize=%d)\n",
        handle, jobz, m, n, A, lda, S, U, ldu, V, ldv, work, lwork, info, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    int m_native = 0;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * S_native = NULL;
    float * U_native = NULL;
    int ldu_native = 0;
    float * V_native = NULL;
    int ldv_native = 0;
    float * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    gesvdjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    S_native = (float *)getPointer(env, S);
    U_native = (float *)getPointer(env, U);
    ldu_native = (int)ldu;
    V_native = (float *)getPointer(env, V);
    ldv_native = (int)ldv;
    work_native = (float *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (gesvdjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSgesvdjBatched(handle_native, jobz_native, m_native, n_native, A_native, lda_native, S_native, U_native, ldu_native, V_native, ldv_native, work_native, lwork_native, info_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // S is a native pointer
    // U is a native pointer
    // ldu is primitive
    // V is a native pointer
    // ldv is primitive
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDgesvdjBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject V, jint ldv, jobject work, jint lwork, jobject info, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnDgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnDgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (V == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'V' is null for cusolverDnDgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldv is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnDgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnDgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnDgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDgesvdjBatched(handle=%p, jobz=%d, m=%d, n=%d, A=%p, lda=%d, S=%p, U=%p, ldu=%d, V=%p, ldv=%d, work=%p, lwork=%d, info=%p, params=%p, batchSize=%d)\n",
        handle, jobz, m, n, A, lda, S, U, ldu, V, ldv, work, lwork, info, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    int m_native = 0;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * S_native = NULL;
    double * U_native = NULL;
    int ldu_native = 0;
    double * V_native = NULL;
    int ldv_native = 0;
    double * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    gesvdjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    S_native = (double *)getPointer(env, S);
    U_native = (double *)getPointer(env, U);
    ldu_native = (int)ldu;
    V_native = (double *)getPointer(env, V);
    ldv_native = (int)ldv;
    work_native = (double *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (gesvdjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDgesvdjBatched(handle_native, jobz_native, m_native, n_native, A_native, lda_native, S_native, U_native, ldu_native, V_native, ldv_native, work_native, lwork_native, info_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // S is a native pointer
    // U is a native pointer
    // ldu is primitive
    // V is a native pointer
    // ldv is primitive
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCgesvdjBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject V, jint ldv, jobject work, jint lwork, jobject info, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnCgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnCgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (V == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'V' is null for cusolverDnCgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldv is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnCgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnCgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnCgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnCgesvdjBatched(handle=%p, jobz=%d, m=%d, n=%d, A=%p, lda=%d, S=%p, U=%p, ldu=%d, V=%p, ldv=%d, work=%p, lwork=%d, info=%p, params=%p, batchSize=%d)\n",
        handle, jobz, m, n, A, lda, S, U, ldu, V, ldv, work, lwork, info, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex * A_native = NULL;
    int lda_native = 0;
    float * S_native = NULL;
    cuComplex * U_native = NULL;
    int ldu_native = 0;
    cuComplex * V_native = NULL;
    int ldv_native = 0;
    cuComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    gesvdjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuComplex *)getPointer(env, A);
    lda_native = (int)lda;
    S_native = (float *)getPointer(env, S);
    U_native = (cuComplex *)getPointer(env, U);
    ldu_native = (int)ldu;
    V_native = (cuComplex *)getPointer(env, V);
    ldv_native = (int)ldv;
    work_native = (cuComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (gesvdjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnCgesvdjBatched(handle_native, jobz_native, m_native, n_native, A_native, lda_native, S_native, U_native, ldu_native, V_native, ldv_native, work_native, lwork_native, info_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // S is a native pointer
    // U is a native pointer
    // ldu is primitive
    // V is a native pointer
    // ldv is primitive
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnZgesvdjBatchedNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject V, jint ldv, jobject work, jint lwork, jobject info, jobject params, jint batchSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnZgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnZgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnZgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnZgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (V == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'V' is null for cusolverDnZgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldv is primitive
    if (work == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'work' is null for cusolverDnZgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lwork is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusolverDnZgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnZgesvdjBatched");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // batchSize is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnZgesvdjBatched(handle=%p, jobz=%d, m=%d, n=%d, A=%p, lda=%d, S=%p, U=%p, ldu=%d, V=%p, ldv=%d, work=%p, lwork=%d, info=%p, params=%p, batchSize=%d)\n",
        handle, jobz, m, n, A, lda, S, U, ldu, V, ldv, work, lwork, info, params, batchSize);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex * A_native = NULL;
    int lda_native = 0;
    double * S_native = NULL;
    cuDoubleComplex * U_native = NULL;
    int ldu_native = 0;
    cuDoubleComplex * V_native = NULL;
    int ldv_native = 0;
    cuDoubleComplex * work_native = NULL;
    int lwork_native = 0;
    int * info_native = NULL;
    gesvdjInfo_t params_native;
    int batchSize_native = 0;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (cuDoubleComplex *)getPointer(env, A);
    lda_native = (int)lda;
    S_native = (double *)getPointer(env, S);
    U_native = (cuDoubleComplex *)getPointer(env, U);
    ldu_native = (int)ldu;
    V_native = (cuDoubleComplex *)getPointer(env, V);
    ldv_native = (int)ldv;
    work_native = (cuDoubleComplex *)getPointer(env, work);
    lwork_native = (int)lwork;
    info_native = (int *)getPointer(env, info);
    params_native = (gesvdjInfo_t)getNativePointerValue(env, params);
    batchSize_native = (int)batchSize;

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnZgesvdjBatched(handle_native, jobz_native, m_native, n_native, A_native, lda_native, S_native, U_native, ldu_native, V_native, ldv_native, work_native, lwork_native, info_native, params_native, batchSize_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // S is a native pointer
    // U is a native pointer
    // ldu is primitive
    // V is a native pointer
    // ldv is primitive
    // work is a native pointer
    // lwork is primitive
    // info is a native pointer
    // params is read-only
    // batchSize is primitive

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnSgesvdj_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint econ, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject V, jint ldv, jintArray lwork, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnSgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // econ is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnSgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnSgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnSgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (V == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'V' is null for cusolverDnSgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldv is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnSgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnSgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnSgesvdj_bufferSize(handle=%p, jobz=%d, econ=%d, m=%d, n=%d, A=%p, lda=%d, S=%p, U=%p, ldu=%d, V=%p, ldv=%d, lwork=%p, params=%p)\n",
        handle, jobz, econ, m, n, A, lda, S, U, ldu, V, ldv, lwork, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    int econ_native = 0;
    int m_native = 0;
    int n_native = 0;
    float * A_native = NULL;
    int lda_native = 0;
    float * S_native = NULL;
    float * U_native = NULL;
    int ldu_native = 0;
    float * V_native = NULL;
    int ldv_native = 0;
    int lwork_native;
    gesvdjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    econ_native = (int)econ;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (float *)getPointer(env, A);
    lda_native = (int)lda;
    S_native = (float *)getPointer(env, S);
    U_native = (float *)getPointer(env, U);
    ldu_native = (int)ldu;
    V_native = (float *)getPointer(env, V);
    ldv_native = (int)ldv;
    // lwork is write-only
    params_native = (gesvdjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnSgesvdj_bufferSize(handle_native, jobz_native, econ_native, m_native, n_native, A_native, lda_native, S_native, U_native, ldu_native, V_native, ldv_native, &lwork_native, params_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // econ is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // S is a native pointer
    // U is a native pointer
    // ldu is primitive
    // V is a native pointer
    // ldv is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnDgesvdj_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint econ, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject V, jint ldv, jintArray lwork, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnDgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // econ is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnDgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnDgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnDgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (V == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'V' is null for cusolverDnDgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldv is primitive
    if (lwork == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'lwork' is null for cusolverDnDgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (params == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'params' is null for cusolverDnDgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusolverDnDgesvdj_bufferSize(handle=%p, jobz=%d, econ=%d, m=%d, n=%d, A=%p, lda=%d, S=%p, U=%p, ldu=%d, V=%p, ldv=%d, lwork=%p, params=%p)\n",
        handle, jobz, econ, m, n, A, lda, S, U, ldu, V, ldv, lwork, params);

    // Native variable declarations
    cusolverDnHandle_t handle_native;
    cusolverEigMode_t jobz_native;
    int econ_native = 0;
    int m_native = 0;
    int n_native = 0;
    double * A_native = NULL;
    int lda_native = 0;
    double * S_native = NULL;
    double * U_native = NULL;
    int ldu_native = 0;
    double * V_native = NULL;
    int ldv_native = 0;
    int lwork_native;
    gesvdjInfo_t params_native;

    // Obtain native variable values
    handle_native = (cusolverDnHandle_t)getNativePointerValue(env, handle);
    jobz_native = (cusolverEigMode_t)jobz;
    econ_native = (int)econ;
    m_native = (int)m;
    n_native = (int)n;
    A_native = (double *)getPointer(env, A);
    lda_native = (int)lda;
    S_native = (double *)getPointer(env, S);
    U_native = (double *)getPointer(env, U);
    ldu_native = (int)ldu;
    V_native = (double *)getPointer(env, V);
    ldv_native = (int)ldv;
    // lwork is write-only
    params_native = (gesvdjInfo_t)getNativePointerValue(env, params);

    // Native function call
    cusolverStatus_t jniResult_native = cusolverDnDgesvdj_bufferSize(handle_native, jobz_native, econ_native, m_native, n_native, A_native, lda_native, S_native, U_native, ldu_native, V_native, ldv_native, &lwork_native, params_native);

    // Write back native variable values
    // handle is read-only
    // jobz is primitive
    // econ is primitive
    // m is primitive
    // n is primitive
    // A is a native pointer
    // lda is primitive
    // S is a native pointer
    // U is a native pointer
    // ldu is primitive
    // V is a native pointer
    // ldv is primitive
    if (!set(env, lwork, 0, (jint)lwork_native)) return JCUSOLVER_STATUS_INTERNAL_ERROR;
    // params is read-only

    // Return the result
    jint jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusolver_JCusolverDn_cusolverDnCgesvdj_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint jobz, jint econ, jint m, jint n, jobject A, jint lda, jobject S, jobject U, jint ldu, jobject V, jint ldv, jintArray lwork, jobject params)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusolverDnCgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // jobz is primitive
    // econ is primitive
    // m is primitive
    // n is primitive
    if (A == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'A' is null for cusolverDnCgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // lda is primitive
    if (S == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'S' is null for cusolverDnCgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    if (U == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'U' is null for cusolverDnCgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTERNAL_ERROR;
    }
    // ldu is primitive
    if (V == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'V' is null for cusolverDnCgesvdj_bufferSize");
        return JCUSOLVER_STATUS_INTER